/*************************************************************************
 *                                                                       *
 *       (c) Copyright 2003                                              *
 *       All rights reserved                                             *
 *       Programs written by Jianghui Liu (NJIT)                         *
 *                                                                       *
 *       Permission to use, copy, modify, and distribute this software   *
 *       and its documentation for any purpose and without fee is her-   *
 *       eby granted, provided that this copyright notice appears in     *
 *       all copies.   Programmer(s) makes no representations about      *
 *       the suitability of this software for any purpose.  It is pro-   *
 *       vided "as is" without express or implied warranty.              *
 *                                                                       *
 *       08/06/2003                                                      *
 *************************************************************************/
package RNA;
import java.util.*;
import java.io.*;

// ----------------------------------------------------------------------
//   Structure-based Multi-alignment: individual member structures are
//   aligned togather using RNAmatcher algorithm. A STOCKHOLM1.0 output
//   format is provided for further processing of S.Eddy's Infernal tool.
// ----------------------------------------------------------------------
public class MultiAlignment
{
    private ArrayList  MemberRNAs;         // member RNAs in the multi-alignment
    private int[][]    AlignmentMatrix;    // multi-alignment matrix
    private double[]   Measures;           // [0]: minimum score
                                           // [1]: maximum score
                                           // [2]: average score
    private String     StructLine;         // consensus structure notation
    private int        nColumns;           // number of columns in the multi-alignment
    private int        nRows;              // number of member of RNAs
    private Hashtable  codeTable;          // IUB code mapping

    private RNA        profileStruct;  // A consensus structure of the multi-alignment using IUB code

    /*public static void main(String[] args)
    {
        String a = "7sk.sto";
        //String[] b = a.split("\\s+", 3);
        //for(int i=0; i<b.length; i++)
        //    System.out.println(b[i]);
        MultiAlignment test = new MultiAlignment(a);
        System.out.println(test.getStoAlignment());
        System.out.println(test.getProfileRNA().dispRNA());
    }*/

    // --------------------------------------------------------------------
    // CONSTRUCTION FUNCTION:
    //  A structural multiple alignment SHOULD always start either from a
    //  pairwise structural alignment, which is the simplest case of
    //  multiple alignment; or from a STOCKHOLM formated multi-alignment.
    // ---------------------------------------------------------------------
    public MultiAlignment(String stoFile)
    {
        try{
            BufferedReader input = new BufferedReader(new FileReader(stoFile));
            String line = input.readLine();
            if( line.startsWith("# STOCKHOLM") == false ){
                System.out.println("Fatal Error: only STOCKHOLM format is accepted!");
                System.exit(0);
            }

            ArrayList block = new ArrayList();
            ArrayList names = new ArrayList();
            boolean finishOneBlock = false;
            StringBuffer secondStruct = new StringBuffer();

            while( (line=input.readLine()) != null && (line.startsWith("#") || line.length() == 0) );

            while( line != null && line.equals("//") == false ){
                if( line.length() != 0){
                    String[] fields = line.split("\\s+", 3);
                    if( fields.length == 2 ){   // line of sequence
                        block.add(fields[1]);
                        if( finishOneBlock == false )
                            names.add(fields[0]);
                    } else if( fields[0].equals("#=GC") && fields[1].equals("SS_cons") )
                        secondStruct.append(fields[2]);
                } else {
                    if( names.size() != 0 && finishOneBlock == false)
                       finishOneBlock = true; 
                }

                line = input.readLine();
            }
            input.close();

            StructLine = secondStruct.toString();

            nRows = names.size();
            nColumns = StructLine.length();
            MemberRNAs = new ArrayList();
            AlignmentMatrix = new int[nRows][nColumns];
            for(int i=0; i<nRows; i++){ 
                int pos = 0;

                StringBuffer psudoSeq = new StringBuffer();
                for(int j=i; j<block.size(); j+=nRows)
                    psudoSeq.append((String)block.get(j));

                StringBuffer curSeq = new StringBuffer();
                StringBuffer curStruct = new StringBuffer();
                for(int j=0; j<nColumns; j++){
                    if( psudoSeq.substring(j, j+1).equals(".") || psudoSeq.substring(j, j+1).equals("-") )
                    {
                        if( StructLine.substring(j, j+1).equals(".") == false ){
                            curSeq.append("A");
                            curStruct.append(StructLine.substring(j, j+1));
                            AlignmentMatrix[i][j] = pos;
                            pos ++;
                        } else
                            AlignmentMatrix[i][j] = -1;
                    } else {
                        curSeq.append(psudoSeq.substring(j, j+1));
                        curStruct.append(StructLine.substring(j, j+1));
                        AlignmentMatrix[i][j] = pos;
                        pos ++;
                    }
                }

                RNA oneMember = new RNA((String)names.get(i), curSeq.toString(), curStruct.toString(), "");
                MemberRNAs.add(oneMember);
            }
        } catch (IOException e) {
            e.printStackTrace();
            System.exit(0);
        }
    }

    public MultiAlignment(RNA a, int[] aMap, RNA b, int[] bMap)
    {
        MemberRNAs = new ArrayList();
        MemberRNAs.add(a);
        MemberRNAs.add(b);
        nRows = MemberRNAs.size();

        int aStart = 0, aEnd = 0;
        int bStart = 0, bEnd = 0;

        int region[] = HomoUtil.getAlignedRegion(a, aMap);
        aStart = region[0];
        aEnd   = region[1];

        region = HomoUtil.getAlignedRegion(b, bMap);
        bStart = region[0];
        bEnd   = region[1];

        int aGap = 0, bGap = 0;
        for( int i = aStart; i <= aEnd; i ++ ) {
            if( aMap[i] == -1 )
                bGap ++;
        }
        for( int i = bStart; i <= bEnd; i ++ ) {
            if( bMap[i] == -1 )
                aGap ++;
        }

        if( aEnd - aStart + 1 + aGap > bEnd - bStart + 1 + bGap )
            nColumns = aEnd - aStart + 1 + aGap;
        else
            nColumns = bEnd - bStart + 1 + bGap;

        AlignmentMatrix = new int[2][nColumns];
        StringBuffer StructLineBuff = new StringBuffer();
        String line1 = a.getStructLine();
        String line2 = b.getStructLine();

        int pos1 = aStart, pos2 = bStart;
        for(int i = 0; i < nColumns; i ++ ) {
            if( aMap[pos1] == pos2 + 1) {        // This is an aligned column
                AlignmentMatrix[0][i] = pos1;
                AlignmentMatrix[1][i] = pos2;

                StructLineBuff.append(line1.substring(pos1, pos1+1));
                pos1 ++;
                pos2 ++;
            } else if( aMap[pos1] > pos2 + 1 ) {   // gap from first line
                AlignmentMatrix[0][i] = -1;
                AlignmentMatrix[1][i] = pos2;

                StructLineBuff.append(line2.substring(pos2, pos2+1));
                pos2 ++;
            } else if( bMap[pos2] > pos1 + 1) {    // gap from second line
                AlignmentMatrix[0][i] = pos1;
                AlignmentMatrix[1][i] = -1;

                StructLineBuff.append(line1.substring(pos1, pos1+1));
                pos1 ++;
            } else if( aMap[pos1] == -1 && bMap[pos2] == -1) { // two gaps from each line at different column
                AlignmentMatrix[0][i] = pos1;
                AlignmentMatrix[1][i] = -1;
                StructLineBuff.append(line1.substring(pos1, pos1+1));
                pos1 ++;

                //i ++;
                //AlignmentMatrix[0][i] = -1;
                //AlignmentMatrix[1][i] = pos2;
                //StructLineBuff.append(line2.substring(pos2, pos2+1));
                //pos2 ++;
            } else {  // Fatal Panic !!
                System.out.println("Something fatal happend when constructing MultiAlignment object:");
                System.out.println("first structure is:");
                System.out.println(a.getInfo());
                System.out.println("second structure is: ");
                System.out.println(b.getInfo());
                System.out.println("first position: " + (pos1+1) + "  with structure: " + 
                                      line1.substring(pos1, pos1+1));
                System.out.println("second position: " + (pos2+1) + " with structure: " +
                                      line2.substring(pos2,pos2+1));
                System.exit(1);
           }

           if( pos1 > aEnd) {
                while( pos2 <= bEnd) {
                    i++;
                    AlignmentMatrix[0][i] = -1;
                    AlignmentMatrix[1][i] = pos2;
                    StructLineBuff.append(line2.substring(pos2,pos2+1));
                    pos2 ++;
                }
                break;
            } else if( pos2 > bEnd) {
                while( pos1 <= aEnd) {
                    i++;
                    AlignmentMatrix[0][i] = pos1;
                    AlignmentMatrix[1][i] = -1;
                    pos1 ++;
                }
                break;
            }
        }

        StructLine = new String(StructLineBuff);
    }

            


    //-----------------------------------------------------
    // The multi-alignment is expanded one more structure
    //-----------------------------------------------------
    public void expand(int[] profileMap, RNA data, int[] dataMap)
    {
        if( profileMap.length != getProfileRNA().seqLength() ) {
            System.out.println("Error of expanding: profileMap NOT matched with profile!");
            System.exit(0);
        }
        int aStart = 0, aEnd = profileMap.length-1;

        int bStart = 0, bEnd = 0;
        int[] region = HomoUtil.getAlignedRegion(data, dataMap);
        bStart = region[0];
        bEnd = region[1];

        int aGap = 0, bGap = 0;
        for( int i = aStart; i <= aEnd; i ++ ) {
            if( profileMap[i] == -1 )
                bGap ++;
        }
        for( int i = bStart; i <= bEnd; i ++ ) {
            if( dataMap[i] == -1 )
                aGap ++;
        }

        int newColumns = 0;
        int newRows = nRows + 1;
        if( aEnd - aStart + 1 + aGap > bEnd - bStart + 1 + bGap )
            newColumns = aEnd - aStart + 1 + aGap;
        else
            newColumns = bEnd - bStart + 1 + bGap;

        int[][] newAlignmentMatrix = new int[newRows][newColumns];

        //System.out.println("\nExpanding the profiles ... ");
        StringBuffer StructLineBuff = new StringBuffer();
        int pos1 = aStart, pos2 = bStart;
        for(int i = 0; i < newColumns; i ++ ) {
            int rowIndex = 0;
            if( profileMap[pos1] == pos2 + 1) {
                for(int j=0; j<nRows; j++){
                    newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];
                }
                newAlignmentMatrix[rowIndex++][i] = pos2;

                StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                pos1 ++;
                pos2 ++;
            } else if( profileMap[pos1] > pos2 + 1 ) {
                for(int j=0; j<nRows; j++)
                    newAlignmentMatrix[rowIndex++][i] = -1;

                newAlignmentMatrix[rowIndex++][i] = pos2;

                StructLineBuff.append(data.getStructLine().substring(pos2, pos2+1));
                pos2 ++;
            } else if( dataMap[pos2] > pos1 + 1) {
                for(int j=0; j<nRows; j++)
                    newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];

                newAlignmentMatrix[rowIndex++][i] = -1;

                StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                pos1 ++;
            } else if( profileMap[pos1] == -1 && dataMap[pos2] == -1 ){
                for(int j=0; j<nRows; j++)
                    newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];

                newAlignmentMatrix[rowIndex++][i] = -1;
                StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                pos1 ++;
 
                //i ++;   // this is the other column
                //rowIndex = 0;
                //for(int j=0; j<nRows; j++)
                //    newAlignmentMatrix[rowIndex++][i] = -1;

                //newAlignmentMatrix[rowIndex++][i] = pos2;
                //StructLineBuff.append(data.getStructLine().substring(pos2, pos2+1));
                //pos2 ++;
            } else { // something really serious happens !!
                System.out.println("Something fatal happenend when trying to merge:");
                System.out.println(getStoAlignment());
                System.out.println(data.dispRNA());
                System.out.println("profile position: " + (pos1+1) + " with structure: " +
                                      StructLine.substring(pos1, pos1+1));
                System.out.println("RNA position: " + (pos2+1) + " with structure: " +
                                      data.getStructLine().substring(pos2, pos2+1));
                System.exit(1);
            }

            if( pos1 > aEnd) {
                while( pos2 <= bEnd) {
                    i++;
                    rowIndex = 0;
                    for(int j=0; j<nRows; j++)
                        newAlignmentMatrix[rowIndex++][i] = -1;

                    newAlignmentMatrix[rowIndex++][i] = pos2;

                    StructLineBuff.append(data.getStructLine().substring(pos2, pos2+1));
                    pos2 ++;
                }
                break;
            }

            if( pos2 > bEnd) {
                while( pos1 <= aEnd) {
                    i++;
                    rowIndex = 0;
                    for(int j=0; j<nRows; j++)
                        newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];

                    newAlignmentMatrix[rowIndex++][i] = -1;

                    StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                    pos1 ++;
                }
                break;
            }
        }

        // expunge those alignment columns which are pure gap
        boolean[] good = new boolean[newColumns];
        int nGood = 0;
        for(int i=0; i<newColumns; i++) {
            good[i] = false;
            for(int j=0; j<newRows; j++) {
                if( newAlignmentMatrix[j][i] != -1) {
                    good[i] = true;
                    nGood ++;
                    break;
                }
            }
        }

        AlignmentMatrix = new int[newRows][nGood];
        for(int i=0, k=0; i<newColumns; i++) {
            if( good[i] == true) {
                for(int j=0; j<newRows; j++)
                    AlignmentMatrix[j][k] = newAlignmentMatrix[j][i];
                k++;
            } else
                StructLineBuff.deleteCharAt(i);
        }

        StructLine = new String(StructLineBuff);

        ArrayList newMemberRNAs = new ArrayList();
        newMemberRNAs.addAll(MemberRNAs);
        newMemberRNAs.add(data);
        MemberRNAs = newMemberRNAs;

        nColumns = nGood;
        nRows = newRows;
        profileStruct = null;
        Measures = null;
    }

    //----------------------------------------------------------------------
    // Get the measurements of his multi-alignment: 
    //  double[0]:  minimam alignment score between member and the profile
    //  double[1]:  maximum alignment score between member and the profile
    //  double[2]:  average alignment score between member and the profile
    //----------------------------------------------------------------------
    public double[] getMeasures(double Penalty)
    {
        if( Measures == null) {
            Measures = new double[3];
            Measures[0] = Double.POSITIVE_INFINITY;
            Measures[1] = Double.NEGATIVE_INFINITY;
            ProfileMatcher matcher = new ProfileMatcher(this);
            double sum = 0;
            for( int i=0; i<nRows; i++ ) {
                RNA rna = (RNA)MemberRNAs.get(i);
                double score = RNA.profileMatch2D(profileStruct, rna, null, null,matcher,Penalty);
                sum += score;

                if( score < Measures[0] )
                    Measures[0] = score;
                if( score > Measures[1] )
                    Measures[1] = score; 
            }
            Measures[2] = sum / nRows;
        }

        double ret[] = new double[3];
        ret[0] = Measures[0];
        ret[1] = Measures[1];
        ret[2] = Measures[2];
        return ret;
    }

    // ----------------------------------------------------------------------------------
    //  Increment current multi-alignment by merging with another MultiAlignment object
    //  based on their profile structures. The derived two profiles are aligned, 
    //  given the scoring scheme and gap penalty.
    // ----------------------------------------------------------------------------------
    public void merge(MultiAlignment source, String scoreFile, int penalty)
    {
        ArrayList newMemberRNAs = new ArrayList();
        newMemberRNAs.addAll(MemberRNAs);
        for(int i=0; i<source.nRows; i++)
            if( contains(((RNA)source.MemberRNAs.get(i)).getName()) == false)
                newMemberRNAs.add(source.MemberRNAs.get(i));

        int newRows = newMemberRNAs.size();
        if( newRows == nRows ){  // nothing new !!
            System.out.println("No new member RNA to be merged!");
            return;
        }

        RNA profile1 = getProfileRNA();
        RNA profile2 = source.getProfileRNA(); 
    
        System.out.println(getStoAlignment());
        System.out.println(source.getStoAlignment());

        // set up code translation table and score matrix
        String codeTableFile = "codeTable.properties";
        Matcher m = new Matcher(codeTableFile, scoreFile);

        int[] aMap = new int[profile1.seqLength()];
        int[] bMap = new int[profile2.seqLength()];
        for(int i=0; i<aMap.length; i++)
            aMap[i] = -1;
        for(int i=0; i<bMap.length; i++)
            bMap[i] = -1;
            
        System.out.println("\naligning profiles ... ");
        if( nColumns > source.nColumns)
            RNA.motifMatch2D(profile2, profile1, bMap, aMap, m, penalty);
        else
            RNA.motifMatch2D(profile1, profile2, aMap, bMap, m, penalty);
        

        System.out.println( RNA.alignMatch(profile1, aMap, profile2, bMap));

        int aStart = 0, aEnd = 0;
        int bStart = 0, bEnd = 0;
        int[] region = HomoUtil.getAlignedRegion(profile1, aMap);
        aStart = region[0];
        aEnd = region[1];

        region = HomoUtil.getAlignedRegion(profile2, bMap);
        bStart = region[0];
        bEnd = region[1];

        int aGap = 0, bGap = 0;
        for( int i = aStart; i <= aEnd; i ++ ) {
            if( aMap[i] == -1 )
                bGap ++;
        }
        for( int i = bStart; i <= bEnd; i ++ ) {
            if( bMap[i] == -1 )
                aGap ++;
        }

        int newColumns = 0;
        if( aEnd - aStart + 1 + aGap > bEnd - bStart + 1 + bGap )
            newColumns = aEnd - aStart + 1 + aGap;
        else
            newColumns = bEnd - bStart + 1 + bGap;

        int[][] newAlignmentMatrix = new int[newRows][newColumns];

        System.out.println("Merging the profiles ... ");
        StringBuffer StructLineBuff = new StringBuffer();
        int pos1 = aStart, pos2 = bStart;
        for(int i = 0; i < newColumns; i ++ ) {
            int rowIndex = 0;
            if( aMap[pos1] == pos2 + 1) {
                for(int j=0; j<nRows; j++){
                    newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];
                }
                for(int j=0; j<source.nRows; j++){
                    if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                        newAlignmentMatrix[rowIndex++][i] = source.AlignmentMatrix[j][pos2];
                    }
                }

                StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                pos1 ++;
                pos2 ++;
            } else if( aMap[pos1] > pos2 + 1 ) {
                for(int j=0; j<nRows; j++)
                    newAlignmentMatrix[rowIndex++][i] = -1;
                for(int j=0; j<source.nRows; j++){
                    if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                        newAlignmentMatrix[rowIndex++][i] = source.AlignmentMatrix[j][pos2];
                    }
                }

                StructLineBuff.append(source.StructLine.substring(pos2, pos2+1));
                pos2 ++;
            } else if( bMap[pos2] > pos1 + 1) {
                for(int j=0; j<nRows; j++)
                    newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];
                for(int j=0; j<source.nRows; j++){
                    if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                        newAlignmentMatrix[rowIndex++][i] = -1;
                    }
                }

                StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                pos1 ++;
            } else if( aMap[pos1] == -1 && bMap[pos2] == -1 ){
                for(int j=0; j<nRows; j++)
                    newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];
                for(int j=0; j<source.nRows; j++){
                    if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                        newAlignmentMatrix[rowIndex++][i] = -1;
                    }
                }
                StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                pos1 ++;
 
                //i ++;   // this is the other column
                //rowIndex = 0;
                //for(int j=0; j<nRows; j++)
                //    newAlignmentMatrix[rowIndex++][i] = -1;
                //for(int j=0; j<source.nRows; j++){
                //    if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                //        newAlignmentMatrix[rowIndex++][i] = source.AlignmentMatrix[j][pos2];
                //    }
                //}
                //StructLineBuff.append(source.StructLine.substring(pos2, pos2+1));
                //pos2 ++;
            } else { // something really serious happens !!
                System.out.println("Something fatal happenend when trying to merge:");
                System.out.println(getStoAlignment());
                System.out.println(source.getStoAlignment());
                System.out.println("first profile position: " + (pos1+1) + " with structure: " +
                                      StructLine.substring(pos1, pos1+1));
                System.out.println("second profile position: " + (pos2+1) + " with structure: " +
                                      source.StructLine.substring(pos2, pos2+1));
                System.exit(1);

            }

            if( pos1 > aEnd) {
                while( pos2 <= bEnd) {
                    i++;
                    rowIndex = 0;
                    for(int j=0; j<nRows; j++)
                        newAlignmentMatrix[rowIndex++][i] = -1;
                    for(int j=0; j<source.nRows; j++){
                        if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                            newAlignmentMatrix[rowIndex++][i] = source.AlignmentMatrix[j][pos2];
                        }
                    }

                    StructLineBuff.append(source.StructLine.substring(pos2, pos2+1));
                    pos2 ++;
                }
                break;
            }

            if( pos2 > bEnd) {
                while( pos1 <= aEnd) {
                    i++;
                    rowIndex = 0;
                    for(int j=0; j<nRows; j++)
                        newAlignmentMatrix[rowIndex++][i] = AlignmentMatrix[j][pos1];
                    for(int j=0; j<source.nRows; j++){
                        if( contains(((RNA)source.MemberRNAs.get(j)).getName()) == false){
                            newAlignmentMatrix[rowIndex++][i] = -1;
                        }
                    }

                    StructLineBuff.append(StructLine.substring(pos1, pos1+1));
                    pos1 ++;
                }
                break;
            }
        }

        // expunge those alignment columns which are pure gap
        boolean[] good = new boolean[newColumns];
        int nGood = 0;
        for(int i=0; i<newColumns; i++) {
            good[i] = false;
            for(int j=0; j<newRows; j++) {
                if( newAlignmentMatrix[j][i] != -1) {
                    good[i] = true;
                    nGood ++;
                    break;
                }
            }
        }

        AlignmentMatrix = new int[newRows][nGood];
        for(int i=0, k=0; i<newColumns; i++) {
            if( good[i] == true) {
                for(int j=0; j<newRows; j++)
                    AlignmentMatrix[j][k] = newAlignmentMatrix[j][i];
                k++;
            } else
                StructLineBuff.deleteCharAt(i);
        }

        StructLine = new String(StructLineBuff);
        MemberRNAs = newMemberRNAs;
        nColumns = nGood;
        nRows = newRows;
        profileStruct = null;
        Measures = null;
        //System.out.println("after merging, the result is: ");
        //System.out.println(getStoAlignment());
    }

    // -------------------------------------------------------------
    //  generates the structure-based multiple alignment in
    //  STOCKHOLM1.0 format.
    // ------------------------------------------------------------
    public String getStoAlignment()
    {
        StringBuffer FinalResult = new StringBuffer();

        FinalResult.append("# STOCKHOLM 1.0\n\n");
        StringBuffer[] lines = new StringBuffer[nRows];
        StringBuffer OneStructLine = null;

        boolean stop = false;
        int col = 0;
        int repeat = 0;
        while( stop == false ) {
            OneStructLine = new StringBuffer();
            OneStructLine.append("#=GC SS_cons ");

            for(int i=0; i<nRows; i++) {  // preparing the leading header of each block
                lines[i] = new StringBuffer();
                lines[i].append( ((RNA)MemberRNAs.get(i)).getName() );
                int offset = 25 - lines[i].length();
                for(int j = 0; j < offset; j ++)
                    lines[i].append(" ");

                offset = 33 - lines[i].length();
                for(int j = 0; j < offset; j ++)
                    lines[i].append(" ");
            }

            int offset = 33 - OneStructLine.length();
            for(int j=0; j<offset; j++)
                OneStructLine.append(" ");

            for(int i=0; i<50 && col<nColumns; i++) {
                for(int j=0; j<nRows; j++) {
                    String base = ((RNA)MemberRNAs.get(j)).getBase(AlignmentMatrix[j][col]);
                    if( base != null) 
                        lines[j].append(base);
                    else
                        lines[j].append("-");
                }
                col ++;
            }

            for(int i=0; i<nRows; i++) {
                lines[i].append("\n");
                FinalResult.append(lines[i]);
            }

            OneStructLine.append(StructLine.substring(repeat*50, col) + "\n");
            FinalResult.append(OneStructLine);
            repeat ++;

            if( col >= nColumns ) {
                FinalResult.append("//\n");
                stop = true;
            }
            else
                FinalResult.append("\n");
        }
        return new String(FinalResult);
    }


    // -----------------------------------------------------------
    // retrieve the derivative profile RNA structure associated with
    // this multi-alignment.
    //------------------------------------------------------------
    public RNA getProfileRNA()
    {
        if( profileStruct == null )
            setupProfile();
        return profileStruct;
    }

    // -----------------------------------------------------------------
    // Accessors: get the number of rows/columns in the multi-alignment
    // -----------------------------------------------------------------
    public int getRow()
    {
         return nRows;
    }
    public int getCol()
    {
        return nColumns;
    }

    //--------------------------------------------------------------------
    // get the nucleotide at specific position [row, col] in the alignment
    // the gap is represented as "-".
    //--------------------------------------------------------------------
    public String getBase(int row, int col)
    {
        if( row < nRows && col < nColumns ) {
            String base = ((RNA)MemberRNAs.get(row)).getBase(AlignmentMatrix[row][col]);
            if( base != null)
                return base;
            else
                return "-";
        } else {
            System.out.println("Fatal error: trying to access multi-alignment outbound!\n");
            System.exit(0);
        }
        return "-";
    }

    //------------------------------------------
    // Prepare the Weigth Matrix for PatSearch
    //------------------------------------------
    public String getPatWeight()
    {
        int[][] weightMatrix = new int[4][nColumns];
        for(int i=0; i<nColumns; i++)
        for(int j=0; j<nRows; j++){
            String nt = getBase(j, i);
            if( nt.equals("A") )
                 weightMatrix[0][i] ++;
            else if( nt.equals("C") )
                 weightMatrix[1][i] ++;
            else if( nt.equals("G") )
                 weightMatrix[2][i] ++;
            else if( nt.equals("T") || nt.equals("U") )
                 weightMatrix[3][i] ++;
        }

        StringBuffer ret = new StringBuffer();
        int count = 0;
        for(int i=0; i<nColumns; i++){
            int total = weightMatrix[0][i] + weightMatrix[1][i] +
                        weightMatrix[2][i] + weightMatrix[3][i];
            
            weightMatrix[0][i] = (weightMatrix[0][i] * 100)/total;
            weightMatrix[1][i] = (weightMatrix[1][i] * 100)/total;
            weightMatrix[2][i] = (weightMatrix[2][i] * 100)/total;
            weightMatrix[3][i] = (weightMatrix[3][i] * 100)/total;
            
            if( ret.length() != 0)
                ret.append(",");
            ret.append("(" + weightMatrix[0][i] + "," + weightMatrix[1][i] + ","
                        + weightMatrix[2][i] + "," + weightMatrix[3][i] + ")");
            count ++;
            if( count % 4 == 0 )
                ret.append("\n");
        }

        ret.insert(0, "{");
        ret.append("}");
        return ret.toString();
    }


    // --------------------------------------------------------------
    // test whether the given RNA (represented as an ID string) is a 
    // member RNA of the multi-alignment.
    // --------------------------------------------------------------
    public boolean contains(String id)
    {
        for(int i=0; i<nRows; i++){
            if( ((RNA)MemberRNAs.get(i)).getName().equals(id) )
                return true;
        }
        return false;
    }

    public String[] getNames()
    {
        String[] ret = new String[nRows];
        for(int i=0; i<nRows; i++)
            ret[i] = ((RNA)MemberRNAs.get(i)).getName();

        return ret;
    }

    //------------------------------------------------------------------------------------
    // A profile structure is setup from the multi-alingment by replacing each column by 
    // IUB code. Along with the profile, a component-position dependent scoring matrix is 
    // also established.
    //------------------------------------------------------------------------------------
    private void setupProfile()
    {
        StringBuffer profileSeq = new StringBuffer();
        for(int i=0; i<nColumns; i++) {
            StringBuffer AlignedCol = new StringBuffer();
            for(int j=0; j<nRows; j++) {
                String nt = ((RNA)MemberRNAs.get(j)).getBase(AlignmentMatrix[j][i]);
                if ( nt != null )
                    AlignedCol.append(nt); 
                else
                    AlignedCol.append(" ");

            }
            profileSeq.append( getIUBCode(AlignedCol.toString()) );
        }

        profileStruct = new RNA("profile_RNA", profileSeq.toString(), StructLine, "");
    }

    //-----------------------------------------------------------------------
    // setup the IUB code translation table. The IUB code information
    // SHOULD be stored within the configuration file: "codeTable.properties"!
    //-----------------------------------------------------------------------
    private void setupCodeTable()
    {
        Properties prop = null;
        try {
            prop = new Properties();
            InputStream in = new FileInputStream("codeTable.properties");
            prop.load(in);
            in.close();
        } catch( IOException e) {
            System.out.println("Fatal Error with codeTable.propertise!");
            e.printStackTrace();
        }
                                                                                                                             
        codeTable = new Hashtable();
        Enumeration keys = prop.propertyNames();
        while( keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            StringBuffer codebuffer = new StringBuffer("");
                                                                                                                             
            String words = prop.getProperty(key);
            StringTokenizer t = new StringTokenizer(words, "|");
            while( t.hasMoreTokens())
                codebuffer.append(t.nextToken());

            char[] code = codebuffer.toString().toCharArray();
            Arrays.sort(code);
            codeTable.put(new String(code), key);
        }
    }

    // translate one aligned column to an IUB code 
    private String getIUBCode(String aString)
    {
        if( codeTable == null)
            setupCodeTable();

        StringBuffer key = new StringBuffer();
        if( aString.indexOf("A") != -1)
            key.append("A");
        if( aString.indexOf("C") != -1)
            key.append("C");
        if( aString.indexOf("G") != -1)
            key.append("G");
        if( aString.indexOf("U") != -1 || aString.indexOf("T") != -1)
            key.append("U");

        String ret = (String)codeTable.get(key.toString());
        if( ret == null) {
            System.out.println("fatal error: no IUB code for " + aString);
            System.exit(1);
        }
        return ret;
    }
}
