/************************************************************************
 *                                                                      *
 *       (c) Copyright 2003                                             *
 *       All rights reserved                                            *
 *       Programs written by Jianghui Liu (NJIT)                        *
 *                                                                      *
 *       Permission to use, copy, modify, and distribute this software  *
 *       and its documentation for any purpose and without fee is her-  *
 *       eby granted, provided that this copyright notice appears in    *
 *       all copies.   Programmer(s) makes no representations about     *
 *       the suitability of this software for any purpose.  It is pro-  *
 *       vided "as is" without express or implied warranty.             *
 *                                                                      *
 *       08/06/2003                                                     *
 ************************************************************************/
package RNA;
import java.io.*;
import java.util.*;
import java.text.NumberFormat;

public class ProfileSearcher 
{
    private MultiAlignment profile;


    public ProfileSearcher(MultiAlignment seed)
    {
        profile = seed;
    }

    public static void main(String[] args)
    {
        if( args.length == 0 || args[0].equalsIgnoreCase("-h") || args[0].equalsIgnoreCase("-help"))
        {
            System.out.println("Usage: ");
            System.out.println("   java ProfileSearcher -d <database> [-q <query> | -f <seed>] ");
            System.out.println("   [-r <repeat> | -n <top>] -o <output> -p <gap penalty>");
            System.out.println("   Default Settings: ");
            System.out.println("   Output File (Default): junk.out");
            System.out.println("   Gap penalty (Default): -0.5");
            System.exit(0);
        }

        String strDB=null;
        double Penalty = -0.5;
        String Output_file = "junk.out";
        int repeat = 4;    // iteratively searching 4 times
        int topN = 0;     // if single iteration, only 'topN' hits are returned
        String SeedFile = null;
        String QueryFile = null;

        for(int i=0; i<args.length; i++) {
            if( args[i].equalsIgnoreCase("-d")) {
                strDB = new String(args[i+1]);
                i++;
            } else if( args[i].equals("-q")) {
                QueryFile = args[i+1];
                i++;
            } else if( args[i].equalsIgnoreCase("-f")) {
                SeedFile = args[i+1];
                i++;
            } else if( args[i].equalsIgnoreCase("-r")) {
                repeat = Integer.parseInt(args[i+1]);
                i++;
            } else if ( args[i].equalsIgnoreCase("-n")) {
                topN =  Integer.parseInt(args[i+1]);
                i ++;
            } else if( args[i].equalsIgnoreCase("-p")) {
                Penalty = Double.parseDouble(args[i+1]);
                i++;
            } else if( args[i].equalsIgnoreCase("-o")) {
                Output_file = args[i+1];
                i++;
            } else {
                System.out.println("Illegal parameter " + args[i]);
                System.exit(0);
            }
        }

        if( strDB == null || (QueryFile == null && SeedFile == null) ) {
            System.out.println(" parameters not set up properly!");
            System.out.println("Try '-h' for help");
            System.exit(0);
        }
        System.out.println("\n\nYour Settings are:");
        if( SeedFile != null )
            System.out.println("Seed Query file: " + SeedFile);
        else
            System.out.println("Structure Query file: " + QueryFile);

        if( topN > 0)
            System.out.println("# of top hits: " + topN);
        else
            System.out.println("Iteration number: " + repeat);

        System.out.println("Gap Penalty: " + Penalty);
        System.out.println("Output File: " + Output_file + "\n");

        RNA query = null;
        try {
            BufferedReader in = new BufferedReader(new FileReader(QueryFile));
            query = RNAReader.getRNA(in);
            in.close();
        } catch(IOException e) {
            e.printStackTrace();
            System.exit(1);
        }

        MultiAlignment seed = null;
        if( SeedFile != null)
            seed = new MultiAlignment(SeedFile);
        else
            seed = HomoUtil.getSeedAlignment(QueryFile, query.getName(), strDB, null);
        if( seed == null) {
            System.out.println("\nFATAL ERROR: NO seed constructed!\n");
            System.exit(0);
        }
        ProfileSearcher searcher = new ProfileSearcher(seed);
 
        System.out.println("\n\nStart searching database ...");
        try {
            PrintWriter output = new PrintWriter(new FileWriter(Output_file));

            for(int i=0; i<repeat; i++) {
                double score = searcher.searchDB(strDB, Penalty);
                String res = searcher.getProfile(Penalty);

                //System.out.println("\n" + (i+1) + "th iteration, got maximum score: " + score);
                //System.out.println("\nThe updated structural multi-aligment is:\n" + res);

                if( score < 0)
                    break;

                //output.println("" + (i+1) + "th iteration find maximum score of " + score);
                //output.println("\nThe updated profile is:\n" + res);
            }

            if( topN > 0){
                output.println("\nSearch for the top " + topN + " hits using current multi-alignment:");
                output.println("\n=== Database Search Result ===\n");
                output.println(searcher.searchDBforTopN(strDB, Penalty, topN));
            }

            System.out.println("\n Result can be found at " + Output_file);
            System.out.println("\nDone!\n");
            output.close();
        } catch(IOException e) {
            e.printStackTrace();
            System.exit(0);
        }
    }

    //---------------------------------------------------------------------------
    // Using the current profile to search database. The result is that the size
    // of the profile increases by including one more structure in the multi-
    // alignment.
    //---------------------------------------------------------------------------
    public double searchDB(String db, double penalty)
    {
        ProfileMatcher matcher = new ProfileMatcher(profile);

        //System.out.println("\nCurrent scoring matrix:");
        //System.out.println(matcher.dispMatrix());

        RNA profileRNA = profile.getProfileRNA();
        int[] profileMap = new int[profileRNA.seqLength()];

        RNAReader in = new RNAReader(db);
        RNA data = in.getNextRNA();
        double max = Double.NEGATIVE_INFINITY;
        RNA holder = null;
        int count = 0;
        String doll = new String("-\\|/");
        while( data != null) {
            count ++;
            int dummy = count % 4;
            System.out.print("\rWaiting " + count + " ... " + doll.substring(dummy, dummy+1));

            if( profile.contains(data.getName()) == true) {
                data = in.getNextRNA();
                continue;
            }

            double score = RNA.profileMatch2D(profileRNA, data, null, null, matcher, penalty);
            if( score > max) {
                max = score;
                holder = data;
            }

            data = in.getNextRNA();
        }

        if( holder != null) {
            for(int k=0; k<profileMap.length; k++)
                profileMap[k] = -1;

            int[] dataMap = new int[holder.seqLength()];
            for(int k=0; k<dataMap.length; k++)
                dataMap[k] = -1;

            RNA.profileMatch2D(profileRNA, holder, profileMap, dataMap, matcher, penalty);
            String interAlign = RNA.alignMatch(profileRNA, profileMap, holder, dataMap);
            if( interAlign != null) {
                //System.out.println("\nAlignment against profile structure is:");
                //System.out.println(interAlign);
                profile.expand(profileMap, holder, dataMap);
            } else {
                //System.out.println("\nNo meaningful result(s) are found!");
            }
        } else {
            //System.out.println("\nNo meaningful result(s) are found!");
            //System.exit(0);
        }

        return max;
    }

    //---------------------------------------------------------------------------
    // Using the current profile to search database. Top 'n' hits are returned.
    //---------------------------------------------------------------------------
    public String searchDBforTopN(String db, double penalty, int topN)
    {
        StringBuffer FinalResult = new StringBuffer();
        String[] results = new String[2*topN];
        String[] alignments = new String[2*topN];

        RNA[] topRNAs = new RNA[2*topN];
        double[] topScores = new double[2*topN];
        for(int i=0; i<topScores.length; i++)
            topScores[i] = Double.NEGATIVE_INFINITY;

        ProfileMatcher matcher = new ProfileMatcher(profile);

        RNA profileRNA = profile.getProfileRNA();
        int[] profileMap = new int[profileRNA.seqLength()];

        RNAReader in = new RNAReader(db);
        RNA data = in.getNextRNA();

        int count = 0;
        String doll = new String("-\\|/");
        while( data != null) {
            count ++;
            int dummy = count % 4;
            System.out.print("\rWaiting " + count + " ... " + doll.substring(dummy, dummy+1));

            if( profile.contains(data.getName()) == true) {
                data = in.getNextRNA();
                continue;
            }

            int[] dataMap = new int[data.seqLength()];
            for(int j=0; j<dataMap.length; j++)
               dataMap[j] = -1;

            for(int j=0; j<profileMap.length; j++)
                profileMap[j] = -1;

            double score = RNA.profileMatch2D(profileRNA, data, profileMap, dataMap, matcher, penalty);
            int slot = getSlot(topScores, score);
            if( slot != -1) {
                boolean found = false;
                for(int k=0; k<topRNAs.length; k++){
                    if(k!=slot && topRNAs[k]!=null && topRNAs[k].getName().equals(data.getName()) ){
                        if( score > topScores[k]){
                            slot = k;
                            break;
                        } else {
                            found = true;
                            break;
                        }
                    }
                }

                if( found == false ){
                    topScores[slot] = score;
                    topRNAs[slot] = data;
                    int[] dataRegion = HomoUtil.getAlignedRegion(data, dataMap);
                    results[slot] = fixedWidth( ((dataRegion[0]+1) + "-" + (dataRegion[1]+1)), 7) +
                                    " " + fixedWidth(data.getName(), 20) + " " + fixedWidth(data.getAnnotate(), 50);
                    alignments[slot] = new String(RNA.alignMatch(profileRNA, profileMap, data, dataMap));
                }
            }

            data = in.getNextRNA();
        }
 
        FinalResult.append("\n#=== Query Seed ===#\n\n");
        FinalResult.append(getProfile(penalty)+"\n");
        FinalResult.append("#=== Hits ===#\n\n");

        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(2);
        
        double threshold = Double.NEGATIVE_INFINITY;
        double score, preScore = 0;
        int rank = 1;
        int i = 0;

        StringBuffer detail = new StringBuffer();
        do {
            int slot = getTopRank(topScores);
            score = topScores[slot];
            if( score <= Double.NEGATIVE_INFINITY )
                break;

            if( score == preScore ) {
                String strRank = new String("" + rank);
                FinalResult.append("" + fixedWidth(strRank, 4) + " " + fixedWidth(nf.format(score), 6) + " "
                           + results[slot] + "\n");
            } else {
                rank = i + 1;
                String strRank = new String("" + rank);
                FinalResult.append("" + fixedWidth(strRank, 4) + " " + fixedWidth(nf.format(score), 6) + " "
                           + results[slot] + "\n");
            }

            detail.append("\nRank: " + rank + "  Score: " + nf.format(score));
            detail.append(alignments[slot]);

            preScore = score;
            i ++;
            if( i == topN )
                threshold = score;
      
            topScores[slot] = Double.NEGATIVE_INFINITY;
        } while ( score >= threshold);
        
        FinalResult.append("\n===== Detailed results ====\n");
        FinalResult.append(detail);
        return FinalResult.toString();
    }
        
    //---------------------------------------------------------------
    // Get current profile in multi-alignment format: STOCKHOLM 1.0
    //---------------------------------------------------------------
    public String getProfile(double penalty)
    {
        double measure[] = profile.getMeasures(penalty);
        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(2);
        String head = "Min: " + nf.format(measure[0]) + "\tMax: " + nf.format(measure[1]) +
                      "\tAvg: " + nf.format(measure[2]) + "\n";
        return head + profile.getStoAlignment();
    }

    public String[] getNames()
    {
        return profile.getNames();
    }

    public String getPatWeight()
    {
        return profile.getPatWeight();
    }

    // ---------------------------------------------------
    // Given a SET, find the minimum item which is smaller
    // than the given the item.
    // ---------------------------------------------------
    private int getSlot(double[] scores, double oneScore)
    {
        double min = scores[0];
        int slot = 0;
        for(int i=1; i<scores.length; i++)
        {
            if( scores[i] < min)
            {
                min = scores[i];
                slot = i;
            }
        }

        if( scores[slot] < oneScore)
            return slot;
        else
            return -1;
    }


    private String fixedWidth(String str, int width)
    {
        StringBuffer ret = new StringBuffer();
        int length = str.length();
        if( length > width)
            length = width;
        ret.append(str.substring(0, length));

        for(int i=0; i<width-length; i++)
            ret.append(" ");
        return ret.toString();
    }

    //--------------------------------------
    // From a set, select the largest item
    //--------------------------------------
    private int getTopRank(double[] scores)
    {
        double max = scores[0];
        int slot = 0;
        for(int i=1; i<scores.length; i++)
        {
            if( scores[i] > max)
            {
                max = scores[i];
                slot = i;
            }
        }


        return slot;
    }

}
