#include <AbstractMatrixWriter.h>
#include <colorFunctions.h>
#include <AlignColorConstants.h>
#include <stemhelp.h>
#include <aligncolor_help.h>
#include <Letter3D.h>

AbstractMatrixWriter::AbstractMatrixWriter() : alphabet("ACGU"), 
					       arrowMode(1),
					       colorMode(5),
					       errorColor(0.7, 0.7, 0.7),
					       errorMode(1), 
					       frequencyLimit(0.05), 
					       gapMode(AlignColorConstants::LOGO_DEFAULT),
					       gridWidth(1.0),
					       groundPlateColor(0.5, 0.5, 0.5),
					       groundPlateMode(false),
					       infLimit(0.5), 
					       infLimitRelative(-1.0), 
					       infUpperLimit(-1.0),
					       legendMode(0),
					       letterMode(AlignColorConstants::LOGO_CHAR_DEFAULT),
					       letterMode2(AlignColorConstants::LOGO_CHAR_NONE), 
					       logoMode(AlignColorConstants::LOGO_DEFAULT),
					       primitiveCounter(0),
					       proteinMode(false),
					       refColor(0.0, 0.0, 0.0),
					       referenceGap(0.05), 
					       stencilColor(0.5, 0.5, 0.5), // Violet-Magenta, color of groundplate
					       stretchZ(10.0),
					       verboseLevel(2) {
}

double
AbstractMatrixWriter::getFrequency(const string& col, char c) const
{
  if (col.size() == 0) {
    return 0.0;
  }
  unsigned int num = 0;
  unsigned int numGap = 0;
  for (unsigned int i = 0; i < col.size(); ++i) {
//     if (alphabet.find(col[i]) >= alphabet.size()) {
//       continue; // ignore cases which have bad letter (including dash, "y" etc)
//     }
    if (isGap(col[i])) {
      ++numGap;
    }
    else if (col[i] == c) {
      ++num;
    }
  }
  if (numGap >= col.size()) {
    return 0.0;
  }
  // unsigned int num = count(col.begin(), col.end(), c);
  return static_cast<double>(num) / static_cast<double>(col.size()-numGap);
  // return CompensationScorer::frequency2(s1, c);
}

/** frequency of occurence of pair of letters (not counting gaps) */
double
AbstractMatrixWriter::getFrequency(const string& col1, 
			      const string& col2, const string& pair) const
{
  char c1 = pair[0];
  char c2 = pair[1];
  unsigned int minSize = col2.size();
  if (col1.size() < col2.size()) {
    minSize = col1.size();
  }
  if (minSize == 0) {
    return 0.0;
  }
  unsigned int num = 0;
  unsigned int numGap = 0;
  for (unsigned int i = 0; i < minSize; ++i) {
//     if ((alphabet.find(col1[i]) >= alphabet.size())
// 	|| (alphabet.find(col2[i]) >= alphabet.size() ) ) {
//       continue; // ignore cases which have bad letter (including dash, "y" etc)
//     }
    if (isGap(col1[i]) || isGap(col2[i])) {
      ++numGap;
    }
    else if ((col1[i]==c1) && (col2[i]==c2)) {
      ++num;
    }
  }
  if (numGap >= minSize) {
    return 0.0;
  }
  // unsigned int num = count(col.begin(), col.end(), c);
  return static_cast<double>(num) / static_cast<double>(minSize-numGap);
  // return CompensationScorer::frequency2(s1, s2, pair[0], pair[1]);
}


Vec<double>
AbstractMatrixWriter::getFrequencies(const string& s1, const string& alphabet) const
{
  Vec<double> result(alphabet.size());
  for (unsigned int i = 0; i < alphabet.size(); ++i) {
    result[i] = getFrequency(s1, alphabet[i]);
  }
  return result;
}

Vec<Vector3D>
AbstractMatrixWriter::getPairColors(const Vec<string>& pairs) const
{
  ERROR("getPairColors not supported anymore!", exception);

  Vec<Vector3D> colors; //  = generateSpherePoints(9 * pairs.size(), 
  return colors;
}

Vec<Vector3D>
AbstractMatrixWriter::getPairColors2(const Vec<string>& pairsOrig) const
{
  Vec<string> pairs = pairsOrig;
  Vec<Vector3D> result(pairs.size());
  Vec<double> mult(4);
  mult[0] = 1.0;
  mult[1] = 0.75;
  mult[2] = 0.5;
  mult[3] = 0.25;
  Vector3D red(0.99, 0.0, 0.0); // matches: AU, GC, GU
  Vector3D green(0.0, 0.99, 0.0); // same pairs: AA, CC, GG, UU
  Vector3D blue(0.0, 0.0, 0.99); // mismatches: AC, AG, CU
  for (unsigned int i = 0; i < pairs.size(); ++i) {
    sort(pairs[i].begin(), pairs[i].end()); // do not distinguish between for example AC and CA
    switch (pairs[i][0]) {
    case 'A':
      switch (pairs[i][1]) {      
      case 'A': result[i] = green;
	break;
      case 'C':result[i] = blue;
	break;
      case 'G':result[i] = blue;
	break;
      case 'T':result[i] = red;
	break;
      case 'U':result[i] = red;
	break;
      default: ERROR("Internal error in line 138!", exception);
      }
      result[i] = result[i] * mult[0];
      break;
    case 'C':
      switch (pairs[i][1]) {      
//       case 'A': result[i] = blue;
// 	break;
      case 'C':result[i] = green;
	break;
      case 'G':result[i] = red;
	break;
      case 'T':result[i] = blue;
	break;
      case 'U':result[i] = blue;
	break;
      default: ERROR("Internal error in line 402!", exception);
      }
      result[i] = result[i] * mult[1];
      break;
    case 'G':
      switch (pairs[i][1]) {      
//       case 'A': result[i] = blue;
// 	break;
//       case 'C':result[i] = red;
// 	break;
      case 'G':result[i] = green;
	break;
      case 'T':result[i] = red;
	break;
      case 'U':result[i] = red;
	break;
      default: ERROR("Internal error in line 414!", exception);
      }
      result[i] = result[i] * mult[2];
      break;
    case 'T':
      switch (pairs[i][1]) {      
	//       case 'A': result[i] = rainbow(0.95);
	// 	break;
	//       case 'C':result[i] = rainbow(0.05);
	// 	break;
	//       case 'G':result[i] = rainbow(0.75);
	// 	break;
      case 'T':result[i] = green;
	break;
      case 'U':result[i] = green;
	break;
      default: ERROR("Internal error in line 427!", exception);
      }
      result[i] = result[i] * mult[3];
      break;
    case 'U':
      switch (pairs[i][1]) {      
	//       case 'A': result[i] = rainbow(0.95);
	// 	break;
	//       case 'C':result[i] = rainbow(0.05);
	// 	break;
	//       case 'G':result[i] = rainbow(0.75);
	// 	break;
      case 'T':result[i] = green;
	break;
      case 'U':result[i] = green;
	break;
      default: ERROR("Internal error in line 427!", exception);
      }
      result[i] = result[i] * mult[3];
      break;
    }
    
  }
  return result;
}

/** use parameter file stored in ResourceBundle as lookup table for colors */
Vec<Vector3D>
AbstractMatrixWriter::getPairColors3(const Vec<string>& pairs) const
{
  ERROR("getPairColors3 not supported anymore!", exception);
  Vec<Vector3D> result(pairs.size());
  for (unsigned int i = 0; i < pairs.size(); ++i) {
    string ending = string("") + pairs[i][0] + pairs[i][1];
    sort(ending.begin(), ending.end()); // do not destinguish between for example AC and CA
    string name = "color." + ending;
    // result[i] = getResourceColor(resources.getString(name));
  }
  return result;
}

/** use rainbow encoding
 @see http://www.visibone.com/colorlab/ */
Vec<Vector3D>
AbstractMatrixWriter::getPairColors4(const Vec<string>& pairs, double mutInf) const
{
  const double MAX_MUTINF = 2.0; // 1.6; // maximum value for mutual information
  Vector3D rain = rainbow2(mutInf/MAX_MUTINF);
  if (rain.x() > 0.99) {
    rain.x(0.99);
  }
  if (rain.y() > 0.99) {
    rain.y(0.99);
  }
  if (rain.z() > 0.99) {
    rain.z(0.99);
  }
  Vec<Vector3D> result(pairs.size(),rain);
  return result;
}

/** another attempt at a good palette for 16 base pairs
    red: complementary, yellow: GU,  green: AA, CC etc, blue: rest 
*/
Vec<Vector3D>
AbstractMatrixWriter::getPairColors5(const Vec<string>& pairsOrig) const
{
  Vec<string> pairs = pairsOrig;
  Vec<Vector3D> result(pairs.size());
  Vec<double> mult(4);
  mult[0] = 1.0;
  mult[1] = 0.75;
  mult[2] = 0.5;
  mult[3] = 0.25;
  Vector3D orange(255.0, 153.0, 0.0);
  orange = orange * (1.0/256.0);
  const Vector3D red(0.99, 0.0, 0.0); // matches: AU, GC, GU
  const Vector3D darkRed(204.0/256.0, 0.0, 0.0);
  const Vector3D lightGreen(204.0/256.0, 255.0/256.0, 0.0); // same pairs: AA, CC, GG, UU
  const Vector3D green(0.0, 0.99, 0.0); // same pairs: AA, CC, GG, UU
  const Vector3D darkGreen(0.0, 204.0/256.0, 0.0); // same pairs: AA, CC, GG, UU
  const Vector3D darkGreen2(153.0/256.0, 204.0/256.0, 153.0/256.0); // same pairs: AA, CC, GG, UU. "Light Weak Green" according t
  const Vector3D darkGreen3(0.0, 102.0/256.0, 0.0); // same pairs: AA, CC, GG, UU. "Obscure Dull Green" according to 
  const Vector3D lightBlue(0.0, 0.99, 0.99); // mismatches: AC, AG, CU
  const Vector3D blue(0.0, 153.0/256.0, 0.99); // mismatches: AC, AG, CU
  const Vector3D darkBlue(0.0, 0.0, 0.99); // mismatches: AC, AG, CU
  const Vector3D yellow(0.99, 0.99, 0.0); // wobble: GU
  for (unsigned int i = 0; i < pairs.size(); ++i) {
    sort(pairs[i].begin(), pairs[i].end()); // do not distinguish between for example AC and CA
    switch (pairs[i][0]) {
    case 'A':
      switch (pairs[i][1]) {      
      case 'A': result[i] = lightGreen;
	break;
      case 'C':result[i] = blue;
	break;
      case 'G':result[i] = darkBlue;
	break;
      case 'T':result[i] = orange;
	break;
      case 'U':result[i] = orange;
	break;
      default: ERROR("Internal error in line 138!", exception);
      }
      // result[i] = result[i] * mult[0];
      break;
    case 'C':
      switch (pairs[i][1]) {      
//       case 'A': result[i] = blue;
// 	break;
      case 'C':result[i] = green;
	break;
      case 'G':result[i] = red;
	break;
      case 'T':result[i] = lightBlue;
	break;
      case 'U':result[i] = lightBlue;
	break;
      default: ERROR("Internal error in line 402!", exception);
      }
      // result[i] = result[i] * mult[1];
      break;
    case 'G':
      switch (pairs[i][1]) {      
//       case 'A': result[i] = blue;
// 	break;
//       case 'C':result[i] = red;
// 	break;
      case 'G':result[i] = darkGreen;
	break;
      case 'T':result[i] = yellow;
	break;
      case 'U':result[i] = yellow;
	break;
      default: ERROR("Internal error in line 414!", exception);
      }
      // result[i] = result[i] * mult[2];
      break;
    case 'T':
      switch (pairs[i][1]) {      
	//       case 'A': result[i] = rainbow(0.95);
	// 	break;
	//       case 'C':result[i] = rainbow(0.05);
	// 	break;
	//       case 'G':result[i] = rainbow(0.75);
	// 	break;
      case 'T':result[i] = darkGreen3;
	break;
      case 'U':result[i] = darkGreen3;
	break;
      default: ERROR("Internal error in line 427!", exception);
      }
      // result[i] = result[i] * mult[3];
      break;
    case 'U':
      switch (pairs[i][1]) {      
	//       case 'A': result[i] = rainbow(0.95);
	// 	break;
	//       case 'C':result[i] = rainbow(0.05);
	// 	break;
	//       case 'G':result[i] = rainbow(0.75);
	// 	break;
      case 'T':result[i] = darkGreen3;
	break;
      case 'U':result[i] = darkGreen3;
	break;
      default: ERROR("Internal error in line 427!", exception);
      }
      // result[i] = result[i] * mult[3];
      break;
    }
    
  }
  return result;
}

/** another attempt at a good palette for 16 base pairs
    red: complementary, yellow: GU,  green: AA, CC etc, blue: rest 
*/
Vec<Vector3D>
AbstractMatrixWriter::getPairColorsProtein1(const Vec<string>& pairsOrig) const
{
  Vec<string> pairs = pairsOrig;
  Vec<Vector3D> result(pairs.size());
  Vec<double> mult(4);
  mult[0] = 1.0;
  mult[1] = 0.75;
  mult[2] = 0.5;
  mult[3] = 0.25;
  Vector3D orange(255.0, 153.0, 0.0);
  orange = orange * (1.0/256.0);
  const Vector3D red(0.99, 0.0, 0.0); // matches: AU, GC, GU
  const Vector3D darkRed(204.0/256.0, 0.0, 0.0);
  const Vector3D lightGreen(204.0/256.0, 255.0/256.0, 0.0); // same pairs: AA, CC, GG, UU
  const Vector3D green(0.0, 0.99, 0.0); // same pairs: AA, CC, GG, UU
  const Vector3D darkGreen(0.0, 204.0/256.0, 0.0); // same pairs: AA, CC, GG, UU
  const Vector3D darkGreen2(153.0/256.0, 204.0/256.0, 153.0/256.0); // same pairs: AA, CC, GG, UU. "Light Weak Green" according t
  const Vector3D darkGreen3(0.0, 102.0/256.0, 0.0); // same pairs: AA, CC, GG, UU. "Obscure Dull Green" according to 
  const Vector3D lightBlue(0.0, 0.99, 0.99); // mismatches: AC, AG, CU
  const Vector3D blue(0.0, 153.0/256.0, 0.99); // mismatches: AC, AG, CU
  const Vector3D darkBlue(0.0, 0.0, 0.99); // mismatches: AC, AG, CU
  const Vector3D yellow(0.99, 0.99, 0.0); // wobble: GU
  for (unsigned int i = 0; i < pairs.size(); ++i) {
    sort(pairs[i].begin(), pairs[i].end()); // do not distinguish between for example AC and CA
    int id = static_cast<int>(pairs[i][0]*alphabet.size()) + static_cast<int>(pairs[i][1]);
    if (id > 255) {
      id = 255;
    }
    result[i] = Vector3D(static_cast<double>(id) / 256.0, 0.0, 0.0);
  }
  return result;
}

/** calls getPairColorsX according to colorMode */
Vec<Vector3D>
AbstractMatrixWriter::getPairColorsMode(const Vec<string>& pairs, double mutInf) const
{
  Vec<Vector3D> colors;
  switch (colorMode) {
  case 1: colors = getPairColors(pairs);
    break;
  case 2: colors = getPairColors2(pairs);
    break;
  case 3: colors = getPairColors3(pairs);
    break;
  case 4: colors = getPairColors4(pairs, mutInf); // rainbow encoding: all pairs have same color
    break;
  case 5: colors = getPairColors5(pairs);
    break;
  case 6: colors = getPairColorsProtein1(pairs);
    break;
  default: ERROR("Undefined color mode!", exception);
  }
  return colors;
}

/** returns true if mutual information is both above the simple threshold
 * and if it is enough standard deviation above zero.
 */
bool 
AbstractMatrixWriter::isInformationAboveThreshold(double mutInf,
						  double mutInfError) const
{
  if (mutInf < infLimit) {
    return false;
  }
  if ((infLimitRelative > 0.0) && (mutInfError > 0.0) && ((mutInf/mutInfError)<infLimitRelative)) {
    return false;
  }
  return true;
}

/** writes one stack of positions */
bool
AbstractMatrixWriter::writePosStack(ostream& os,
				    const Alignment& ali,
				    const Vec<Stem>& refStems,
				    unsigned int i,
				    unsigned int j,
				    double mutInf,
				    double mutInfError,
				    int letterStyle) const
{
  string s1 = ali.getSliceString(i);
  string s2 = ali.getSliceString(j);
  // double mutInf = scorer.compensationScore(s1, s2, ali.getWeights());
//   if (mutInf > 2.0) {
//     cout << "warning from writeVrmlPosStack: mutInf " << mutInf << endl;
//   }
//   const Vector3D refColor(0.0, 0.0, 0.0);
//   const Vector3D errorColor(0.7, 0.7, 0.7);
  const double refHeight = 0.1;
  bool drawFlag = isInformationAboveThreshold(mutInf, mutInfError);
  double mutInfErrorHeight = 0.01;
  if (drawFlag) {
    if (mutInfError < 0.001) {
      mutInfError = 0.001; // TODO check with previous version (0.1, maybe bug?)
    }
    Vec<string> pairs = getPairs();
    Vec<double> frequencies = getPairFrequencies(s1, s2, pairs);
    string lettersFront(pairs.size(), ' ');
    string lettersLeft(pairs.size(), ' ');
    if (letterStyle != AlignColorConstants::LOGO2_CHAR_NONE) {
      for (string::size_type ii = 0; ii < lettersFront.size(); ++ii) {
	lettersFront[ii] = pairs[ii][0];
	lettersLeft[ii] = pairs[ii][1];
      }
    }
    //   cout << "writePosStack: " << i +1 << " " << j + 1 << " " << mutInf << " " 
    //        << lettersFront << " " << lettersLeft << endl;
    Vec<Vector3D> colors = getPairColorsMode(pairs, mutInf);
    writeBoxStackBoth(os, Vector3D(static_cast<double>(i), static_cast<double>(j), 0.0),
		      stretchZ*mutInf, 0.0, frequencies, colors, lettersFront, lettersLeft, infUpperLimit);
    if (errorMode == 1) {
      if (verboseLevel > 2) {
	string errorComment = "Writing error box with height: " + dtos(stretchZ*mutInfError);
	cout << errorComment << endl;
	writeComment(os, errorComment);
      }
      writeBox(os, Vector3D(static_cast<double>(i), static_cast<double>(j), stretchZ*(mutInf + 0.5 * mutInfError)),
	       Vector3D(0.5, 0.5, stretchZ * mutInfError), errorColor, true); // true: faceMode switched on !?
      writeBox(os, Vector3D(static_cast<double>(i), static_cast<double>(j), stretchZ*(mutInf + mutInfError + 0.5 * mutInfErrorHeight)),
	       stretchZ * mutInfErrorHeight, errorColor);
      if (verboseLevel > 2) {
	string errorComment = "Done writing error box with height: " + dtos(stretchZ*mutInfError);
	writeComment(os, errorComment);
      }
    }
  }
  if (isPosInStems(i, j, refStems)) {
    if ((errorMode == 1) && drawFlag) {
      writeBox(os, Vector3D(static_cast<double>(i), static_cast<double>(j), stretchZ*(mutInf + mutInfError + mutInfErrorHeight + 0.5*refHeight + referenceGap)),
	       stretchZ * refHeight, refColor);
    }
    else {
      writeBox(os, Vector3D(static_cast<double>(i), static_cast<double>(j), stretchZ*(mutInf + 0.5*refHeight + referenceGap)),
	       stretchZ * refHeight, refColor);
    }
  }
  return drawFlag;
}

void
AbstractMatrixWriter::copy(const AbstractMatrixWriter& other) {
  alphabet = other.alphabet;
  arrowMode = other.arrowMode;
  colorMode = other.colorMode;
  errorColor = other.errorColor;
  errorMode = other.errorMode;
  frequencyLimit = other.frequencyLimit;
  gapMode = other.gapMode;
  gridWidth = other.gridWidth;
  groundPlateColor = other.groundPlateColor;
  groundPlateMode = other.groundPlateMode;
  infLimit = other.infLimit;
  infLimitRelative = other.infLimitRelative;
  infUpperLimit = other.infUpperLimit;
  legendMode = other.legendMode;
  letterMode = other.letterMode;
  letterMode2 = other.letterMode2;
  logoMode = other.logoMode;
  primitiveCounter = other.primitiveCounter;
  proteinMode = other.proteinMode;
  refColor = other.refColor;
  referenceGap = other.referenceGap;
  stencilColor = other.stencilColor;
  stretchZ = other.stretchZ;
  verboseLevel = other.verboseLevel;
  wwwImageBase = other.wwwImageBase;
}

Vec<double>
AbstractMatrixWriter::getPairFrequencies(const string& s1, const string& s2, const Vec<string>& pairs) const
{
  Vec<double> result(pairs.size());
  for (unsigned int i = 0; i < pairs.size(); ++i) {
    result[i] = getFrequency(s1, s2, pairs[i]);
  }
  return result;
}

/** returns all possible 16 pairs */
Vec<string>
AbstractMatrixWriter::getPairs() const
{
  unsigned int nn = alphabet.size();
  Vec<string> pairs;
  string s = "xx";
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < nn; ++j) {
      s[0] = alphabet[i];
      s[1] = alphabet[j];
      pairs.push_back(s);
    }
  }
  return pairs;
}

/* TODO : drawing of error stack not yet implemented */
void
AbstractMatrixWriter::writeBoxStackBoth(ostream& os,
					const Vector3D& pos,
					double height, // this value has be scaled with "stretchZ" before calling this method !
					double errorHeight, // this value has be scaled with "stretchZ" before calling this method !
					const Vec<double>& frequencies,
					const Vec<Vector3D>& colors,
					const string& lettersFront,
					const string& lettersLeft,
					double localInfUpperLimit) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
//   if (wwwImageBase.size() == 0) { // no texture mapping
//     writeBoxStack(os, pos, height, frequencies, colors);
//     return;
//   }
  string comment = "start writeBoxStack with letters " + dtos(height);
  Vector3D capColor(0.99, 0.99, 0.99); // use white to indicate capped height
  writeComment(os, comment);
  double currentHeight = 0.0;
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  if ((localInfUpperLimit > 0) && (height > (localInfUpperLimit * stretchZ) )) {
    return; // upper limit activated!
  }
  const Vector3D black(0.0, 0.0, 0.0);
  const Vector3D white(0.99, 0.99, 0.99);
  double emptyHeight = 0.0;
  Vec<unsigned int> order = documentedOrder(frequencies);
  bool drawn = false;
  bool capMode = false;
  int letterStyle = 0; // TODO : different letter styles?
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
      drawn = true;
      double usedFreq = frequencies[i];
      Vector3D usedColor = colors[i];
      if ((localInfUpperLimit > 0) && (height > 0) && ((currentHeight + height * usedFreq) > (stretchZ * localInfUpperLimit))) {
	usedFreq = ((stretchZ * localInfUpperLimit) - currentHeight) / height;
	capMode = true; // height was capped!
	usedColor = capColor; // draw in special color
      }
      if (ii == (colors.size()-1)) {
	// if (letterMode2 == AlignColorConstants::LOGO2_CHAR_TEXTURE) {
	writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), currentHeight + (0.5 * height * usedFreq)),
			   height*usedFreq, usedColor, lettersFront[i], lettersLeft[i], letterStyle);
      }
      else { // not topmost block
	writeOpenBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
					    currentHeight + (0.5 * height * usedFreq)),
			       height*usedFreq, usedColor, lettersFront[i], lettersLeft[i], letterStyle);
	
      }
      if (capMode) {
	return;
      }
    }
    else {
      emptyHeight += height*frequencies[i]; // accumulate height of boxes not drawn
    }
    currentHeight += height * frequencies[i];
  }
  if (drawn && (emptyHeight > 0.0)) {
    writeBox(os, Vector3D(pos.x(), pos.y(), 0.0 + (0.5 * emptyHeight)),
	     Vector3D(1.0, 1.0, emptyHeight), white, true ); // draw cube with white faces
  }
  if (errorMode) {
    if (errorHeight < 0.001) {
      errorHeight = 0.001;
    }
    double topWidth = 0.1; // height of top "hat plate"
    writeBox(os, 
	     Vector3D(pos.x(), pos.y(), currentHeight + (0.5 * errorHeight)),
	     Vector3D(0.5, 0.5, errorHeight), errorColor, false);
    writeBox(os, 
	     Vector3D(pos.x(), pos.y(), currentHeight + errorHeight + (0.5*topWidth)),
	     Vector3D(1.0, 1.0, topWidth), errorColor, false);
  }
  // draw "stencil": needed for 3D printing!
  if (groundPlateMode) {
    // cout << "Writing stencil: " << pos.x() << " " << pos.y() << " " << gridWidth << " " << stencilColor << endl;
    writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), -0.5*gridWidth), gridWidth, stencilColor, ' ', ' ', letterStyle);
  }

}

Vector3D
AbstractMatrixWriter::getResourceColor(const string& name) const
{
  Vector3D result(0.0, 0.0, 0.0);
  string s;
  double  x = 0;
  double  y = 0;
  double  z = 0;
  string trans=resources.getString(name);
  
  vector<string> words = getTokens(trans);
  // cout << "Found number strings: " << name << endl;
//   for (unsigned int i = 0; i < words.size(); ++i) {
//     cout << words[i] << " ";
//   }
  if (words.size() != 3) {
    cout << name << " " << trans << " " << words.size() << endl;
    ERROR_IF(words.size() != 3,
	     "Three values for red green blue expected in color definition!",
	     exception);
  }
  x = stod(words[0]); // convert to double
  y = stod(words[1]); // convert to double
  z = stod(words[2]); // convert to double

//   cout << "The resource int is: " << " "
//        << x << " " << y << " " << z << endl;
  result.x(x);
  result.y(y);
  result.z(z);

  // cout << "The resource color is: " << result << endl;
  return result; 
}

Vector3D
AbstractMatrixWriter::getSingleColor(char letter) const
{  
//   const Vector3D white(0.99, 0.99, 0.99);
//   const Vector3D yellow(0.99, 0.99, 0.0);
//   const Vector3D magenta(204.0/256.0, 0.0, 0.99);
//   const Vector3D black(0.0, 0.0, 0.0);
  const Vector3D red(0.99, 0.0, 0.0);
  const Vector3D blue(0.0, 0.9372, 0.99);
  const Vector3D green(0.0, 0.99, 0.0);
  const Vector3D orange(0.99, 0.7, 0.0);
  if (resources.size() > 0) {
    string resourceName;
    switch (letter) {
    case 'A': 
      resourceName = "color.A";
      break;
    case 'C':
      resourceName = "color.C";
      break;
    case 'G':
      resourceName = "color.G";
      break;
    case 'T':
      resourceName = "color.T";
      break;
    case 'U':
      resourceName = "color.U";
      break;
    }
    // cout << "The resource name is: " << resourceName << endl;
    return getResourceColor(resourceName);
  }
  else if (!proteinMode) { 
    switch (letter) {
    case 'A': 
      return green;
      break;
    case 'C': return blue;
      break;
    case 'G':return orange;
      break;
    case 'T':return red;
      break;
    case 'U':return red;
      break;
    default: ERROR("Internal error in line 463!", exception);
    }
  }
  else { 
    switch (letter) {
    case 'A': 
      return green;
      break;
    case 'C': return blue;
      break;
    case 'D':return orange;
      break;
    case 'E':return red;
      break;
    case 'F':return red;
      break;
    case 'G': 
      return green;
      break;
    case 'H': return blue;
      break;
    case 'I':return orange;
      break;
    case 'K':return red;
      break;
    case 'L':return red;
      break;
    case 'M': 
      return green;
      break;
    case 'N': return blue;
      break;
    case 'P':return orange;
      break;
    case 'Q':return red;
      break;
    case 'R':return red;
      break;
    case 'S':return red;
      break;
    case 'T':return red;
      break;
    case 'V':return red;
      break;
    case 'W':return red;
      break;
    case 'Y':return red;
      break;
    default: ERROR("Internal error in line 719!", exception);
    }
  }
  ERROR("Undefined single color detected!", exception);
  return Vector3D(0.0, 0.0, 0.0);
}

Vec<Vector3D>
AbstractMatrixWriter::getSingleColors(const string& letters) const
{
  Vec<Vector3D> result(letters.size());
  for (unsigned int i = 0; i < letters.size(); ++i) {
    result[i] = getSingleColor(letters[i]);
  }
  return result;
}

void
AbstractMatrixWriter::writeLogo(ostream& os,
				const Alignment& ali,
				double inf,
				double infError,
				int pos) const
{
  string slice = ali.getSliceString(pos);
  Vec<double> frequencies = getFrequencies( slice, alphabet);
  Vec<Vector3D> colors = getSingleColors(alphabet);
  // double inf = scorer.singleEntropy9(slice);
  string spaceString(alphabet.size(), ' ');
  string comment = "starting to write logo position " + itos(pos)
    + " inf: " + dtos(inf) + " " + dtos(stretchZ*inf);
  writeComment(os, comment);
  string usedLetters = alphabet;
  if (letterMode == AlignColorConstants::LOGO_CHAR_NONE) {
    usedLetters = spaceString; // no characters if switched off
  }
//   cout << comment << endl;
//   cout << "Used letters by logo: " << usedLetters << endl;
//   cout << "Calling writeBoxStackBoth!" << endl;
  writeBoxStackBoth(os, Vector3D(pos, -5.0, 0.0), stretchZ*inf, stretchZ*infError,
		    frequencies, colors, usedLetters, spaceString, -1);
  writeBoxStackBoth(os, Vector3D(-5, pos, 0.0), stretchZ*inf, stretchZ*infError,
 		    frequencies, colors, spaceString, usedLetters, -1); // -1: do not use upper limit 
}

void
AbstractMatrixWriter::writeLogoFlat(ostream& os,
				    const Alignment& ali,
				    double inf,
				    double infError,
				    int pos) const
{
  string slice = ali.getSliceString(pos);
  Vec<double> frequencies = getFrequencies( slice, alphabet);
  Vec<Vector3D> colors = getSingleColors(alphabet);
  // double inf = scorer.singleEntropy9(slice);
  string spaceString(alphabet.size(), ' ');
  string comment = "starting to write logo position " + itos(pos);
  writeComment(os, comment);
  string usedLetters = alphabet;
  if (letterMode == AlignColorConstants::LOGO_CHAR_NONE) {
    usedLetters = spaceString; // no characters if switched off
  }
  // TODO : Letter3D::FACE_XY90 just made up! Verify!
  writeBoxStackFlat(os, Vector3D(pos, -5.0, 0.0), stretchZ*inf, stretchZ*infError, 
		    frequencies, colors, usedLetters, Letter3D::FACE_XY180, letterMode);
  writeBoxStackFlat2(os, Vector3D(-5, pos, 0.0), stretchZ*inf, stretchZ*infError,
		     frequencies, colors, usedLetters, Letter3D::FACE_XY90, letterMode);
}

void
AbstractMatrixWriter::writeLogos(ostream& os, const Alignment& ali,
				 const Vec<double>& infVec,
				 const Vec<double>& infErrorVec) const
{
#ifdef VRML2MATRIXWRITER_VERBOSE
  cout << "starting writeVrmlLogos!" << endl;
#endif
  string comment = "starting to write logos";
  writeComment(os, comment);
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    writeLogo(os, ali, infVec[i], infErrorVec[i], i);
  }
  double szd = ali.getLength();
  const Vector3D black = Vector3D(0.0, 0.0, 0.0);
  double bw = 0.25;
  const double LD20 = log2(20.0);
  if (!proteinMode) {
    writeBox(os, 
	     Vector3D(- 5, 0.5 * szd, 2.0 * stretchZ + 0.5),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(- 5, 0.5 * szd, - 0.5),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5, 2.0 * stretchZ + 0.5),
	     Vector3D(szd, bw, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5 , - 0.5),
	     Vector3D(szd, bw, bw),
	     black);
  }
  else {
    writeBox(os, 
	     Vector3D(- 5, 0.5 * szd, LD20 * stretchZ + 0.5),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(- 5, 0.5 * szd, - 0.5),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5, LD20 * stretchZ + 0.5),
	     Vector3D(szd, bw, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5 , - 0.5),
	     Vector3D(szd, bw, bw),
	     black);
  }
  
}


void
AbstractMatrixWriter::writeLogosFlat(ostream& os, const Alignment& ali,
				     const Vec<double>& infVec,
				     const Vec<double>& infErrorVec) const
{
  if (verboseLevel > 2) {
    cout << "starting writeLogos (flat)!" << endl;
  }
  string comment = "starting to write logos (flat)";
  writeComment(os, comment);
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    writeLogoFlat(os, ali, infVec[i], infErrorVec[i], i);
  }
  double szd = ali.getLength();
  const Vector3D black = Vector3D(0.0, 0.0, 0.0);
  double bw = 0.25;
  const double LD20 = log2(20.0);
  if (!proteinMode) {
    writeBox(os, 
	     Vector3D(- 5 - 2.0 * stretchZ, 0.5 * szd, 0.0),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(- 5, 0.5 * szd, - 0.5),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5 - ( 2.0 * stretchZ ), 0.0),
	     Vector3D(szd, bw, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5 , - 0.5),
	     Vector3D(szd, bw, bw),
	     black);
  }
  else {
    writeBox(os, 
	     Vector3D(- 5 - LD20 * stretchZ, 0.5 * szd, 0.0),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(- 5, 0.5 * szd, - 0.5),
	     Vector3D(bw, szd, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5 - ( LD20 * stretchZ ), 0.0),
	     Vector3D(szd, bw, bw),
	     black);
    writeBox(os, 
	     Vector3D(0.5 * szd, - 5 , - 0.5),
	     Vector3D(szd, bw, bw),
	     black);

  }
}
