// -*- C++ -*------------------------------------------------------------------
//  $Id: Alignment.cc,v 1.16 2005/09/19 16:41:09 bindewae Exp $
//
//  Class:              Alignment
//
//  Description:
//    This class implements a simple alignment type.
//
// ---------------------------------------------------------------------------

// Includes:
#include <math.h>
#include <Alignment.h>
#include <IoTools.h>
#include <sstream>

// #define DEBUG_VERBOSE

// CONSTRUCTORS/DESTRUCTOR:
Alignment::Alignment() : targetMode(false), readScoreMode(false), 
			 outputLineLength(60),
  target(), targetName(), seqTemplate(), 
  seqTemplateName(), startAaTarget(0), score(),  evalue()
{
}

Alignment::Alignment(const Alignment& orig)
{
  this->copy(orig);
}

Alignment::~Alignment()
{
}

// PREDICATES:

bool
Alignment::isConserved(unsigned int p, unsigned int index) const
{
  if ((p > target.length()) || (index > this->size())) {
    ERROR("Argument out of scope.", exception);
  }
  if (index == 9999)
    {
      for (unsigned int i = 0; i < seqTemplate.size(); i++)
	if (target[p] != seqTemplate[i][p])
	  return false;
    }
  else
    if (target[p] != seqTemplate[index][p])
      return false;

  return true;
}

bool
Alignment::isInsertion(unsigned int p, unsigned int index) const
{
  if ((p > target.length()) || (index >= seqTemplate.size())) {
    ERROR("Argument out of scope.", exception);
  }
  if (isGap(p, index)) {
      return true;
  }
  return false;
}

bool
Alignment::isDeletion(unsigned int p) const
{
  if (p > target.length()) {
    ERROR("Argument out of scope.", exception);
  }
  if (isGapLetter(target[p]) ) {
    return true;
  }
  return false;
}

bool 
Alignment::isGap(unsigned int p, unsigned int index) const
{
   if ((p > target.length()) || (index >= seqTemplate.size()))
    ERROR("Argument out of scope.", exception);

   if (isGapLetter(target[p]) && isGapLetter(seqTemplate[index][p]) ) {
     return true;
   }
   
   return false;
}

/** returns true if column consists only of gap characters */
bool 
Alignment::isGapColumn(unsigned int p) const
{
  PRECOND((isValid()) && (p < seqTemplate[0].size()), exception);
  if (targetMode && (!isGapLetter(target[p]))) {
    return false;
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if (!isGapLetter(seqTemplate[i][p])) {
      return false;
    }
  }
  return true;
}


// count number of completetely conserved residues
double 
Alignment::calculateIdentity() const
{
  double tmp = 0;
  for (unsigned int i = 0; i < target.length(); i++)
    {
      char tmpC = target[i];
      bool add = true;
      for (unsigned int j = 0; j < seqTemplate.size(); j++)
	if (seqTemplate[j][i] != tmpC)
	  {
	    add = false;
	    break;
	  }
      if (add)
	tmp++;
    }

  return (tmp / target.length());
}

void
Alignment::sSaveFasta(string t, string tName, ostream& output) const
{
  //  output << "> " << tName << "\n"; 
  output << ">" << tName << "\n"; // taken out space after '>'
  for (unsigned int i = 0; i < t.length(); i++)
    {
      output << t[i];
      if (outputLineLength > 1) {
	if ((i+1) % outputLineLength == 0) {
	  output << "\n";
	}
      }
    }
  output << "\n";
}

void 
Alignment::saveFasta(ostream& output) const
{
  if (targetMode) {
    sSaveFasta(target, targetName, output);
  }
  for (unsigned int j = 0; j < seqTemplate.size(); j++)
    sSaveFasta(seqTemplate[j], seqTemplateName[j], output);
}

/** write single line for Clustal type output. */
void
sSaveClustal(string t, string tName, ostream& output)
{
  string tNameLocal;
  if (tName != "") {
    tNameLocal = tName;
  }
  else {
    tName = "?"; // give default dummy name
  }
  output << tName;
  for (int i = 0; i < 14-static_cast<int>(tName.length()); ++i) {
    output << " ";
  }
  output << t << endl;
}

/** write single line for Clustal type output. Used for 
    reading with Blast. 
    see: Readme file for blast (README.bls) */
void 
Alignment::saveClustal(ostream& output) const
{
  sSaveClustal(target, targetName, output);
  for (unsigned int j = 0; j < seqTemplate.size(); j++)
    sSaveClustal(seqTemplate[j], seqTemplateName[j], output);
}

/** write single line for Clustal type output. */
void
sSaveMSAF(string t, string tName, int offset, ostream& output)
{
  string tNameLocal;
  if (tName != "") {
    tNameLocal = tName;
  }
  else {
    tName = "?"; // give default dummy name
  }
  output << tName;
  output << " " << (offset +1);
  for (int i = 0; i < 14-static_cast<int>(tName.length()); ++i) {
    output << " ";
  }
  output << t << endl;
}


/** write single line for Clustal type output. Used for 
    reading with Blast. 
    see: Readme file for blast (README.bls) */
void 
Alignment::saveMSAF(ostream& output) const
{
  sSaveMSAF(target, targetName, startAaTarget, output);
  for (unsigned int j = 0; j < seqTemplate.size(); j++)
    sSaveMSAF(seqTemplate[j], seqTemplateName[j], 
		  startAaTemplates[j],output);
}

/** write single line for Clustal type output. Used for 
    reading with Blast. 
    see: Readme file for blast (README.bls) */
void 
Alignment::saveDSAF(ostream& os) const
{
  os << (this->size() + 1) << endl;
  os <<  target.size() << endl;
  os << target << endl;
  for (unsigned int j = 0; j < seqTemplate.size(); j++) {
    os << seqTemplate[j] <<  endl;
  }
}

/** return number of superposed amino acids for target 
    and template n
*/
unsigned int 
Alignment::numSuperposed(unsigned int n, unsigned int m) const
{
  if ((n >= seqTemplate.size())
      || (m >= seqTemplate.size())) {
    return 0;
  }
  unsigned int result = 0;
  unsigned int maxRes = min(seqTemplate[n].size(),
			    seqTemplate[m].size());
  for (unsigned int i = 0; i < maxRes; ++i) {
    if ((!isGapLetter(seqTemplate[m][i]) )
	&& (!isGapLetter(seqTemplate[n][i]) ) ) {
      ++result;
    }
  }
  POSTCOND((result <= seqTemplate[n].size())
        && (result <= seqTemplate[m].size()), exception);
  return result;
}

/** return number of superposed amino acids for target 
    and template n
*/
unsigned int 
Alignment::numSuperposed(unsigned int n) const
{
  if (n >= seqTemplate.size()) {
    return 0;
  }
  unsigned int result = 0;
  for (unsigned int i = 0; i < target.size(); ++i) {
    ASSERT(i < seqTemplate[n].size(), exception);
    if (i >= seqTemplate[n].size()) {
      break;
    }
    if ((!isGap(target[i])) && 
	(!isGapLetter(seqTemplate[n][i]) ) ) {
      ++result;
    }
  }
  return result;
}

bool
Alignment::isConsistent() const
{
  bool result = ( (evalue.size() == score.size())
		  && (evalue.size() == seqTemplate.size())
		  && (seqTemplateName.size() == seqTemplate.size() ) );
  if (result && (seqTemplate.size() > 0) ) {
    // check if all sequences have same length
    if ((targetMode) && (target.size() != seqTemplate[0].size()) ) {
      result = false;
      cout << "Target has wrong size! " << target.size() << " " 
	   << seqTemplate[0].size() << endl;
    }
    else {
      for (unsigned int i = 1; i < seqTemplate.size(); ++i) {
	if (seqTemplate[i - 1].size() != seqTemplate[i].size() ) {
	  result = false;
	  cout << "Template has wrong size! " << i - 1 << " " << i
	       << endl
	       << seqTemplate[i-1].size() << endl
	       << seqTemplate[i].size() << endl;
	  break;
	}
      }
    }
  }
  POSTCOND(((evalue.size() == score.size())
	   && (evalue.size() == seqTemplate.size())
	   && (seqTemplateName.size() == seqTemplate.size())) || (!result), exception);
  return result;
}

/** returns index of template with name containing s */
unsigned int
Alignment::findTemplateByName(const string& s) const
{
  for (unsigned int i = 0; i < seqTemplateName.size(); ++i) {
    if (seqTemplateName[i].find(s) == 0) {
      return i;
    }
  }
  return seqTemplateName.size();
}

/** returns true if all sequences of both alignments are equal */
bool
Alignment::isEqual(const Alignment& other) const
{
  if (size() != other.size()) {
    return false;
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if (seqTemplate[i].compare(other.seqTemplate[i]) != 0) {
      return false;
    }
  }
  return true;
}

/** saves as typical printable output */
void 
Alignment::savePrintable(ostream& os, unsigned int id, unsigned int pos, unsigned int len) const
{
  os << getTemplateName(id) << "\t";
  for (unsigned int i = pos; i < pos + len; ++i) {
    if (i >= seqTemplate[id].size()) {
      break;
    }
    if ((i > 0) && ((i % 10)==0)) {
      os << " ";
    }
    os << seqTemplate[id][i];
  }
}

/** saves as typical printable output */
void 
Alignment::savePrintable(ostream& os) const
{
  unsigned int pc = 0;
  const int OUT_LENGTH = 60;
  while (pc < getLength()) {
    os << pc + 1 << endl;
    for (unsigned int i = 0; i < size(); ++i) {
      savePrintable(os, i, pc, OUT_LENGTH);
      os << endl;
    }
    os << endl;
    pc += OUT_LENGTH;
  }
}

/** return "slice" of profile at position pos  */
Vec<char> 
Alignment::getSlice(unsigned int pos) const {
  Vec<char> result(size(),'-');
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if (pos < seqTemplate[i].size()) {
      result[i] = seqTemplate[i][pos];
    }
  }
  return result;
}

/** return "slice" of profile at position pos  */
string
Alignment::getSliceString(unsigned int pos) const {
  string result(size(),'-');
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if (pos < seqTemplate[i].size()) {
      result[i] = seqTemplate[i][pos];
    }
  }
  return result;
}

const string&
Alignment::getTemplate(unsigned int index) const
{
  if (index >= seqTemplate.size()) {
    ERROR("Alignment::getTemplate() : Invalid template requested.", 
	  exception);
  }
  return seqTemplate[index];
}

string 
Alignment::getTemplateName(unsigned int index) const
{
  if (index >= seqTemplateName.size()) {
    ERROR("Alignment::getTemplateName() : Invalid template requested.", 
	  exception);
  }
  return seqTemplateName[index];
}

double 
Alignment::getScore(unsigned int index) const
{
  if (index >= score.size()) {
    ERROR("Invalid template requested.", exception);
  }
  return score[index];
}

double 
Alignment::getEvalue(unsigned int index) const
{
  if (index >= evalue.size()) {
    ERROR("Invalid template requested.", exception);
  }
  return evalue[index];
}

char 
Alignment::getTargetPos(unsigned int p) const
{
  if (p >= target.length()) {
    ERROR("Alignment::getTargetPos() : Invalid position requested.", 
	  exception);
  }
  return target[p];
}

char 
Alignment::getTemplatePos(unsigned int p, unsigned int index) const
{
  if (index >= seqTemplate.size()) {
    ERROR("Alignment::getTemplatePos() : Invalid template requested.", 
	  expection);
  }
  if (p >= seqTemplate[index].length()) {
    ERROR("Alignment::getTemplatePos() : Invalid position requested.", 
	  exception);
  }
  return seqTemplate[index][p];
}

/** return offset of n'th template (counting from zero) */
int 
Alignment::getTemplateAminoAcidOffset(unsigned int index) const
{
  PRECOND(index < startAaTemplates.size(), exception);
  return startAaTemplates[index];
}

/* MODIFIERS */

void 
Alignment::clearAlignment()
{
  target = "";
  targetName = "";
  startAaTarget = 0;
  startAaTarget = 0;
  clearTemplate();
}

void 
Alignment::clearTemplate()
{
  seqTemplate.clear();
  seqTemplateName.clear();
  score.clear();
  evalue.clear();
  eccodes.clear();
  startAaTemplates.clear();
}

void
Alignment::addTemplate(const string& seq, const string& name) {
  seqTemplate.push_back(seq);
  seqTemplateName.push_back(name);
  eccodes.push_back(string(""));
  startAaTemplates.push_back(0);
  score.push_back(0.0);
  evalue.push_back(0.0);
  seqWeights.push_back(1.0);
}		   

/** transform character in sequences */
void
Alignment::replaceChar(char cOld, char cNew) {
  target = translate(target, cOld, cNew);
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    seqTemplate[i] = translate(seqTemplate[i], cOld, cNew);
    if (seqTemplate[i].find(cOld) < seqTemplate[i].size()) {
      cout << seqTemplate[i] << endl;
      ERROR("internal error in line 171!", exception);
    }
  }
}

/** transform sequences to upper case */
void
Alignment::upperCaseSequences() {
  upperCase(target);
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    upperCase(seqTemplate[i]);
  }
}


/** add gap characters at left and right ends */
void
Alignment::addAllGapEnds(unsigned int numLeft, unsigned int numRight) {
  if (targetMode) {
    target = addGapEnds(target, numLeft, numRight);
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    seqTemplate[i] = addGapEnds(seqTemplate[i], numLeft, numRight);
  }
}

void
Alignment::deleteSequence(unsigned int n)
{
  PRECOND(n < size(), exception);
  seqTemplate.erase(seqTemplate.begin() + n);
  seqTemplateName.erase(seqTemplateName.begin() + n);
  if (n < eccodes.size()) {
    eccodes.erase(eccodes.begin() + n);
  }
  if (n < startAaTemplates.size()) {
    startAaTemplates.erase(startAaTemplates.begin() + n);
  }
  if (n < score.size()) {
    score.erase(score.begin() + n);
  }
  if (n < evalue.size()) {
    evalue.erase(evalue.begin() + n);
  }
  if (n < seqWeights.size()) {
    seqWeights.erase(seqWeights.begin() + n);
  }
}

void
Alignment::setSlice(const string& s, unsigned int pos)
{
  PRECOND((s.size() == size()) && (pos < getLength()), exception);
  for (unsigned int i = 0; i < s.size(); ++i) {
    seqTemplate[i][pos] = s[i];
  }
}

void 
Alignment::setTarget(string t, string tName)
{
  if (seqTemplate.size() > 0)
    if (t.length() != seqTemplate[0].length()) {
      ERROR("Alignment::setTarget() : Target length does not match template.",
	    exception);
    }
  target = t;
  targetName = tName;
}

void 
Alignment::swapTemplate(unsigned int index1, unsigned int index2)
{
  if ((index1 >= this->size()) || (index2 >= this->size())) {
    ERROR("Alignment::swapTarget() : Index out of range.", exception);
  }
  swap(seqTemplate[index1], seqTemplate[index2]);
  swap(seqTemplateName[index1], seqTemplateName[index2]);
  swap(startAaTemplates[index1], startAaTemplates[index2]);
  swap(score[index1], score[index2]);
  swap(evalue[index1], evalue[index2]);
}

void 
Alignment::setTemplate(string t, string tName, double tScore, double tEvalue)
{
//    if (t.length() != target.length()) {
//      ERROR("Alignment::setTemplate() : Template length does not match target.",
//  	  exception);
//    }
  seqTemplate.push_back(t);
  seqTemplateName.push_back(tName);
  score.push_back(tScore);
  evalue.push_back(tEvalue);
  seqWeights.push_back(1.0);
}

void 
Alignment::setScore(double val, unsigned int index)
{
  if (index >= score.size()) {
    ERROR("Index out of scope.", exception);
  }
  score[index] = val;
}

void 
Alignment::setEvalue(long double val, unsigned int index)
{
  if (index >= evalue.size()) {
    ERROR("Index out of scope.", exception);
  }
  evalue[index] = val;
}

void 
Alignment::cutTemplate(unsigned int index)
{  // removes all templates below index
  if (index > seqTemplate.size()) {
    ERROR("Index out of scope.", exception);
  }
  // resize templates...
  seqTemplate.resize(index);
  seqTemplateName.resize(index);
  startAaTemplates.resize(index); 
  score.resize(index);
  evalue.resize(index);
  if (seqWeights.size() > 0) {
    seqWeights.resize(index);
  }

  // cut empty positions...
  /*
  unsigned int i = 0; 
  while (i < target.size())
    if (isGapLetter(target[i]) )
      {
	bool gap = true;
	for (unsigned int j = 0; j < seqTemplate.size(); j++)
	  gap = gap && (isGapLetter(seqTemplate[j][i]) );
	if (gap)
	  deletePos(i);
	else
	  i++;
      }
    else 
      i++;
  */
}

/** returns true if correct initialization character is given.
    fastaMode = true: fasta format, otherwise gde format 
    (output of clustalw with option -output=gde
*/
bool
Alignment::isFastaChar(char c, bool fastaMode) const
{
  if (fastaMode) {
    return (c == '>');
  }
  return (c == '%') || (c == '#');
}

void 
Alignment::loadFasta(istream& input)
{
  string tmp;
  bool fastaMode = true; // if false: assume gde format with "%" instead of ">"
  tmp = readLine(input);
  if (tmp[0] == '>') {
    fastaMode = true;
  } else if ((tmp[0] == '%') || (tmp[0] == '#')) {
    fastaMode = false; // assume gde format
  }
  else {
    cout << "Weird input line:" << endl << tmp << endl;
    ERROR("Unrecognized start character.", exception);
  }

  if (targetMode) {
    targetName = tmp.substr(1); // skip first '>' character
    
    //  cout << "loadFasta: found target name " << tmp.substr(1) << endl;
    
    target = "";
    
    while (input) {
      tmp = getLine(input);
      if (!isFastaChar(tmp[0], fastaMode)) {
	target += tmp;
      }
      else { 
	break;
      }
    }
  }

  unsigned int count = 0;

  if (!input) {
    ERROR("Abnormal input file end.", exception);
  }

  while (input)
    {
      if (isFastaChar(tmp[0], fastaMode)) {
	seqTemplateName.push_back(tmp.substr(1));
	// cout << "loadFasta: found template name " << tmp.substr(1) << endl;
      }
      else {
	seqTemplateName.push_back(tmp);
      }
      seqTemplate.push_back("");
      score.push_back(0);
      evalue.push_back(-1);
      seqWeights.push_back(1.0);
      startAaTemplates.push_back(0);
      eccodes.push_back(string(""));
      while (input) {
	  tmp = readLine(input);
	  tmp = removeWhiteSpaceFromString(tmp);
	  if (tmp.size() == 0) {
	    continue;
	  }
	  // cout << "read line: " << tmp << endl;
	  if (!input) {  // Warning: This 'if' serves to remove a bug
	    // which causes the last line in a file to be read twice.
            if ((!isFastaChar(tmp[0], fastaMode))
                && (seqTemplate[count].size()!=getLength())) {
              seqTemplate[count] += tmp;
            }
	    break;  
	  }
	  if (!isFastaChar(tmp[0], fastaMode)) {
	    seqTemplate[count] += tmp;
	  }
	  else {
	    break;
	  }
      }
      count++;
      // delete spaces
//        seqTemplate[count] = removeFromString( 
//  		   seqTemplate[count] ,' ');
    }
  // cout << "load fasta finished!" << endl;
}

void 
Alignment::loadMase(istream& input)
{
  char delim = ';';
  string tmp;
  string lastLine;
  // bool fastaMode = true; // if false: assume gde format with "%" instead of ">"
  while (input && (tmp.size() == 0) ) {
    lastLine = tmp;
    tmp = readLine(input);
    // cout << "reading line: " << tmp << endl;
  }
//   if (tmp[0] == ';') {
//     fastaMode = true;
//   } else if ((tmp[0] == '%') || (tmp[0] == '#')) {
//     fastaMode = false; // assume gde format
//   }
//   else {
//     cout << "Weird input line:" << endl << tmp << endl;
//     ERROR("Unrecognized start character.", exception);
//   }

//  skip ";" characters
  while (input && ((tmp.size() == 0) || (tmp[0] == delim))) {
    lastLine = tmp;
    tmp = readLine(input);
    // cout << "reading line: " << tmp << endl;
  }
  if (targetMode) {
    targetName = tmp.substr(1); // skip first '>' character    
    //  cout << "loadFasta: found target name " << tmp.substr(1) << endl;
    target = "";
    while (input) {
      tmp = getLine(input);
      if (tmp[0] != delim) {
	target += tmp;
      }
      else { 
	break;
      }
    }
  }

  unsigned int count = 0;

  if (!input) {
    ERROR("Abnormal input file end.", exception);
  }

  while (input && ((tmp.size() == 0) || (tmp[0] == delim))) {
    lastLine = tmp;
    tmp = readLine(input);
    // cout << "reading line: " << tmp << endl;
  }

  while (input)
    {
      if (lastLine[0] == delim) {
	seqTemplateName.push_back(tmp);
	// cout << "loadFasta: found template name " << tmp.substr(1) << endl;
      }
      else {
	ERROR("Header line expected!", exception);
	// seqTemplateName.push_back(tmp);
      }
      seqTemplate.push_back("");
      score.push_back(0);
      evalue.push_back(-1);
      seqWeights.push_back(1.0);
      startAaTemplates.push_back(0);
      eccodes.push_back(string(""));
      while (input) {
	  tmp = readLine(input);
	  // cout << "reading line: " << tmp << endl;
	  // cout << "read line: " << tmp << endl;
	  if (!input) {  // Warning: This 'if' serves to remove a bug
	    break;    // which causes the last line in a file to be read twice.
	  }
	  if (tmp[0] != delim) {
	    seqTemplate[count] += tmp;
	  }
	  else {
	    lastLine = tmp;
	    tmp = readLine(input); // skip one line
	    // cout << "skipping line: " << tmp << endl;
	    break;
	  }
      }
      count++;
      // delete spaces
//        seqTemplate[count] = removeFromString( 
//  		   seqTemplate[count] ,' ');
    }
  // cout << "load fasta finished!" << endl;
}


/* read Jeffrey Skolnicks personal format */
void 
Alignment::loadDanforth(istream& input)
{
  clearAlignment(); // delete all info

  unsigned int nseq, nres;

  input >> nseq >> nres;
  for (unsigned int i = 0; i < nseq; ++i) {
    string seq = "";
    while (seq.size() < nres) {
      string line = getLine(input);
      line = removeFromString(line, ' ');
      line = removeFromString(line, '\t');
      line = removeFromString(line, '\n');
      line = removeFromString(line, '\r');
      line = removeFromString(line, '*');
      seq = seq + line;
    }
    if ((i > 0) || (!targetMode)) {
      seqTemplate.push_back(seq);
      seqTemplateName.push_back(string("template")+uitos(i));
      score.push_back(0);
      evalue.push_back(-1);
      seqWeights.push_back(1.0);
      startAaTemplates.push_back(0);
      eccodes.push_back(string(""));
    }
    else {
      targetName = "target";
      target = seq;
    }
  }

}



/** 
    read output "header" of CE structure alignment program
    author: Eckart Bindewald 03/2001
*/
void 
Alignment::loadCEHeader(istream& input)
{
  string tmp;
  Vec<string> words;
  tmp = readLine(input);

  targetName = tmp;
  target = "";

  do // read header
    {
      tmp = readLine(input);
      words = getTokens(tmp);
      if ((words.size() > 0) && (words[0] == "Alignment") )  {
	break; // end of "header info"
      }
      if ((words.size() > 2) && (words[0] == "Chain"))
	if (words[1] == "1:") {
	  targetName = words[2];
	}
	else if (words[1] == "2:") {
	  seqTemplateName.push_back(words[2]); // name of template
	}
    }
  while (input); 
  if (!input) {
    ERROR("Abnormal input file end.", exception);
  }
}

/** 
    read output "body" of CE structure alignment program
    author: Eckart Bindewald 03/2001
*/
void 
Alignment::loadCEBody(istream& input)
{
  PRECOND(input, exception);
  string tmp;
  Vec<string> words;
  if (targetName == "") {
    targetName = "?";
  }
  target = "";

  seqTemplate.push_back(""); // start with empty template sequence
  startAaTemplates.push_back(0); // yet undefined
  score.push_back(0);
  evalue.push_back(-1);
  seqWeights.push_back(1.0);
  if (seqTemplateName.size() < seqTemplate.size()) {
    seqTemplateName.push_back("?");
  }
  bool startAaTargetRead = false;
  bool startAaTemplateRead = false;
  int counter = 1; 
  do // read body
    {
      ++counter;
      tmp = readLine(input);
      words = getTokens(tmp);
      if (words.size() > 0) {
	if ( ((words[0] == "Starting") 
	     || (words[0] == "Alignment") )
	     && (target.size() > 0) ) {
	  break; // only quit loop if some chain has been read
	}
      }
      if ((words.size() > 3) && (words[0] == "Chain")) {
	if  (words[1] == "1:") {
	  if (!startAaTargetRead) {
	    startAaTarget = stoi(words[2]) - 1; // read offset
	    startAaTargetRead = true;
	  }
	  target = target + words[3]; // add sequence
	}
	else if  (words[1] == "2:") {
	  if (!startAaTemplateRead) {
	    startAaTemplates[0] = stoi(words[2]) - 1; // read offset
	    startAaTemplateRead = true;
	  }
	  seqTemplate[0] = seqTemplate[0] + words[3]; // add sequence
	}
      }
      if ((words.size() > 0) && (words[0] == "X2")) {
	break; // end of body. start of transformation data
      }
    }
  while (input);
  /* here should be the code for parsing the transformation data... :-) */
     

}

string
Alignment::fixedSequence(const string& ref, const string& cmp) 
{
  if (cmp.size() == ref.size()) {
    return cmp; // nothing to do
  }
  else if (cmp.size() > ref.size()) {
    return cmp.substr(0, ref.size());
  }
  // must be smaller
  unsigned int delta = ref.size() - cmp.size();
  string added;
  for (unsigned int i = 0; i < delta; ++i) {
    added += "-";
  }
  string result = cmp + added;
  POSTCOND(result.size() == ref.size(), exception);
  POSTCOND((evalue.size() == score.size())
	   && (evalue.size() == seqTemplate.size())
	   && (seqTemplateName.size() == seqTemplate.size()), exception);
  return result;
}


/* fill up template sequences with "-" until size equal target sequence length */
void
Alignment::fixAlignmentBody() 
{
  if (seqTemplate.size() == 0) {
    return; // nothing to do
  }
  if (target.size() > 0) {
    target = "-";
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    seqTemplate[i] = string("-"); // fixedSequence(refSeq, seqTemplate[i]);
  }
  POSTCOND(isConsistent(), exception);
}

/** 
    read output of CE structure alignment program
    author: Eckart Bindewald 03/2001
*/
void 
Alignment::loadCE(istream& input)
{
  loadCEHeader(input);
  loadCEBody(input);

}


/** read BLAST output produced with blast option -m 6 
    this stands for a convinient form of multiple sequence alignment 
*/
void 
Alignment::loadBlastMode6(istream& input)
{
  // clear existing info
  targetName = "";
  target = "";
  unsigned int templateCount = 0;
  seqTemplate.clear(); 
  seqTemplateName.clear();
  int state = 1; // state of parser
  while (input) {
    string line = readLine(input);
    Vec<string> words = getTokens(line);
    if (words.size() == 0) {
      continue; // skip empty lines
    }
//     else {
//       cout << "Parsing line: " << line << ":: Words: " << words << endl;
//     }
    if (words[0] == "Sequences")
      {
	state = 0;
	continue;
      }

    switch (state) {
    case 0: // read scores & E-values:
      if ((words.size() >= 2 ) && (words[0].find("QUERY") != 0)
	  && (words[0].find("Query") != 0) && (words[0][0] != '>')
	  && ( !isspace(line[0]) ) )
	// 	if (words[0].find("|") <= 10)
	{
	  score.push_back(stod(words[words.size()-2]));
	  // cout << "pushing score " << words[words.size()-2] << endl;
	  if (words[words.size()-1][0] == 'e') {
	    evalue.push_back(stod('1'+words[words.size()-1]));
	    seqWeights.push_back(1.0);
	  }
	  else {
	    evalue.push_back(stod(words[words.size()-1]));
	    seqWeights.push_back(1.0);
	  }
	  // cout << "pushed e-value: " << evalue[evalue.size() - 1] << endl;
	  if (readScoreMode) {
	    seqTemplateName.push_back(words[0]); // new inserted! might mess up previous version :-(
	    seqTemplate.push_back(string("-"));
	    // cout << "pushing template name and sequence: " << words[0] << " - " << endl;
	    ASSERT((evalue.size() == score.size())
		   && (evalue.size() == seqTemplate.size())
		   && (seqTemplateName.size() == seqTemplate.size()), exception);
	  }
	}
    case 1: // looking for start of alignment
      if (words[0] == "Query=") { // name of query found
	if (words.size() > 1) {
	  targetName = words[1]; // set proper name
	  // cout << "found target name: " << targetName << endl;
	} // do not change state of parser
      }
      else if (words[0] == "QUERY") {
	state = 2;
	templateCount = 0;
	if (targetName == "") {
	  targetName = "QUERY";
	}
	if (words.size() == 4) {
	  target = words[2];
	  startAaTarget = stoi(words[1]) - 1; // read offset
	}
	else if (words.size() == 2) {
	  target = words[1]; // line consists only of "-" characters
	  startAaTarget = 0; // no offset
	}
	else {
	  cout << "weird line: " << endl << line << endl;
	  ERROR("Strange line encountered.", exception);
	}
      }
      break;
    case 2:  // reading body of alignment
      if (readScoreMode) {
	fixAlignmentBody();
	ASSERT(isConsistent(), exception());
	return; // do not read body!
      }
      if (words[0] == "QUERY") {
	state = 2;
	templateCount = 0; 
	if (words.size() == 4) {
	  target += words[2];
	}
	else if (words.size() == 2) {
	  target += words[1]; // line consists only of "-" characters
	}
	else {
	  cout << "weird line: " << endl << line << endl;
	  ERROR("Strange line encountered.", exception);
	}
      }
      else if (words[0].find("Searching..") == 0) {
	// must be psi-blast! This was not the last psi-blast round. 
	// discard the stuff read so far and continue searching
	state = 1; 
	templateCount = 0;
	target = "";
	seqTemplate.clear(); // clear existing info
	seqTemplateName.clear();
	score.clear();
	evalue.clear();
	seqWeights.clear();
      }
      else if (words[0] == "Database:") {
	return;  // done!
      }   // reading next template:
      else if ((words.size() <= 4) && (words.size() >= 2)) {
	string seq; 
	// size == 3 is a strange case: 
	// happens (I think) only, if template sequence is like :   
	// identifier 0 ----------------------------
	if ((words.size() == 4) || (words.size() == 3)) {
	  seq = words[2];
	}
	else {
	  seq = words[1];
	}
	if (templateCount < seqTemplate.size() ) {
	  seqTemplate[templateCount] += seq;
	}
	else {
	  seqTemplate.push_back(seq);
	  seqTemplateName.push_back(words[0]);
	  score.push_back(0);
	  evalue.push_back(-1);
	  seqWeights.push_back(1.0);
	  if ((words.size() == 4) || (words.size() == 3)) {
	    startAaTemplates.push_back(stoi(words[1]) -1);
	  }
	  else { // no offset
	    startAaTemplates.push_back(0);
	  }
	}
	++templateCount;
      }
      else {
	cout << "weird line: " << endl << line << endl;
	ERROR("Strange line encountered.", exception);
      }
      break;
    default: ERROR("Internal error line 442.", exception);
    }
  } 
}

/** write single line for Clustal type output. Used for 
    reading with Blast. 
    see: Readme file for blast (README.bls) */
void 
Alignment::loadMSAF(istream& is)
{
  seqTemplate.clear();
  seqTemplateName.clear();
  startAaTemplates.clear();
  int startTemp = 0;
  Vec<string> words = getTokens(readLine(is));
  if (words.size() != 3) {
    ERROR("Strange first line in loadMSAF.", exception);
  }
  targetName = words[0];
  startAaTarget = stoi(words[1]) - 1;
  target = words[2];
  while (is) {
    words = getTokens(readLine(is));
    if (words.size() != 3) {
      return; 
    }
    string tmpName, tmpSeq;
    is >> tmpName >> startTemp >> tmpSeq;
    seqTemplateName.push_back(tmpName);
    seqTemplate.push_back(tmpSeq);
    score.push_back(0);
    evalue.push_back(-1);
    seqWeights.push_back(1.0);
    startAaTemplates.push_back(startTemp);
  }
}


void 
Alignment::copy(const Alignment& orig)
{
  targetMode = orig.targetMode;
  readScoreMode = orig.readScoreMode;
  outputLineLength = orig.outputLineLength;
  target = orig.target;
  targetName = orig.targetName;
  startAaTarget = orig.startAaTarget;
  startAaTemplates = startAaTemplates;
  seqTemplate = orig.seqTemplate;
  seqTemplateName = orig.seqTemplateName;
  seqWeights = orig.seqWeights;
  score = orig.score; 
  evalue = orig.evalue; 
}


/** insert "-" in target and all templates at position p 
    (also counting the other "-" characters as position) 
    Eckart Bindewald 03/2001 
*/
void 
Alignment::insertCharacter(unsigned int p, char c){
  PRECOND(isConsistent(), exception);
  if (p <= target.size()) {
    target.insert(p,1,c);
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if (p <= seqTemplate[i].size()) {
      seqTemplate[i].insert(p,1,c);
    }
  }
  POSTCOND(isConsistent(), exception);
}

/** 
    insert "-" in target and all templates at position p (also counting the other "-" characters as position) 
    Eckart Bindewald 03/2001 
*/
void 
Alignment::insertDash(unsigned int p){
  PRECOND(isConsistent(), exception);
  insertCharacter(p, GAP_CHAR);
  POSTCOND(isConsistent(), exception);
}

/** delete character of position p (or '-') from target and all templates */
void 
Alignment::deletePos(unsigned int p)
{
  if ((target.size() > 0) && (p < target.size() ) ) {
    target = deleteChar(target,p);
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if (p < seqTemplate[i].size()) {
      seqTemplate[i] = deleteChar(seqTemplate[i], p);
    }
  }
}

/** delete character of position p (or '-') from target and all templates */
void 
Alignment::purgeTargetInsertions()
{
  unsigned int i = 0; 
  string useTarget = target;
  // if no explicit target given, use topmost template sequence
  if ((useTarget.size() == 0) && (seqTemplate.size() == 0)) {
    return;
  }
  if (useTarget.size() == 0) {
    useTarget = seqTemplate[0];
  }
  while (i < useTarget.size()) {
    if (isGapLetter(useTarget[i])) {
      // || (useTarget[i] == 'X')) { // BUGFIX ???? November 2002
      deletePos(i);
      i = 0;
    }
    else {
      ++i;
    }
    if (target.size() == 0) {
      useTarget = seqTemplate[0];
    }
    else {
      useTarget = target;
    }
  }
  // Postcondition: no more dashes can be found
  //   POSTCOND(target.find('-') == target.size(), exception);
}

/** delete character of position p (or '-') from target and all templates */
void 
Alignment::purgeTargetInsertions(unsigned int n)
{
  string useTarget = target;
  // if no explicit target given, use topmost template sequence
  int i = seqTemplate[n].size() - 1; 
  while (i >= 0) {
    if (isGapLetter(seqTemplate[n][i])) {
      // || (useTarget[i] == 'X')) { // BUGFIX ???? November 2002
      deletePos(i);
    }
    --i;
  }
  // Postcondition: no more dashes can be found
  //   POSTCOND(target.find('-') == target.size(), exception);
}


/** delete character of position p (or '-') from target and all templates */
void 
Alignment::purgeGapColumns()
{
  PRECOND(isConsistent(), exception);
  if (!isValid()) {
    return;
  }
  int len = seqTemplate[0].size();
  for (int i = len-1; i >= 0; --i) {
    if (isGapColumn(i)) {
      deletePos(i);
    }
  }
  POSTCOND(isConsistent(), exception);
}


/** combine two multiple sequence alignments of same target
    The algorithm is so long, because 6 cases are distinguished:
    leading insertion for target A or target B
    normal  insertion for target A or target B
    trailing insertion for target A or target B
    if ignoreInsertion is set, insertions of target A are ignored
 */
void 
Alignment::addAlignment(const Alignment& orig)
{
  // sequence without dashes
  unsigned int i = 0; 
  unsigned int origPos = 0;
  unsigned int newTemplatePos = 0;
  unsigned int leftGapCount1 = 0;
  unsigned int leftGapCount2 = 0;
  unsigned int rightGapCount1 = 0;
  unsigned int rightGapCount2 = 0;
  unsigned int leftGapCount2Orig = leftGapCount2;
  unsigned int rightGapCount2Orig = rightGapCount2;
  int diff = 0; 
  Alignment  other= orig; // make safety copy
  string pure1 = getPureSequence(target); // sequence without "-"
  string pure2 = getPureSequence(other.target); // sequence without "-"
  if (target == other.target) {
    goto COMBINE_END; // nothing to do if sequences have same insertions.
  }
  // prepare sequence for adding: provisorically 
  // add 'Z' for undefined amino acids:
  if (orig.startAaTarget > startAaTarget) {
    for (int i = 0; i < (orig.startAaTarget - startAaTarget); ++i) {
      other.insertCharacter(0,'~');
    }
  }
  else if (orig.startAaTarget < startAaTarget) {
    for (int i = 0; i < (orig.startAaTarget - startAaTarget); ++i) {
      insertCharacter(0,'~');
    }
    startAaTarget = orig.startAaTarget;
  }
  // count still missing letters and add at end:
  // getPureSequence also counts '-'!
  diff = (static_cast<int>(getPureSequence(target).size())
	  - static_cast<int>(getPureSequence(other.target).size()) );
  if (diff < 0) {  // still not same size: add "X" at end
    for (int i = 0; i < abs(diff); ++i) {
      insertCharacter(target.size(),'~');
    }
  }
  else if (diff > 0) { // still not same size: add "X" at end
    for (int i = 0; i < abs(diff); ++i) {
      other.insertCharacter(other.target.size(),'~');
    }
  }
#ifdef DEBUG_VERBOSE
  cout << "After preprocessing: " << endl;
  cout << target << endl << other.target << endl; 
#endif
  //   ASSERT(getPureSequence(target).size() 
  // == getPureSequence(other.target).size(), exception);
  // count leading gaps
  while ((leftGapCount1 < target.size() ) 
	 && isGapLetter(target[leftGapCount1]) ) {
    ++leftGapCount1;
  }
  while ((leftGapCount2 < other.target.size() ) 
	 && isGapLetter(other.target[leftGapCount2]) ) {
    ++leftGapCount2;
  }
  leftGapCount2Orig = leftGapCount2;
#ifdef DEBUG_VERBOSE
  cout << "Initial gap 1: " << leftGapCount1 << endl;
  cout << "Initial gap 2: " << leftGapCount2 << endl;
#endif
  // count trailing gaps
  while ((rightGapCount1 < target.size() ) 
	 && isGapLetter(target[target.size()-rightGapCount1-1]) ) {
    ++rightGapCount1;
  }
  while ((rightGapCount2 < other.target.size() ) 
     && isGapLetter(other.target[other.target.size() - rightGapCount2 - 1]) ) {
    ++rightGapCount2;
  }
  rightGapCount2Orig = rightGapCount2;
#ifdef DEBUG_VERBOSE
  cout << "Trailing gap 1: " << rightGapCount1 << endl;
  cout << "Trailing gap 2: " << rightGapCount2 << endl;
#endif
  if (leftGapCount1 > leftGapCount2) {
    for (unsigned int k = 0; k < (leftGapCount1 - leftGapCount2); ++k) {
#ifdef DEBUG_VERBOSE
      cout << "Inserting gap into pos 0 of target 2: " << endl;
#endif
      other.insertDash(0);
    }
    leftGapCount2 = leftGapCount1;
  }
  else if (leftGapCount1 < leftGapCount2) {
    for (unsigned int k = 0; k < (leftGapCount2 - leftGapCount1); ++k) {
#ifdef DEBUG_VERBOSE
      cout << "Inserting gap into pos 0 of target 1: " << endl;
#endif
      insertDash(0);
    }
    leftGapCount1 = leftGapCount2;
  }
#ifdef DEBUG_VERBOSE
  cout << target << endl << other.target << endl;
#endif
  if (rightGapCount1 > rightGapCount2) {
    for (unsigned int k = 0; k < (rightGapCount1 - rightGapCount2); ++k) {
#ifdef DEBUG_VERBOSE
      cout << "Inserting gap into pos " 
	   << target.size() << " of target 2: " << endl;
#endif
      other.insertDash(other.target.size());
    }
    rightGapCount2 = rightGapCount1;
  }
  else if (rightGapCount1 < rightGapCount2) {
    for (unsigned int k = 0; k < (rightGapCount2 - rightGapCount1); ++k) {
#ifdef DEBUG_VERBOSE
      cout << "Inserting gap into pos "
	   << (target.size()) << " of target 1: " << endl;
#endif
      insertDash(target.size());
    }
    rightGapCount1 = rightGapCount2;
  }
#ifdef DEBUG_VERBOSE
  cout << target << endl << other.target << endl;
#endif
  i = leftGapCount1; // now skip leading gaps
  while (i < (target.size()-rightGapCount1)) {
    if (isGapLetter(target[i]) ) {
      origPos = getOrigPos(target, i); // position of first non-dash char
      // compute position on other sequence:
      newTemplatePos = getNewPos(other.target,origPos);
      // count length of insertion:
      unsigned int l = i + 1; 
      for (l = i + 1; (l < target.size())&& isGapLetter(target[l]); ++l) {
      }
      unsigned int l2 = l - i; // length of insertion
      unsigned int l3 = newTemplatePos; 
#ifdef DEBUG_VERBOSE
      cout << "Insertion of length " << l2 << " at position " 
	   << i << " found in target 1 " << endl;
      cout << "Corresponding position on target 2: " << l3 << endl;
#endif
      for (l3 = newTemplatePos; 
        (l3 < other.target.size())&& isGapLetter(other.target[l3]) ; ++l) {
      }
      // length of insertion of template:
      unsigned int l4 = l3 - newTemplatePos; 
#ifdef DEBUG_VERBOSE
      cout << "Found insertion of length " << l4 << " at position " 
	   << newTemplatePos << " of target 2" << endl;
#endif
      // insert this many dashes:
      if (l4 < l2) {
	unsigned int dl = l2 -l4; 
	// insertion of other alignement is "dl" residues too short. Insert:
	for (unsigned int j = 0; j < dl; ++j) {
#ifdef DEBUG_VERBOSE
	  cout << " inserting dash after position " << newTemplatePos +1
	       << " on target 2 " << endl;
#endif
	  other.insertDash(newTemplatePos+1);
	}
	i += dl;
      }
      else {
	++i;
      }
    }
    else {
      ++i;
    }
  }
  i = leftGapCount2Orig; // start again from zero
#ifdef DEBUG_VERBOSE
  cout << "Other direction: " << endl;
#endif
  // go through all "-" regions in target 2
  while (i < (orig.target.size()- rightGapCount2Orig)) {
    if (isGapLetter(orig.target[i]) ) {
#ifdef DEBUG_VERBOSE
      cout << "character " << i << " of target 2" << endl;
      cout << " dash found. " << endl;
#endif
      origPos = getOrigPos(orig.target, i);
      // compute position on other sequence:
      newTemplatePos = getNewPos(target,origPos);
      // count length of insertion:
      unsigned int l = i +1; 
      for (l = i + 1; (l < orig.target.size())
	     && isGapLetter(orig.target[l]); ++l) {
      }
      unsigned int l2 = l - i; // length of insertion
      // count length of insertion at other alignment:
      unsigned int l3 = newTemplatePos; 
      for (l3 = newTemplatePos; 
	   (l3 < target.size())&& isGapLetter(target[l3]); ++l) {
      }
      // length of insertion of template:
      unsigned int l4 = l3 - newTemplatePos;
#ifdef DEBUG_VERBOSE
      cout << "Found insertion of length " << l2 << " at position " 
	   << i << " of target 2" << endl;
      cout << "Found corresponding insertion of length " 
	   << l4 << " at position " 
	   << newTemplatePos << " of target 1" << endl;
#endif
      // insert this many dashes:
      if (l4 < l2) {
	unsigned int dl = l2 -l4; 
	// insertion of other alignement is "dl" residues too short. Insert:
	for (unsigned int j = 0; j < dl; ++j) {
#ifdef DEBUG_VERBOSE
	  cout << "Inserting dash into template 1 at position "
	       << (newTemplatePos+1) << endl;
#endif
	  insertDash(newTemplatePos+1);
	}
	i += dl;
      }
      else {
	++i;
      }
    } 
    else {
      ++i;
    }
  }
 COMBINE_END:
#ifdef DEBUG_VERBOSE
  cout << "Result: " << endl << target << endl << other.target << endl;
#endif
  ASSERT(target.size() == other.target.size(), exception);
  // append other templates:
  for (unsigned int i = 0; i < other.seqTemplate.size(); ++i) {
    seqTemplate.push_back(other.seqTemplate[i]);
    seqTemplateName.push_back(other.seqTemplateName[i]);
    score.push_back(other.score[i]);
    evalue.push_back(other.evalue[i]);
    if (i < other.seqWeights.size()) {
      seqWeights.push_back(other.seqWeights[i]);
    }
  }
  // do clean up : replace "~" with "-":
  for (unsigned int i = 0; i < target.size(); ++i) {
    if (target[i] == '~') {
      target[i] = '-';
    }
  }
  for (unsigned int j = 0; j < seqTemplate.size(); ++j) {
    for (unsigned int i = 0; i < seqTemplate[j].size(); ++i) {
      if (seqTemplate[j][i] == '~') {
	seqTemplate[j][i] = '-';
      }
    }
  }
}



// OPERATORS:

// STATIC
/* 
   return sequence without "-" characters
   Eckart Bindewald 03/2001
*/
string 
Alignment::getPureSequence(const string& s)
{
  string result = "";
  for (unsigned int i = 0; i < s.size(); ++i) {
    if (!isGapLetter(s[i]) ) {
      result.append(1,s[i]);
    }
  }
  return result;
}

/* return position of index if "-" would not be there.
Counting starts from zero.
If original index points to a dash, it returns the position 
of the first non-dash left character, again taking "-" not into account. */
unsigned int
Alignment::getOrigPos(const string& s, unsigned int p)
{
  PRECOND(p < s.size(), exception);
  int i;
  // move pointer to first "non-dash" character
  for (i = static_cast<int>(p); i >= 0; --i) {
    if (isGapLetter(s[i])) {
      break;
    }
  }
  if ( i < 0 ) {
    ERROR("No defined original position.", exception);
    return 0; // left bound dashes
  }
  unsigned int counter = 0;
  // count the dashes before this position:
  for (int j = 0; j < i; ++j) {
    if (isGapLetter(s[j]) ) {
      ++counter;
    }
  }
  return i - counter; // position minus number of dashes counted
}

/* return position of index (not counting "-") if "-" are now present.
Counting starts from zero.
Eckart Bindewald 03/2001
 */
unsigned int
Alignment::getNewPos(const string& s, unsigned int p)
{
  PRECOND(p < s.size(), exception);
  int count = -1; // count non-dash characters
  int pi = static_cast<int>(p);
  for (unsigned int i = 0; i < s.size(); ++i) {
    if (!isGapLetter(s[i]) ) {
      ++count;
      if (count == pi) {
	return i;
      }
    }
  }
  ERROR("Could not assign new index for sequence with insertions.",
	exception);
  return s.size(); // dummy
}

// tokenize text
Vec<string> 
Alignment::getTokens(const string& text) 
{
  Vec<string> v;
  if (text.size() == 0) {
    return v;
  }
  // istrstream ist(text.c_str());
  istringstream ist(text.c_str());
  char* charLine = new char[text.size()+1]; // size of string

  string s;
  while (!ist.eof()) {
    ist >> charLine;
    s = charLine; // assignment of c-strings to string!
    //    DUMP(s);
    if (s != "") { 
      v.push_back(s);
    }
  }
  delete[] charLine;
  return v;
}

/** delete n'th character. I know this can be solved much more
    elegantly using the Stl, but I do not have a book here... */
string 
Alignment::deleteChar(const string& s, unsigned int n)
{
  PRECOND(n < s.size(), exception);
  string result;
  result = s.substr(0,n); 
  if (n < (s.size() -1)) {
    result = result + s.substr(n+1,s.size());
  }
//    cout << "deleting position " << n << " from " << s << " results in: " 
//  	<< endl	<< result << endl;
  POSTCOND(result.size() == (s.size() - 1) , exception);
  return result;
}

/** for all sequences, keep only portion starting at minPos with length len */
void
Alignment::prune(unsigned int minPos, unsigned int len)
{
  if (targetMode) {
    if ((minPos + len) <= target.size()) {
      target = target.substr(minPos, len);
    }
    else if (minPos < target.size()) {
      target = target.substr(minPos);
    }
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if ((minPos + len) <= seqTemplate[i].size()) {
      seqTemplate[i] = seqTemplate[i].substr(minPos, len);
    }
    else if (minPos < seqTemplate[i].size()) {
      seqTemplate[i] = seqTemplate[i].substr(minPos);
    }
  }
}

/** for all sequences, erase portion starting at minPos with length len */
void
Alignment::cleave(unsigned int minPos, unsigned int len)
{
  if (targetMode) {
    if ((minPos + len) <= target.size()) {
      target = target.erase(minPos, len);
    }
  }
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    if ((minPos + len) <= seqTemplate[i].size()) {
      seqTemplate[i] = seqTemplate[i].erase(minPos, len);
    }
  }
}

/** removes all info about n'th template */
void 
Alignment::clearTemplate(unsigned int n)
{
  PRECOND(n < seqTemplate.size(), exception);
  seqTemplate.erase(seqTemplate.begin()+n);
  seqTemplateName.erase(seqTemplateName.begin()+n);
  score.erase(score.begin()+n);
  evalue.erase(evalue.begin()+n);
  eccodes.erase(eccodes.begin()+n);
  startAaTemplates.erase(startAaTemplates.begin()+n);
  if (n < seqWeights.size()) {
    seqWeights.erase(seqWeights.begin()+n);
  }
}

/** counts number of characters and gaps */
void
Alignment::countCharacters(unsigned int& numChars, unsigned int& numGaps) const
{
  numChars = 0;
  numGaps = 0;
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    for (unsigned int j = 0; j < seqTemplate[i].size(); ++j) {
      if (seqTemplate[i][j] == GAP_CHAR) {
	++numGaps;
      }
      else {
	++numChars;
      }
    }
  }
}

/** counts number of occurence of certain character */
int
Alignment::countCharacter(char c) const 
{
  int counter = 0;
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    for (unsigned int j = 0; j < seqTemplate[i].size(); ++j) {
      if (seqTemplate[i][j] == c) {
	++counter;
      }
    }
  }
  return counter;
}

  /** delete gaps in n'th sequence. Careful: alignment might be inconsistent after this! */
void
Alignment::deleteGaps(unsigned int n)
{
  PRECOND(n < seqTemplate.size(), exception);
  seqTemplate[n] = removeFromString(seqTemplate[n], GAP_CHAR);
}

/** delete gaps in all sequences. Careful: alignment might be inconsistent after this! */
void
Alignment::deleteGaps()
{
  for (unsigned int i = 0; i < seqTemplate.size(); ++i) {
    deleteGaps(i);
  }
}




