#include <HashCorrelationFinderTest.h>
#include <NucleotideTools.h>
#include <Timer.h>
#include <CorrelationTools.h>

void
HashCorrelationFinderTest::testFilterIsolatedCorrelations() {
  string methodName = "testFilterIsolatedCorrelations";
  cout << "Starting " << methodName << endl;
  HashCorrelationFinder::length_type corrDistMin = 5;
  Vec<Correlation> correlations;
  correlations.push_back(Correlation(4,20));
  correlations.push_back(Correlation(5,19));
  correlations.push_back(Correlation(7,21));
  Vec<Correlation> result = CorrelationTools::filterIsolatedCorrelations2(correlations,corrDistMin); // ok, method has been move to new class
  cout << correlations << endl;
  cout << result << endl;
  ASSERT(result.size() == 2);
  cout << "Finished " << methodName << endl;
}


void
HashCorrelationFinderTest::testSearchMafColumn() {
  string methodName = "testSearchMafColumn";
  cout << "Starting " << methodName << endl;  
  bool allowGu = true;
  bool allowGap = false;
  string filename = "../../test/fixtures/anticodon_simple2.fa";
  ifstream ifs(filename.c_str());
  ERROR_IF(!ifs, "Error opening filename!");  
  MAFAlignment maf;
  maf.readFASTA(ifs);
  ASSERT(maf.size() == 1);
  HashCorrelationFinder::result_type resultBins(maf.getTotalLength());
  MAFSearchTables searchTables(&maf);
  searchTables.run();
  ASSERT(searchTables.validate());
  HashCorrelationFinder finder(&maf, &searchTables, &resultBins);
  finder.setVerbose(3); // extremely high verbosity
  finder.setOutIntervall(10);
  for (length_type colid = 0; colid < maf.getTotalLength(); ++colid) {
    cout << "Working on column (internal counting): " << colid << endl;
    length_type aliId = maf.getAlignmentId(colid);
    length_type colId = maf.getAlignmentColumnId(colid); // column id with respect to individual alignment
    string slice = maf.getSlice(colid); // getAlignmentSlice(maf[[aliId]], colId)
    string column = NucleotideTools::dnaComplement(slice); // use T instead of U
    cout << "Column " << colid << " " << slice << " : " << column << " " << aliId << " " << colId << endl;
    // this is the vector of all assemblices of that alignment
    Vec<string> colAssemblies = maf.getAlignmentAssemblies(aliId); // = names(extractAssemblyIds.alignment(maf[aliId])) // get names of involved assemblies, like c("hg18", "mm8") etc
    length_type posMin = 0;
    set<length_type> result = finder.searchMafColumn(column, colAssemblies, posMin);
    cout << "Found correlated columns: " << endl;
    for (set<length_type>::const_iterator i = result.begin(); i != result.end(); i++) {
      cout << (*i) << " " << maf.getSlice(*i) << endl;
      ASSERT(NucleotideTools::isComplementary(slice, maf.getSlice(*i), allowGu, allowGap));
    }
  }
  cout << "Finished " << methodName << endl;
}


void
HashCorrelationFinderTest::testRun() {
  string methodName = "testFilterIsolatedCorrelations";
  cout << "Starting " << methodName << endl;  
  string filename = "../../test/fixtures/anticodon_simple2.fa";
  ifstream ifs(filename.c_str());
  ERROR_IF(!ifs, "Error opening filename!");  
  MAFAlignment maf;
  maf.readFASTA(ifs);
  ASSERT(maf.size() == 1);
  HashCorrelationFinder::result_type resultBins(maf.getTotalLength());
  MAFSearchTables searchTables(&maf);
  searchTables.run();
  ASSERT(searchTables.validate());
  HashCorrelationFinder finder(&maf, &searchTables, &resultBins);
  finder.setVerbose(3);
  finder.setRemoveIsolated(false);
  finder.setOutIntervall(10);
  finder.run();
  CorrelationFinder::result_container result = finder.getResults();
  cout << "Final result: Found correlated columns: " << endl;
  for (CorrelationFinder::result_container::iterator i = result.begin(); i != result.end(); i++) {
    cout << (*i) << endl;
  }
  cout << "Finished " << methodName << endl;
}

void
HashCorrelationFinderTest::testRunSlow() {
  string filename = "../../test/fixtures/chr16_18S.maf";
  ifstream ifs(filename.c_str());
  ERROR_IF(!ifs, "Error opening filename!");  
  MAFAlignment maf;
  maf.read(ifs);
  ASSERT(maf.size() > 1);
  HashCorrelationFinder::result_type resultBins(maf.getTotalLength());
  MAFSearchTables searchTables(&maf);
  searchTables.run();
  HashCorrelationFinder finder(&maf, &searchTables, &resultBins);
  finder.run();
  CorrelationFinder::result_container result = finder.getResults();
  for (CorrelationFinder::result_container::iterator i = result.begin(); i != result.end(); i++) {
    cout << (*i) << endl;
  }
}

/** Uses complete mitochondrial genome as test case. */
void
HashCorrelationFinderTest::testRunChrM() {
  string methodName = "testRunChrM";
  cout << "Starting " << methodName << endl;
  string filename = "../../test/fixtures/chrM.maf";
  cout << "Starting to read MAF alignment file " << filename << endl;
  ifstream ifs(filename.c_str());
  ERROR_IF(!ifs, "Error opening filename!");  
  MAFAlignment maf;
  Timer timer;
  timer.start();
  maf.read(ifs);
  timer.stop();
  ASSERT(maf.size() > 1);
  cout << "Succesfully read MAF alignment in time: " << timer << endl; 
  timer.reset();
  timer.start();
  HashCorrelationFinder::result_type resultBins(maf.getTotalLength());
  MAFSearchTables searchTables(&maf);
  searchTables.run();
  HashCorrelationFinder finder(&maf, &searchTables, &resultBins);
  timer.stop();
  cout << "Succesfully initialized MAF alignment in time: " << timer << endl 
       << "Starting search!" << endl; 
  timer.reset();
  timer.start();
  finder.run();
  timer.stop();
  CorrelationFinder::result_container result = finder.getResults();
  cout << "Finished search with " << result.size() << " entries in time " << timer << endl;
  for (CorrelationFinder::result_container::iterator i = result.begin(); i != result.end(); i++) {
    cout << (*i) << endl;
  }
  ASSERT(result.size() == 7);
  // check first entry; remember internal column counting is one less than output to user
  ASSERT(    ((*(result.begin())).getStart() == 2406)
          && ((*(result.begin())).getStop() == 2791) ); 
  //   cout << "Performing second run: " << endl;
  // finder.run();
  // cout << finder.getResults() << endl;
  // ASSERT(result == finder.getResults());
  string refAssembly = "hg18";
  CorrelationTools::convertInternalToAssemblyCoordinates(result.begin(), result.end(), refAssembly, maf);
  cout << "Converted coordinates with respect to assembly (counting starts at one):" << refAssembly << " " << result.size() << " entries in time " << timer << endl;
  for (CorrelationFinder::result_container::iterator i = result.begin(); i != result.end(); i++) {
    cout << (i->getStart() + 1) << " " << (i->getStop() + 1) << endl;
  }
  // hg18 coordinates of first column pair should be: 2024 2271
  ASSERT(result.begin()->getStart() == 2024);
  ASSERT(result.begin()->getStop() == 2271);
  cout << "Finished " << methodName << endl;
}

