#ifndef _INTEGRAL3D_TOOLS_
#define _INTEGRAL3D_TOOLS_

#include <math.h>
#include <debug.h>
#include <Limits.h>

#ifndef PI
#define PI 3.141592654
#endif

class GaussianRadial {
  
 private:

  double alpha;

  double weight;
  
 public:

  GaussianRadial(double _sigma, double _weight) : weight(_weight) { 
    ASSERT(_sigma > 0.0);
    alpha = 1.0/(_sigma*_sigma);
  };

  GaussianRadial(double _sigma) { 
    ASSERT(_sigma > 0.0);
    alpha = 1.0/(_sigma*_sigma);
    weight = 1.0/(_sigma * pow(2.0 * PI, 3.0/2.0));
  };

  double operator () (double radius) const {
    return weight * exp(-alpha * radius * radius);
  }

};

class HardSphereRadial {
  
 private:
  double radius;
  double valInner;
  double valOuter;
  
 public:
  HardSphereRadial(double _radius, double _valInner, double _valOuter) : radius(_radius), 
    valInner(_valInner), valOuter(_valOuter) {
    ASSERT(radius >= 0.0);
  };

  HardSphereRadial(double _radius) : radius(_radius), 
    valInner(1.0), valOuter(0.0) {
    ASSERT(radius >= 0.0);
  };

  HardSphereRadial(double _radius, double _valInner) : radius(_radius), valInner(_valInner), valOuter(0.0) {
    ASSERT(radius >= 0.0);
  };

  double getRadius() const { return radius; }

  double operator () (double r) const {
    return (r <= radius) ? valInner : valOuter;
  }
};


class Integral3DTools {

 public:

  typedef unsigned int size_type;

  template<typename F1, typename F2>
    static
    double radialFunctionOverlapIntegral(const F1& f1, const F2 f2, double d, double dr, size_type N, double rMin) {
    ASSERT(d >= 0.0);
    ASSERT(dr > 0.0);
    ASSERT(N > 0);
    ASSERT(rMin >= 0.0);
    double sum = 0.0;
    for (size_type i = 0; i < N; ++i) {
      double r = i * dr;
      double r1 = r;
      double r2 = r;
      if ((r1 + r2) < d) {
	continue;
      }
      double term1 = d*d - r1*r1 + r2*r2;
      double sqrtArg = 4*d*d*r2-term1*term1;
      if (sqrtArg < 0.0) {
	cout << "Warning: strange internal error: " << sqrtArg << " " << d << " " << r << " " << term1 << endl;
	continue;
      }
      double ringRadius = (1.0/(2.0*d)) * sqrt(sqrtArg);
      ASSERT(isDefined(ringRadius));
      double ringVolume = ringRadius * dr * dr;
      double f1Val = f1(r);
      double f2Val = f2(r);
      ASSERT(isDefined(f1Val));
      ASSERT(isDefined(f2Val));
      sum += ringVolume * f1Val * f2Val;
      // cout << "sum: " << sum << " " << f1Val << " " << f2Val << endl;
    }
    return sum;
  }

  static void testRadialFunctionOverlapIntegral() {
    // double d = 1.0;
    double dr = 0.1;
    size_type N = 100;
    double rMin = 0.0;
    HardSphereRadial sphere1(1.0);
    HardSphereRadial sphere2(1.0);
    double dStep = 0.1;
    for (unsigned int step = 1; step <= 30; ++step) {
      double d = step * dStep; 
      double volume = radialFunctionOverlapIntegral(sphere1, sphere2, d, dr, N, rMin);
      cout << "The volume of two sphere with distance " << d << " and radii "
	   << sphere1.getRadius() << " " << sphere2.getRadius() << " is: " 
	   << volume << " (dr, N, rMin: " << dr << " " << N << " " << rMin << endl;
      ASSERT(((d < 2.0) && (volume > 0.0)) || ((d >= 2.0) && (volume == 0.0)));
    }
  }

  static double gaussianOverlapIntegral(double sigma1, double weight1, 
					double sigma2, double weight2, 
					double d, double dr, size_type N, double rMin) {
    ASSERT(d >= 0.0);
    GaussianRadial g1(sigma1, weight1);
    GaussianRadial g2(sigma2, weight2);
    return radialFunctionOverlapIntegral(g1, g2, d, dr, N, rMin);
  }

};



#endif
