
#ifndef __GEOMETRY_H__
#define __GEOMETRY_H__

#include <math.h>
#include <Vector3D.h>
#include <Matrix3D.h>
#include <Vec.h>

/* result is between 0 and 2PI */
double
phiFromXY(double x, double y);

/* result is between -PI/2 and 3/2PI */
double
thetaFromXYZ(double x, double y, double z);

void
cartesianToPolar(double x, double y, double z,
		 double& r, double& theta, double& phi);
/*
inline
void
cartesianToPolar(const Vector3D& cart,
		 Vector3D& pol);
*/

/* transform polar coordinates to cartesion coordinates.
   as always, angles are measured in radians */
void
polarToCartesian(double &x, double& y, double& z,
		 double  r, double theta, double phi);

/*
  define rotation matrix M, such that
  M * u is parallel to v
*/
Matrix3D
generateAlignMatrix(const Vector3D& u, const Vector3D& v);

// return volume of sphere
double
sphereVolume(double r);

// return intersecting volume (double-sided cap of sphere) of two spheres with radius r1, r2 and distance d
double
sphereIntersectionVolume(double r1, double r2, double d);

// return vector which contains n points belonging to line
Vec<Vector3D>
linePoints(const Vector3D& a, const Vector3D& b, unsigned int n);

// for point set return  union of all line points between these points
Vec<Vector3D>
fillPoints(const Vec<Vector3D>& s, double grid);

// thin out dense points, not closer than grid
Vec<Vector3D>
thinPoints(const Vec<Vector3D>& s, double grid);

inline
void
addVector(Vec<Vector3D>& graph, const Vector3D& v)
{
  for (unsigned int i = 0; i < graph.size(); ++i) {
    graph[i] = graph[i] + v;
  }
}

// return rectengular projection of second vector to first vector
inline
Vector3D
rectangulize(const Vector3D& v1, const Vector3D& v2)
{
  Vector3D v1Norm = v1;
  v1Norm.normalize();
  Vector3D projection = v1Norm * (v1Norm * v2);
  Vector3D result = v2 - projection;
  POSTCOND(isTiny(result * v1)); // must be rectangular
  return result;
}

/* convert list of 3D vectors to distance matrix */
void pointsToField(const Vec<Vector3D>& v, Vec<Vec<double> >& m);

Vec<Vector3D>
generateCubeGridPoint(const Vector3D& minVec, const Vector3D& maxVec, 
		      double step);

Vec<Vector3D>
generateSphereGridPoint(Vector3D& origin, double radius, double step);

/* compute bouding box of fixPoints, such that docked varPoints
   have always space for docking */
void
computeBoundingBox(const Vec<Vector3D>& varPoints,
		   const Vec<Vector3D>& fixPoints,
		   double border,
		   Vector3D& minVec, Vector3D& maxVec);

/* compute bouding box of fixPoints, such that docked varPoints
   have always space for docking */
void
computeBoundingBox(const Vec<Vector3D>& varPoints,
		   const Vec<Vector3D>& fixPoints,
		   double border, double varReduction,
		   Vector3D& minVec, Vector3D& maxVec);

/* compute bouding sphere of fixPoints, such that docked varPoints
   have always space for docking */
void
computeBoundingSphere(const Vec<Vector3D>& varPoints,
		      const Vec<Vector3D>& fixPoints,
		      double border, Vector3D& origin, double& radius );

// use only surface points, which cannot be erased using eraser with radius r
Vec<Vector3D>
erasePoints(const Vec<Vector3D>& surfPoints, const Vec<Vector3D>& solidPoints, double eraserRadius);

// use only surface points, which cannot be erased using eraser with radius r
Vec<Vector3D>
erasePoints(const Vec<Vector3D>& surfPoints, const Vec<Vector3D>& solidPoints, double eraserRadius,
	    double tooCloseRadius, double regrowRadius);

// use only surface points, which cannot be erased using eraser with radius r
Vec<Vector3D>
erasePoints(const Vec<Vector3D>& surfPoints, const Vec<Vector3D>& solidPoints, double eraserRadius,
	    double eraserStep, double tooCloseRadius, double regrowRadius);

// eraser maxClusterNumber number of highest volume clusters
Vec<Vector3D>
eraseSmallClusterPoints(const Vec<Vector3D>& v, double step, unsigned int numLigAtoms, unsigned int maxClusterNumber,
			Vec<unsigned int>& clusterNumbers, double volRedFactor, unsigned int verboseLevel);

/* different version: only allow biggest cluster and all clusters whose volume
   is at least half as large as the volume of the ligand */
Vec<Vector3D>
eraseSmallClusterPoints2(const Vec<Vector3D>& v, double step, 
			 unsigned int numLigAtoms, 
			 Vec<unsigned int>& clusterNumbers, double volRedFactor, unsigned int verboseLevel);

// return largest minus smallest distance from center of mass
double
getClusterDepth(const Vec<Vector3D>& v, const Vector3D& center);

// return to which cluster vector x belongs
unsigned int
getClusterNumber(const Vector3D& x,
		 const Vec<Vector3D>& points,
		 const Vec<unsigned int>& clusterNumbers,
		 double maxDist);

// leave only maxClusterNumber number of most deep clusters
Vec<Vector3D>
eraseShallowClusterPoints(const Vec<Vector3D>& v, double step, 
			  unsigned int numLigAtoms, unsigned int maxClusterNumber,
			  const Vector3D& center, Vec<unsigned int>& clusterNumbers, 
			  double volRedFactor, unsigned int verboseLevel);


// return only points, which have at least minCount neighbours closer or equal to radius
Vec<Vector3D>
eraseSparsePoints(const Vec<Vector3D>& points, double radius, unsigned int minCount);

/* generate vector of points which represent spheres with radius r, 
   those spheres cover all points
*/
Vec<Vector3D>
generateCoveringSpheres(double r, const Vec<Vector3D>& points);

/* return vector which are x, y, z axis plus and minus, with length of r around origin */
Vec<Vector3D>
getCubicVectors(double r, const Vector3D& origin);

double
drms(const Vec<Vector3D>& v1, const Vec<Vector3D>& v2);

/**
 * stretch points with center point and 3 different stretch factors
 */
void
stretchPoints(Vec<Vector3D>& points,
              const Vector3D& center,
              double stretchX,
              double stretchY,
              double stretchZ);

/**
 * bend around center with x coordinates constant
 */
void
bendX(Vec<Vector3D>& pos, const Vector3D& center, double anglePerDist);

/**
 * bend around center with x coordinates constant
 */
void
bendY(Vec<Vector3D>& pos, const Vector3D& center, double anglePerDist);

/**
 * bend around center with x coordinates constant
 */
void
bendZ(Vec<Vector3D>& pos, const Vector3D& center, double anglePerDist);

Vec<Vector3D>
deleteTooClosePoints(const Vec<Vector3D>& dockPoints,
		     const Vec<Vector3D>& pCoord,
		     double minProteinCloseDist);
Vec<Vector3D>
deleteTooFarPoints(const Vec<Vector3D>& dockPoints,
		     const Vec<Vector3D>& pCoord,
		     double minProteinCloseDist);


Vec<unsigned int>
furthestSpanningSubset(const Vec<Vector3D>& v,
                       unsigned int numPoints);


#endif

