// --*- C++ -*------x---------------------------------------------------------
// $Id: histogram2d.cc,v 1.4 2007/03/01 11:43:14 bindewae Exp $
//
// Program:         histogram
//
// Author:          Eckart Bindewald
//
// Project name:    Bayesfold
//
// Date:            $Date: 2007/03/01 11:43:14 $
//
// Description:     Compute histogram from input data. See help output.
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <debug.h>
#include <Vec.h>
#include <StringTools.h>
#include <GetArg.h>
#include <vectornumerics.h>

using namespace std;

const int INTERPOLATE_OUT_OF_RANGE = 1;
const int IGNORE_OUT_OF_RANGE = 2;

void
helpOutput(ostream& os) 
{
  os << "Compute histgram from data stream. Usage: " << endl;
  os << "histogram [--binx value][--biny value][--deltax value][--deltay value][--eval x y][--field value][--file filename][--line 0|1][--minx value][--miny value][--norm 0|1|2] | --pseudo pseudocount][--range 1|2][--sum 0|1][--of outputformat][--vector][--verbose value] < inputdata" << endl;
  os << "Data is in gnuplot-like format (default) or vector format (--vector)" 
     << endl;
  
}

/** return historgramm: vector of bins filled with counts */
Vec<Vec<double> >
compute2DHistogram(const Vec<double>& dataX,
		   const Vec<double>& dataY,
		   double minX,
		   double minY,
		   double deltaX, 
		   double deltaY, 
		   unsigned int numBinsX,
		   unsigned int numBinsY,
		   double pseudoCount)
{
  Vec<Vec<double> > result(numBinsX, Vec<double>(numBinsY, pseudoCount));
  int binX = 0; 
  int binY = 0; 
  for (unsigned int i = 0; i < dataX.size(); ++i) {
    binX = static_cast<int>((dataX[i] - minX) / deltaX);
    binY = static_cast<int>((dataY[i] - minY) / deltaY);
    if ((binX >= 0) && (binX < static_cast<int>(result.size()) )  
	&& (binY >= 0) && (binY < static_cast<int>(result[0].size()) ) )  {
      ++result[binX][binY];
    }
  }
  return result;
}

/** multiplies each element with factor scale */
void
scaleHistogram(Vec<Vec<double> >& histData, 
	       double scale) {
  for (Vec<Vec<double> >::size_type i = 0; i < histData.size(); ++i) {
    for (Vec<double>::size_type j = 0; j < histData[i].size(); ++j) {
      histData[i][j] *= scale;
    }
  }
}

double
evaluate(const Vec<Vec<double> >& hist,
	 double minX,
	 double minY,
	 double deltaX, 
	 double deltaY,
	 double x, double y,
	 int interpolationMode,
	 int outOfRangeMode) 
{
  int binX = static_cast<int>((x - minX) / deltaX);
  if (binX >= static_cast<int>(hist.size())) {
    if (outOfRangeMode == IGNORE_OUT_OF_RANGE) {
      return 0.0;
    }
    binX = hist.size() - 1;
    interpolationMode = 0;
  }
  else if (binX < 0) {
    if (outOfRangeMode == IGNORE_OUT_OF_RANGE) {
      return 0.0;
    }
    binX = 0;
    interpolationMode = 0;
  }
  int binY = static_cast<int>((y - minY) / deltaY);
  if (binY >= static_cast<int>(hist[0].size() ) ) {
    if (outOfRangeMode == IGNORE_OUT_OF_RANGE) {
      return 0.0;
    }
    binY = hist[0].size() - 1;
    interpolationMode = 0;
  }
  else if (binY < 0) {
    if (outOfRangeMode == IGNORE_OUT_OF_RANGE) {
      return 0.0;
    }
    binY = 0;
    interpolationMode = 0;
  }
  ASSERT((binX >= 0) && (binX < static_cast<int>(hist.size()) )  
	 && (binY >= 0) && (binY < static_cast<int>(hist[0].size() )));
  switch (interpolationMode) {
  case 0: return hist[binX][binY];
  case 1: ERROR("Interpolation not yet implemented!");
  default: ERROR("Unknown interpolation mode!");
  }

  ERROR("Should not reach this point!");
  return 0.0;
}


/** reads matrix in format that is also used by program "R" */
/*
Vec<Vec<double> >
readPlainMatrix(istream& is)
{
  Vec<Vec<double > > result;
  while (is) {
    string line = getLine(is);
    Vec<string> words = getTokens(line);
    if (words.size() > 0) {
      Vec<double> row(words.size());
      for (unsigned int i = 0; i < words.size(); ++i) {
	row[i] = stod(words[i]);
      }
      result.push_back(row);
    }
  }
  return result;
}
*/

int
main(int argc, char ** argv)
{
  int inputFormat = 1;
  int normalizeMode = 0;
  int outOfRangeMode = INTERPOLATE_OUT_OF_RANGE;
  bool vectorMode = false;
  unsigned int field = 1;
  int interpolationMode = 0;
  unsigned int lineMode = 0;
  unsigned int numBinx  = 100; // number of bins
  unsigned int numBiny  = 100; // number of bins
  int outputFormat = 1;
  unsigned int sumMode = 0;
  unsigned int verboseLevel = 1;

  string histFileName;
  string line;
  // read data
  Vec<double> datax;
  Vec<double> datay;
  vector<double> evalPos;
  double minx = 0.0; // minimum value counted in bins
  double deltax = 1.0; // breadth of bins
  double miny = 0.0; // minimum value counted in bins
  double deltay = 1.0; // breadth of bins
  double pseudoCount = 0.0;

  getArg("-eval", evalPos, argc, argv);
  getArg("f", field, argc, argv, field);
  getArg("-file", histFileName, argc, argv, histFileName);
  getArg("-minx", minx, argc, argv, minx);
  getArg("-deltax", deltax, argc, argv, deltax);
  getArg("-binx", numBinx, argc, argv, numBinx);
  getArg("-miny", miny, argc, argv, miny);
  getArg("-deltay", deltay, argc, argv, deltay);
  getArg("-biny", numBiny, argc, argv, numBiny);
  getArg("-line", lineMode, argc, argv, lineMode);
  getArg("-normalize", normalizeMode, argc, argv, normalizeMode);
  getArg("-norm", normalizeMode, argc, argv, normalizeMode);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("-pseudo", pseudoCount, argc, argv, pseudoCount);
  getArg("-range", outOfRangeMode, argc, argv, outOfRangeMode);
  getArg("-sum", sumMode, argc, argv, sumMode);
  getArg("-vector", vectorMode, argc, argv);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);

  if (verboseLevel > 0) {
    helpOutput(cout);
  }

  ERROR_IF(field == 0, "Field descriptor (-f value) must be great zero!" );
  --field;  // internal counting starts at zero
  if (verboseLevel > 1) {
    cout << "Reading input data...";
  }

//   if (vectorMode) {
//     cin >> data;
//   }
//   else {
  Vec<Vec<double> > result;

  if (histFileName.size() == 0) {
    // input of raw data"
    while (cin) {
      line = getLine(cin);
      Vec<string> words = getTokens(line);
      if (words.size() == 0) {
	continue;
      }
      if (words[0][0] == '#') {
	continue;
      }
      if (words.size() == 2) {
	datax.push_back(stod(words[0]));
	datay.push_back(stod(words[1]));
      }
    }
  
    //   }
    if (verboseLevel > 1) {
      cout << "done." << endl;
      cout << datax.size() << " value pairs read." << endl;
    }
    if (verboseLevel > 2) {
      cout << "Read data:" << endl;
      outList(cout,datax);
      cout << endl;
    }      
    result = compute2DHistogram(datax, datay, minx, miny, deltax, deltay, numBinx, numBiny, pseudoCount);
  }
  else {
    ifstream histFile(histFileName.c_str());
    ERROR_IF(!histFile, "Error opening histogram file!");
    switch (inputFormat) {
    case 0:
      // no input at all
      break;
    case 1:
      result = readPlainMatrix(histFile);
      break;
    case 3: // repeat for compatibility reasons
      result = readPlainMatrix(histFile);
      break;
    default:
      ERROR("Unknown input format!");
    }
  }

  double norm = 1.0;
  switch (normalizeMode) {
  case 0: 
    break;
  case 1: // divide by size of x vector !?
    cerr << "Normalizing!" << endl;
    norm = 1.0 / static_cast<double>(datax.size());
    scaleHistogram(result, norm);
    break;
  case 2: // dividide such that sum of all elements is one:
    norm = 1.0 / elementSum(result);
    scaleHistogram(result, norm);
    break;
  default:
    ERROR("Unknown normalization mode!");
  }

  switch (outputFormat) {
  case 0: // not output at all
    break;
  case 1:
    for (unsigned int i = 0; i < result.size(); ++i) {
      for (unsigned int j = 0; j < result[0].size(); ++j) {
	//       if (lineMode && (i > 0) && (result[i] == result[i-1])) {
	//         x += deltax;
	//         continue;
	//       }
	cout << result[i][j] << " ";
	//x += delta;
      }
      cout << endl;
    }
    break;
  default:
    ERROR("Unknown output format!");
  }

  if (evalPos.size() == 2) {
    double evalResult = evaluate(result,
				 minx, miny, deltax, deltay,
				 evalPos[0], evalPos[1], interpolationMode,
				 outOfRangeMode);
    cout << "Evaluation result: " << evalResult << endl;
  }

  return 0;
}
