#include <vectornumerics.h>
#include <math.h>
// #define NRANSI
// #include <nrutil.h>
// #define TINY 1.0e-20;

#include <debug.h>
#include <Vec.h>

// #include <Matrix4D.h>
#include <Limits.h>
#include <Random.h>
#include <RankedSolution5.h>

/** n trials with counts.size() possible outcomes are performed.
 * Given a vector of probabilities of observing each outcome,
 * it returns the probability of observing a particular patterns of counts of outcomes.
 * Corresponds to trials "with replacement"; use hypergeometric distribution if "without replacement"
 */
double
logMultinomial(const Vec<int>& counts, const Vec<double>& probs, int n) {
  double result = logFactorial(n);
  for (Vec<double>::size_type i = 0; i < probs.size(); ++i) {
    result += log(pow(probs[i], counts[i])) - logFactorial(counts[i]);
  }
  return result;
}

// return error function for positive values of z
// approximation according to Abramowitz and Stegun, 1964 
// (http://www.geo.vu.nl/users/hydro/students/hd537/error_function.pdf)
/*
double
errorFunction(double z)
{
  const double a1 = 0.254829592;
  const double a2 = -0.284496736;
  const double a3 = 1.421413741;
  const double a4 = -1.453152027;
  const double a5 = 1.061405429;
  const double a6 = 0.3275911;
  double b = 1.0 / (1.0 + a6 * z);
  double result = 1.0 - (a1 * b + a2*b*b + a3*b*b*b + a4 *b*b*b*b + a5 * pow(b, 5.0)) * exp(-b*b);
  POSTCOND((result >= 0.0) && (result <= 1.0));
  return result;
}
*/

/** error function.
    from: http://dehesa.freeshell.org/NCSE/12/error_f.f
*/
double
errorFunction(double x)
{
  double t = 1.0/(1.0 + 0.3275911*fabs(x));    
  double erfun_c = t * exp(-(x*x))*(0.254829592 + t*(-0.284496736 +
	     + t *( 1.421413741 + t * (-1.453152027 + t*1.061405429 ) ) ) );  
  if(x < 0.0) {
    erfun_c = 2.0-erfun_c;
  }
  return 1.0 - erfun_c;
}

Vec<int>
uivtoiv(const Vec<unsigned int>& v) 
{
  Vec<int> result(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    result[i] = static_cast<int>(v[i]);
  }
  return result;
}

Vec<unsigned int>
ivtouiv(const Vec<int>& v) 
{
  Vec<unsigned int> result(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    result[i] = static_cast<unsigned int>(v[i]);
  }
  return result;
}

double
varianceFromSum(double sum, double sumSquares, unsigned int n)
{
  if (n < 2) {
    return 0.0;
  }
  double nDub = static_cast<double>(n);
  double result = (nDub/(nDub-1.0)) * ((sumSquares/n) - ((sum * sum) / (n * n)));
  if (result < 0.0) {
    result = 0.0;
  }
  return result;
} 


/* my version of "winner takes most" ranking */
double
computeWinnerTakesMostScore(const Vec<double>& vOrig, double maxVal)
{
  PRECOND(maxVal > 0.0);
  Vec<double> v = vOrig;
  sort(v.begin(), v.end());
  reverse(v.begin(), v.end());
  double result = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (result >= maxVal) {
      break;
    }
    double restFrac = (maxVal - result) / maxVal;
    result += v[i] * restFrac;
  }
  return result;
}

/** output of histogram x axis */
void
outBins(ostream& os, double min, double delta, unsigned int numBins)
{
  for (unsigned int i = 0; i < numBins; ++i) {
    os << i << "\t";
  }
  os << endl;
  for (unsigned int i = 0; i < numBins; ++i) {
    os << min + i * delta << "\t";
  }
  os << endl;
}

// generate array starting with 0 , incresing like a stair
Vec<unsigned int>
generateStair(unsigned int maxi, unsigned int mini, unsigned int step)
{
  PRECOND(step > 0);
  if (maxi <= mini) {
    return Vec<unsigned int>(); // return "empty" stair
  }
  unsigned int d = (maxi -mini)/step;
  Vec<unsigned int> result(d);
  for (unsigned int i = 0; i < d; ++i) {
    result[i] = (i*step) + mini;
    ASSERT(result[i] < maxi);
  }
  return result;
}

/**
 * generates loopup table: i'th element of result is position of value of i in v
 */
Vec<unsigned int>
getIndexSet(const Vec<unsigned int>& v) {
  // find maximum index:
  unsigned int maxSize = v[findMaxIndex(v)] + 1;
  Vec<unsigned int> result(maxSize, 0U);
  for (unsigned int i = 0; i < v.size(); ++i) {
    result[v[i]] = i;
  }
  return result;
}

/* return vector with size numReturn, which has randomly assigned non-redundant bin values between min and max */
Vec<double>
rouletteWheel(double min, double max, unsigned int numBins, unsigned int numReturn)
{
  PRECOND(numBins > 0);
  Vec<double> result(numReturn);
  Vec<double> stair(numBins);
  Random& rnd = Random::getInstance();
  double step = (max - min) / numBins;
  for (unsigned int i = 0; i < numBins; ++i) {
    stair[i] = min + step * i;
  }
  random_shuffle(stair.begin(), stair.end(), rnd);
  for (unsigned int i = 0; i < numReturn; ++i) {
    result[i] = stair[i];
  }
  POSTCOND(result.size() == numReturn);
  return result;
}

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 */
unsigned int
chooseRouletteWheel(const Vec<double>& pVec)
{
  PRECOND(pVec.size() > 0);
  if (pVec.size() == 1) {
    return 0;
  }
  double totalSum = elementSum(pVec);
  Random& rnd = Random::getInstance();
  double y = rnd.getRandf() * totalSum;
  double sum = 0.0;
  unsigned int result = 0;
  bool found = false;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    ASSERT(pVec[i] >= 0.0);
    if ((y >= sum) && (y <= (sum + pVec[i]))) {
      result = i;
      found = true;
      break;
    }
    sum += pVec[i];
  }
  ERROR_IF(!found, "Internal error in chooseRouletteWheel!"); // should never be here!
  POSTCOND(result < pVec.size());
  return result; // dummy result
}

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel(const Vec<double>& pVec, const Vec<unsigned int>& tabooList)
{
  PRECOND(pVec.size() > 0);
  PRECOND(tabooList.size() < pVec.size());
  if (pVec.size() == 1) {
    return 0;
  }
  double totalSum = 0.0;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    bool found = false; // check taboo list:
    if (findFirstIndex(tabooList, i) < tabooList.size()) {
	found = true;
    }
    if (!found) {
      totalSum += pVec[i];
    }
  }
  Random& rnd = Random::getInstance();
  double y = rnd.getRandf() * totalSum;
  double sum = 0.0;
  unsigned int result = 0;
  bool found = false;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    ASSERT(pVec[i] >= 0.0);
    found = false; // check taboo list:
    if (findFirstIndex(tabooList, i) < tabooList.size()) {
      found = true;
      ASSERT(tabooList.size() > 0);
    }
    if (! found) {
      if ((y >= sum) && (y <= (sum + pVec[i]))) {
	result = i;
	found = true;
	break;
      }
      sum += pVec[i];
    }
  }
  if (!found) {
    cout << "chooseRouletteWheel called with: " << y << " " << pVec << endl << tabooList << endl;
  }
  ERROR_IF(!found, "Internal error in chooseRouletteWheel!"); // should never be here!
  POSTCOND(result < pVec.size());
  POSTCOND(findFirstIndex(tabooList, result) >= tabooList.size());
  return result; // dummy result
}

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel2(const Vec<double>& pVec, const Vec<unsigned int>& tabooList)
{
  PRECOND(pVec.size() > 0);
  PRECOND(tabooList.size() < pVec.size());
  if (pVec.size() == 1) {
    return 0;
  }
  if (tabooList.size() == 0) {
    return chooseRouletteWheel(pVec);
  }
  double totalSum = 0.0;
  Vec<unsigned int> indexList;
  Vec<double> newP;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    bool found = false; // check taboo list:
    if (findFirstIndex(tabooList, i) < tabooList.size()) {
	found = true;
    }
    if (!found) {
      totalSum += pVec[i];
      indexList.push_back(i);
      newP.push_back(pVec[i]); // new pie without taboo elements
    }
  }
  ASSERT(newP.size() > 0);
  unsigned int result = chooseRouletteWheel(newP);
  ASSERT(result < indexList.size());
  result = indexList[result]; // translate to old indices
  POSTCOND(result < pVec.size());
  POSTCOND(findFirstIndex(tabooList, result) >= tabooList.size());
  return result; // dummy result
}

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 * Fast implementation based on set
 */
unsigned int
chooseRouletteWheel3(const Vec<double>& pVec, const set<unsigned int>& tabooList)
{
  PRECOND(pVec.size() > 0);
  PRECOND(tabooList.size() < pVec.size());
  if (pVec.size() == 1) {
    return 0;
  }
  double totalSum = 0.0;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    bool found = false; // check taboo list:
    if (tabooList.find(i) != tabooList.end()) {
      found = true;
    }
    if (!found) {
      totalSum += pVec[i];
    }
  }
  Random& rnd = Random::getInstance();
  double y = rnd.getRandf() * totalSum;
  double sum = 0.0;
  unsigned int result = 0;
  bool found = false;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    ASSERT(pVec[i] >= 0.0);
    found = false; // check taboo list:
    if (tabooList.find(i) != tabooList.end()) {
      found = true;
    }
    if (! found) {
      if ((y >= sum) && (y <= (sum + pVec[i]))) {
	result = i;
	found = true;
	break;
      }
      sum += pVec[i];
    }
  }
  ERROR_IF(!found, "Internal error in chooseRouletteWheel!"); // should never be here!
  POSTCOND(result < pVec.size());
  // POSTCOND(findFirstIndex(tabooList, result) >= tabooList.size());
  return result;
}

/**
 * Uses thread-safe version of rand_r for random number generation
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 * Fast implementation based on set
 */
unsigned int
chooseRouletteWheel4(const Vec<double>& pVec, const set<unsigned int>& tabooList, unsigned int * seed)
{
  PRECOND(pVec.size() > 0);
  PRECOND(tabooList.size() < pVec.size());
  if (pVec.size() == 1) {
    return 0;
  }
  double totalSum = 0.0;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    bool found = false; // check taboo list:
    if (tabooList.find(i) != tabooList.end()) {
      found = true;
    }
    if (!found) {
      totalSum += pVec[i];
    }
  }
  // Random& rnd = Random::getInstance();
  // double y = rnd.getRandf() * totalSum;
  double rf = rand_r(seed) / static_cast<double>(RAND_MAX);
  ASSERT(rf >= 0.0);
  ASSERT(rf <= 1.0);
  double y = rf * totalSum;
  double sum = 0.0;
  unsigned int result = 0;
  bool found = false;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    ASSERT(pVec[i] >= 0.0);
    found = false; // check taboo list:
    if (tabooList.find(i) != tabooList.end()) {
      found = true;
    }
    if (! found) {
      if ((y >= sum) && (y <= (sum + pVec[i]))) {
	result = i;
	found = true;
	break;
      }
      sum += pVec[i];
    }
  }
  ERROR_IF(!found, "Internal error in chooseRouletteWheel!"); // should never be here!
  POSTCOND(result < pVec.size());
  // POSTCOND(findFirstIndex(tabooList, result) >= tabooList.size());
  return result;
}

/**
 * Uses thread-safe version of rand_r for random number generation
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 * Fast implementation based on set
 */
unsigned int
chooseRouletteWheel4(const Vec<double>& pVec, const Vec<unsigned int>& tabooList, unsigned int * seed)
{
  PRECOND(pVec.size() > 0);
  PRECOND(tabooList.size() < pVec.size());
  if (pVec.size() == 1) {
    return 0;
  }
  double totalSum = 0.0;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    bool found = false; // check taboo list:
    if (findFirstIndex(tabooList, i) < tabooList.size()) {
      found = true;
    }
    if (!found) {
      totalSum += pVec[i];
    }
  }
  // Random& rnd = Random::getInstance();
  // double y = rnd.getRandf() * totalSum;
  double rf = rand_r(seed) / static_cast<double>(RAND_MAX);
  ASSERT(rf >= 0.0);
  ASSERT(rf <= 1.0);
  double y = rf * totalSum;
  double sum = 0.0;
  unsigned int result = 0;
  bool found = false;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    ASSERT(pVec[i] >= 0.0);
    found = false; // check taboo list:

    if (findFirstIndex(tabooList, i) < tabooList.size()) {
      found = true;
    }
    if (! found) {
      if ((y >= sum) && (y <= (sum + pVec[i]))) {
	result = i;
	found = true;
	break;
      }
      sum += pVec[i];
    }
  }
  ERROR_IF(!found, "Internal error in chooseRouletteWheel!"); // should never be here!
  POSTCOND(result < pVec.size());
  // POSTCOND(findFirstIndex(tabooList, result) >= tabooList.size());
  return result;
}


/**
 * Uses thread-safe version of rand_r for random number generation
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 * Fast implementation based on set
 */
unsigned int
chooseRouletteWheel4(const Vec<double>& pVec, unsigned int * seed)
{
  PRECOND(pVec.size() > 0);
  if (pVec.size() == 1) {
    return 0;
  }
  double totalSum = 0.0;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    totalSum += pVec[i];
  }
  // Random& rnd = Random::getInstance();
  // double y = rnd.getRandf() * totalSum;
  double rf = rand_r(seed) / static_cast<double>(RAND_MAX);
  ASSERT(rf >= 0.0);
  ASSERT(rf <= 1.0);
  double y = rf * totalSum;
  double sum = 0.0;
  unsigned int result = 0;
  bool found = false;
  for (unsigned int i = 0; i < pVec.size(); ++i) {
    ASSERT(pVec[i] >= 0.0);
    if ((y >= sum) && (y <= (sum + pVec[i]))) {
      result = i;
      found = true;
      break;
    }
    sum += pVec[i];
  }
  ERROR_IF(!found, "Internal error in chooseRouletteWheel!"); // should never be here!
  POSTCOND(result < pVec.size());
  // POSTCOND(findFirstIndex(tabooList, result) >= tabooList.size());
  return result;
}



/**
 * generates vecor of unsigned int with num entries between minId and maxId
 * no indices can be double, so no "replacement" (see also: generateBootSubset)
 */
Vec<unsigned int>
generateRandomIndexSubset(unsigned int num,
                          unsigned int maxId, unsigned int minId) {
  Vec<unsigned int> stair = generateStair(maxId, minId);
  Random& rnd = Random::getInstance();
  random_shuffle(stair.begin(), stair.end(), rnd);
  Vec<unsigned int> result(num);
  for (unsigned int i = 0; i < num; ++i ) {
    result[i] = stair[i];
  }
  return result;
}

/**
 * generates vecor of unsigned int with num entries between 0 and num
 * no indices can be double, WITH "replacement", use for bootstrap!
 */
Vec<unsigned int>
generateBootSubset(unsigned int num) {
  if (num == 0) {
    return Vec<unsigned int>();
  }
  Random& rnd = Random::getInstance();
  Vec<unsigned int> result(num);
  for (unsigned int i = 0; i < num; ++i ) {
    result[i] = rnd.getRand(static_cast<long>(num));
  }
  return result;
}

/** return false if not a number (NaN) */
bool isDefined(const Vector3D& v) {
  return isDefined(v.x()) && isDefined(v.y()) && isDefined(v.z());
}

Vec<int>
uiVecToIVec(const Vec<unsigned int>& v)
{
  Vec<int> result(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    result[i] = static_cast<int>(v[i]);
  }
  return result;
}

Vec<unsigned int>
iVecToUiVec(const Vec<int>& v)
{
  Vec<unsigned int> result(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    ERROR_IF(v[i] < 0, 
      "Error converting unsigned int vector to int!");
    result[i] = static_cast<unsigned int>(v[i]);
  }
  return result;
}

void
unitMatrix(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      if (i != j) {
	matrix[i][j] = 0.0;
      }
      else {
	matrix[i][j] = 1.0;
      }
    }
  }
  POSTCOND(isReasonable(matrix));
}

/** a <- b a  */
void
scalarMul(Vec<double>& a, double b)
{
  for (unsigned int i = 0; i < a.size(); ++i) {
    a[i] = a[i] *b;
  }
}

/** c = a . b */
double
dotProduct(const Vec<double>& a,
	   const Vec<double>& b)
{
  PRECOND((a.size() == b.size()));
  double result = 0.0;
  for (unsigned int i = 0; i < a.size(); ++i) {
    result += a[i]*b[i];
  }
  return result;
}

/** return sum of elements */
double
elementSum(const Vec<double>& v) {
  double sum = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += v[i];
  }
  return sum;
}

// transform such that sum of values is 1.0
void
probabilityNormalize(Vec<double>& v) {
  double sum = elementSum(v);
  if (v.size() == 0) {
    return;
  }
  if (sum == 0.0) {
    double tmp = 1.0 / static_cast<double>(v.size());
    for (unsigned int i = 0; i < v.size(); ++i) {
      v[i] = tmp;
    }
    return;  // do nothing
  }
  ERROR_IF(sum < 0.0, "Vector not normalizable, sum negative.");
//   ERROR_IF(!isReasonable(sum), "Vector not normalizable, weird members.");
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] /= sum;
  }
}

// transform such that sum of values is targetSum
void
probabilityNormalize(Vec<double>& v, double targetSum) {
  PRECOND(targetSum > 0.0);
  double sum = elementSum(v);
  if (sum == 0.0) {
    return;  // do nothing
  }
  sum /= targetSum;
  //  ERROR_IF(sum < 0.0, "Vector not normalizable.");
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] /= sum;
  }
}

// transform such that sum of values is 1.0, must start from raw counts, add apriori probability
void
bayesianNormalize(Vec<double>& v) {
  double sum = elementSum(v) + v.size();
  //  ERROR_IF(sum < 0.0, "Vector not normalizable.");
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] = (v[i] + 1.0)/sum;
  }
}


// transform such that sum of values is 1.0
void
probabilityNormalize(Vec<Vec<double> >& v) {
  double sum = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += elementSum(v[i]);
  }
  if (sum == 0.0) {
    return;  // do nothing
  }
  //  ERROR_IF(sum < 0.0, "Vector not normalizable.");
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      v[i][j] /= sum;
    }
  }
}

// transform such that sum of values is 1.0. Must start from raw counts (instead of x/n we use x+1/(n+c), c being number of classes)
void
bayesianNormalize(Vec<Vec<double> >& v) {
  double sum = v.size()*v.size();
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += elementSum(v[i]);
  }
  //  ERROR_IF(sum < 0.0, "Vector not normalizable.");
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      v[i][j] = (v[i][j] + 1.0) / sum;
    }
  }
}

// compute euclidian norm of vector
/*
double
euclidianNorm(const Vec<double>& v) {
  double sum = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += v[i] * v[i];
  }
  if (sum <= 0.0) {
    return 0.0;  // do nothing
  }
  sum = sqrt(sum);
  POSTCOND(sum >= 0.0);
  return sum;
}
*/

// transform such that sum of SQUARED values is 1.0
void
normalizeEuclidian(Vec<double>& v) {
  double sum = euclidianNorm(v);
  //  ERROR_IF(sum < 0.0, "Vector not normalizable.");
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] /= sum;
  }
}

/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
*/
void
transformToZScores(Vec<double>& v)
{
  double mean = vecMean(v);
  double sigma = vecVariance(v); // variance
  if (sigma <= 0.0) {
    for (unsigned int i = 0; i < v.size(); ++i) {
      v[i] = 0.0;
    }
    return; // all components are identical
  }
  sigma = sqrt(sigma); // deviation
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] = (v[i] - mean) / sigma;
  }
}

/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
*/
void
transformToZScores(Vec<Vec<double> >& m)
{
  Vec<double> v = flatten(m);
  double mean = vecMean(v);
  double sigma = vecVariance(v); // variance
  if (sigma <= 0.0) {
    for (unsigned int i = 0; i < m.size(); ++i) {
      for (unsigned int j = 0; j < m[i].size(); ++j) {
	m[i][j] = 0.0;
      }
    }
    return; // all components are identical
  }
  sigma = sqrt(sigma); // deviation
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      m[i][j] = (m[i][j] - mean) / sigma;
    }
  }
}


/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
    use only subset indices for computing mean and std
*/
void
transformToZScores(Vec<double>& v,
		   const Vec<unsigned int>& subSet)
{
  PRECOND(subSet.size() > 0);
  double mean = vecMean(getSubset(v, subSet));
  double sigma = vecVariance(getSubset(v, subSet)); // variance
  if (sigma <= 0.0) {
    for (unsigned int i = 0; i < v.size(); ++i) {
      v[i] = 0.0;
    }
    return; // all components are identical
  }
  sigma = sqrt(sigma); // deviation
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] = (v[i] - mean) / sigma;
  }
}


/** moves centroid of Vec <Vector3D> to ( 0, 0, 0)
  */
Vector3D moveCenter( Vec <Vector3D> &v1 )
{
  double xmid   = 0.0;
  double ymid   = 0.0;
  double zmid   = 0.0;
  double weight = 1.0;
  double norm   = 0.0;
  
  for (unsigned int i=0; i < v1.size(); i++ )
    { 
      xmid = xmid + v1[i].x()*weight;
      ymid = ymid + v1[i].y()*weight;
      zmid = zmid + v1[i].z()*weight;
      norm = norm + weight;
    };
  
  xmid = xmid / norm;
  ymid = ymid / norm;
  zmid = zmid / norm;
  
  for (unsigned int i=0; i < v1.size(); i++ )
    {
      v1[i].x( v1[i].x() - xmid );
      v1[i].y( v1[i].y() - ymid );
      v1[i].z( v1[i].z() - zmid );
    };
  return Vector3D(xmid, ymid, zmid);
}

/** compute optimal superposition of coordinates of
    molecules m1 and m2
    m2' = (m2-center(m2)) * rot + v
*/
void
minRmsSuperpose(const Vec<Vector3D>& v1Orig, Vec<Vector3D>& v2, 
		Matrix3D& rot, Vector3D& v)
{
  PRECOND(v1Orig.size() == v2.size());
  Vec<Vector3D> v1 = v1Orig; // safety copy
  Vector3D c1 = moveCenter(v1);
  Vector3D c2 = moveCenter(v2);
  ERROR("Sorry, quatfit not supported anymore!");
  // rot = quatfit(v1,v2); 
  // shiftAtoms(v2, c1); // move to center of m1!
  vectorAdd(v2, c1, v2);
  v = c1 - c2; // translation which is performed on m2 // NOT SURE IF CORRECT!??
}


/** compute optimal superposition of coordinates of
    molecules m1 and m2
    m2' = rot * (m2-c2)  + c1
    @deprecated
*/
void
minRmsSuperpose(const Vec<Vector3D>& v1Orig, Vec<Vector3D>& v2, 
		Matrix3D& rot, Vector3D& c1, Vector3D& c2)
{
  PRECOND(v1Orig.size() == v2.size());
  Vec<Vector3D> v1 = v1Orig; // safety copy
  c1 = moveCenter(v1);
  c2 = moveCenter(v2);
  ERROR("Sorry, quatfit not supported anymore!");
  // rot = quatfit(v1,v2); 
  // shiftAtoms(v2, c1); // move to center of m1!
  vectorAdd(v2, c1, v2);
  //    v = c1 - c2; // translation which is performed on m2
  //    oldCenter = c2;
}


/** compute optimal superposition of coordinates of
    molecules m1 and m2
*/
void
minRmsSuperpose(const Vec<Vector3D>& v1, Vec<Vector3D>& v2)
{
  PRECOND(v1.size() == v2.size());
  Vector3D v(0.0,0.0,0.0);
  Matrix3D m(1.0,1.0,1.0);
  minRmsSuperpose(v1, v2, m, v);
}



double
getMinDist(const Vector3D& v, const Vec<Vector3D>& c)
{
  double result = vecDistance(v,c[0]);
  for (unsigned int i = 1; i < c.size(); ++i) {
    double dist = vecDistance(c[i],v);
    if ( dist < result) {
      result = dist;
    }
  }
  return result;
}

Vec<unsigned int>
getConnected(const Vec<Vector3D> v, double dist, unsigned int start)
{
  PRECOND(start < v.size());
  Vec<unsigned int> result;
  Vec<unsigned int> touched(v.size(), 0U);
  result.push_back(start);
  unsigned int clustered = 1;
  touched[start] = 1;
  while (clustered < v.size()) {
    // get next best vector to existing cluster
    double bestDist = dist;
    unsigned int bestIndex = v.size();
    Vec<Vector3D> subs = getSubset(v,result);
    for (unsigned int i = 1; i < v.size(); ++i) {
      if (touched[i] == 0) {
	double d = getMinDist(v[i], subs);
	if (d < bestDist) {
	  bestDist = d;
	  bestIndex = i;
	}
      }
    }
    if (bestIndex >= v.size()) {
      return result; // nothing further found
    }
    result.push_back(bestIndex);
    touched[bestIndex] = 1;
    ++clustered;
  }
  return result; // all connected!
}


unsigned int findLargestDistanceIndex(const Vector3D& v, 
				       const Vec<Vector3D>& data)
{
  double bestDist = -1.0;
  unsigned int bestIndex = 0;
  for (unsigned int i = 0; i < data.size(); ++i) {
    double dist = vecDistance(v,data[i]);
    if (dist > bestDist) {
      bestDist = dist;
      bestIndex = i;
    }
  }
  return bestIndex;
}

unsigned int findSmallestDistanceIndex(const Vector3D& v, 
				       const Vec<Vector3D>& data)
{
  double bestDist = -1.0;
  unsigned int bestIndex = 0;
  for (unsigned int i = 0; i < data.size(); ++i) {
    double dist = vecDistance(v,data[i]);
    if ((bestDist < 0) || (dist < bestDist)) {
      bestDist = dist;
      bestIndex = i;
    }
  }
  return bestIndex;
}

/* find smallest distance point not belonging to forbidden set */
unsigned int findSmallestDistanceIndex(const Vector3D& v, 
       const Vec<Vector3D>& data, const Vec<unsigned int>& forbidden)
{
  double bestDist = -1.0;
  unsigned int bestIndex = 0;
  for (unsigned int i = 0; i < data.size(); ++i) {
    bool found = false;
    for (unsigned int j = 0; j < forbidden.size(); ++j) {
      if (i == forbidden[j]) {
	found = true;
	break;
      }
    }
    if (found) {
      continue; // ignore this value because in forbidden set
    }
    double dist = vecDistance(v,data[i]);
    if ((bestDist < 0) || (dist < bestDist)) {
      bestDist = dist;
      bestIndex = i;
    }
  }
  return bestIndex;
}

/* find n closes points */
Vec<unsigned int>
findSmallestDistanceIndices(const Vector3D& v, 
			    const Vec<Vector3D>& data,
			    unsigned int n)
{
  PRECOND(n <= data.size());
  Vec<unsigned int> result;
  while (result.size() < n) {
    result.push_back(findSmallestDistanceIndex(v, data, result));
  }
  return result;
}

/* find n closes points */
Vec<unsigned int>
findSmallestDistanceIndices(const Vector3D& v, 
			    const Vec<Vector3D>& data,
			    unsigned int n,
			    double minRadius,
			    double& maxRadius)
{
  PRECOND(n <= data.size());
  Vec<unsigned int> result;
  maxRadius = 0.0;
  while (result.size() < n) {
    unsigned int ind = findSmallestDistanceIndex(v, data, result);
    double d = vecDistance(v, data[ind]);
    if (d > minRadius) {
      continue;
    }
    if (d > maxRadius) {
      maxRadius = d;
    }
    result.push_back(ind);
  }
  return result;
}

void
modulateColumn(Vec<Vec<double> >& matrix,
	       const Vec<double>& modulateValues,
	       Vec<Vec<double> >::size_type colId) {
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    matrix[i][colId] *= modulateValues[i];
  }

}

void
modulateRow(Vec<Vec<double> >& matrix,
	    const Vec<double>& modulateValues,
	    Vec<Vec<double> >::size_type rowId) {
  for (Vec<Vec<double> >::size_type i = 0; i < matrix[0].size(); ++i) {
    matrix[rowId][i] *= modulateValues[i];
  }

}

void
modulateColumns(Vec<Vec<double> >& matrix,
		const Vec<double>& modulateValues) {
  for (Vec<Vec<double> >::size_type i = 0; i < matrix[0].size(); ++i) {
    modulateColumn(matrix, modulateValues, i);
  }
}

void
modulateRows(Vec<Vec<double> >& matrix,
		const Vec<double>& modulateValues) {
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    modulateRow(matrix, modulateValues, i);
  }
}

/** Multiplies all row and columns with vector entries */
void
modulateMatrix(Vec<Vec<double> >& matrix,
	       const Vec<double>& modulateValues) {
  modulateColumns(matrix, modulateValues);
  modulateRows(matrix, modulateValues);
}

unsigned int
centralPoint(const Vec<Vector3D>& choose, const Vec<Vector3D>& data,
	     double& radius)
{
  double bestDist = -1.0;
  unsigned int bestIndex = 0;
  for (unsigned int i = 0; i < choose.size(); ++i) {
    unsigned int largestGuy = findLargestDistanceIndex(choose[i], data);
    ASSERT(largestGuy < data.size());
    double dist = vecDistance(choose[i], data[largestGuy]);
    if ((bestDist < 0) || (dist < bestDist)) {
      bestDist = dist;
      bestIndex = i;
    }
  }
  radius = bestDist;
  return bestIndex;
}

/** return point of m, which are closer than limitDist 
    to subset sub, but do not belong to sub 
*/
Vec<unsigned int>
getClosePoints(const Vec<unsigned int>& sub,
	       const Vec<Vector3D>& m,
	       double limitDist)
{
  Vec<unsigned int> result;
  for (unsigned int i = 0; i < sub.size(); ++i) {
    ASSERT(sub[i] < m.size());
    for (unsigned int j = 0; j < m.size(); ++j) {
      if (sub[i] == j) {
	continue;
      }
      if (vecDistance(m[sub[i]], m[j]) <= limitDist) {
	// check if not member of sub:
	bool found = false;
	for (unsigned int k = 0; k < sub.size(); ++k) {
	  if (j == sub[k]) {
	    found = true;
	    break;
	  }
	}
	if (!found) {
	  result.push_back(j);
	}
      }
    }    
  }
  result = uniqueSet(result);
  POSTCOND(result.size() <= m.size());
  return result;
}

/** return point of m, which are closer than limitDist 
    to subset sub, but do not belong to sub 
*/
Vec<unsigned int>
getClosePoints(const Vec<Vector3D>& m1, const Vec<Vector3D>& m2,
	       double limitDist)
{
  Vec<unsigned int> result;
  for (unsigned int i = 0; i < m1.size(); ++i) {
    for (unsigned int j = 0; j < m2.size(); ++j) {
      if (vecDistance(m1[i], m2[j]) <= limitDist) {
	result.push_back(j);
      }
    }    
  }
  return result;
}

/** return true, if matrix is symmetrix */
bool
isSymmetric(const Vec<Vec<double> >& m)
{
  unsigned int mdim = m.size();
  for (unsigned int i = 0; i < m.size(); ++i) {
    if (m[i].size() != mdim) {
      return false;
    }
    for (unsigned int j = 0; j < i; ++j) {
      if (fabs(m[i][j] - m[j][i]) > 0.1) {
	return false;
      }
    }
  }
  return true;
}

/** symmetrizes matrix */
void
symmetrize(Vec<Vec<double> >& m)
{
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      m[i][j] = 0.5 * (m[i][j] + m[j][i]);
      m[j][i] = m[i][j];
    }
  }
}

/** see Goldstein, Classical Mechanics Ch. "The inertia tensor and the
    moment of inertia */
Matrix3D
computeInertiaTensor(const Vec<Vector3D>& v,
		     const Vec<double> weights) 
{
  PRECOND(weights.size() == v.size());
  Matrix3D m;
  Matrix3D unitMatrix(1.0,1.0,1.0);
  ASSERT(isSymmetric(translateMatrix3DToVec(unitMatrix)));
  for (unsigned int i = 0; i < v.size(); ++i) {
    m += ( (unitMatrix * (v[i]*v[i])) -matrixProduct(v[i],v[i]) ) * weights[i];
    ASSERT(isSymmetric(translateMatrix3DToVec(m)));
  }
  POSTCOND( isReasonable(translateMatrix3DToVec(m))
	 && isSymmetric(translateMatrix3DToVec(m)));
  return m;
}


Vec<Vec<double> >
translateMatrix3DToVec(const Matrix3D& m)
{
  Vec<double> row(3,0.0);
  Vec<Vec<double> > matrix(3,row);
  matrix[0][0] = m.xx();
  matrix[0][1] = m.xy();
  matrix[0][2] = m.xz();
  matrix[1][0] = m.yx();
  matrix[1][1] = m.yy();
  matrix[1][2] = m.yz();
  matrix[2][0] = m.zx();
  matrix[2][1] = m.zy();
  matrix[2][2] = m.zz();
  return matrix;
}

/** see Goldstein, Classical Mechanics Ch. "The inertia tensor and the
    moment of inertia */
/*
void
computeGyrationGeometry(const Vec<Vector3D>& v,
			const Vec<double> weights,
			Vec<double>& eigenValuesOrig,
			Vec<Vec<double> >& eigenVectorsOrig)
{
  PRECOND(v.size() == weights.size());
  Matrix3D inertiaTensor = computeInertiaTensor(v,weights);
  Vec<double> eigenValues(3, 0.0);
  Vec<Vec<double> > eigenVectors(3, eigenValues);
  jacobi(translateMatrix3DToVec(inertiaTensor), eigenValues,
	 eigenVectors);
  eigenValuesOrig = eigenValues;
  eigenVectorsOrig = eigenVectors;
}
*/

/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
*/
void
transformToZScores(Vec<double>& v, double minSigma)
{
  double mean = vecMean(v);
  double sigma = vecVariance(v);
  if (sigma <= 0.0) {
    for (unsigned int i = 0; i < v.size(); ++i) {
      v[i] = 0.0;
    }
    return; // all components are identical
  }
  sigma = max(minSigma,sqrt(sigma));
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] = (v[i] - mean) / sigma;
  }
}

/** multiply all entries of matrix with this factor */
void
matrixScalarMul(Vec<Vec<double> >& matrix, double val) 
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      matrix[i][j] *= val;
    }
  }
}

/** b = m a */
void
matrixVectorMul(const Vec<Vec<double> >& m,
		const Vec<double>& a,
		Vec<double>& b)
{
  PRECOND((a.size() == b.size()));
  for (unsigned int i = 0; i < m.size(); ++i) {
    b[i] = 0.0;
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      b[i] += m[i][j]*a[j];
    }
  }
}

/** b = a(T) b */
void
vectorMatrixMul(const Vec<Vec<double> >& m,
		const Vec<double>& a,
		Vec<double>& b)
{
  PRECOND((m.size() > 0) 
	  && (a.size() == m.size()) && (b.size() == m[0].size()));
  // number of columns
  for (unsigned int i = 0; i < m[0].size(); ++i) {
      b[i] = 0.0;
      // loop over rows
      for (unsigned int j = 0; j < m.size(); ++j) {
	b[i] += m[j][i]*a[j];
      }
  }
}

void 
matrixMul(const Vec<Vec<double> >& a,
	  const Vec<Vec<double> >& b,
		Vec<Vec<double> >& c)
{
  PRECOND( (a.size() > 0) && (b.size() > 0) && (c.size() > 0)
	   && (a.size() == c.size()) 
	   && (b[0].size() == c[0].size()) 
	   && (a[0].size() == b.size()));
  for (unsigned int i = 0; i < c.size(); ++i) {
    for (unsigned int j = 0; j < c[i].size(); ++j) {
      c[i][j] = 0.0;
      for (unsigned int k = 0; k < a[i].size(); ++k) {
	c[i][j] += a[i][k] * b[k][j];
      }
    }
  }
}

/** simple element-wise (non-standard!) multiplication */
void 
matrixElementMul(const Vec<Vec<double> >& a,
		 const Vec<Vec<double> >& b,
		 Vec<Vec<double> >& c)
{
  PRECOND( (a.size() > 0) && (b.size() > 0) && (c.size() > 0)
	   && (a.size() == c.size()) 
	   && (b[0].size() == c[0].size()) 
	   && (a[0].size() == b.size()));
  for (unsigned int i = 0; i < c.size(); ++i) {
    for (unsigned int j = 0; j < c[i].size(); ++j) {
      c[i][j] = a[i][j] * b[i][j];
    }
  }
}

/** transpose matrix */
void
matrixTranspose(const Vec<Vec<double> >& m1,
		Vec<Vec<double> >& m2 )
{
  unsigned int m = m1.size(); 
  ERROR_IF(m == 0, "Matrix with zero elements encountered.");
  unsigned int n = m1[0].size(); 
  Vec<double> newRow(m,0.0);
  Vec<Vec<double> > helpField(n, newRow);
  m2 = helpField;
  for (unsigned int i = 0; i < m1.size(); ++i) {
    for (unsigned int j = 0; j < m1[i].size(); ++j) {
      ASSERT((j < m2.size())&& (i < m2[j].size()));
      m2[j][i] = m1[i][j];
    }
  }
}

/** transpose matrix itself. Works only for square matrices.  */
void
matrixTranspose(Vec<Vec<double> >& m)
{
  PRECOND(m.size() == m[0].size()); // only for square matrices!
  cout << "inside matrixTranspose!" << endl;
  double help = 0.0;
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      ASSERT((j < m.size())&& (i < m[j].size()));
      help = m[j][i]; // swap content
      m[j][i] = m[i][j];
      m[i][j] = help;
    }
  }
}

/** set matrix elements to one, if i=j, zero otherwise
 */
void
unitMatrix(Vec<Vec<double> >& matrix);



/** compute matrix inverse. See numerical recepies */
void
matrixInverse(const Vec<Vec<double> >& m, Vec<Vec<double> >& y);

/** compute matrix inverse and determinante. See numerical recepies */
void
matrixInverseAndDet(const Vec<Vec<double> >& m, Vec<Vec<double> >& y,
		    double& d);

/** compute matrix inverse and log of determinante . See numerical recepies */
void
matrixInverseAndLogDet(const Vec<Vec<double> >& mOrig,
		    Vec<Vec<double> >& y, double& logDet);

/** reverse matrix triangular decomposition (stored in lum) as
    computed by ludcmp
*/
void
matrixReverseDecomposition(const Vec<Vec<double> >& lum, 
			   Vec<Vec<double> >& m);

/** return bin of histogram */
int getBin(double val, double min, double delta)
{
  PRECOND(delta > 0.0);
  return static_cast<int>((val - min) / delta);
}

/** return historgramm: vector of bins filled with counts */
Vec<unsigned int> computeHistogram(const Vec<double>& data,
			      double min,
			      double delta, 
			      unsigned int numBins)
{
  PRECOND(delta > 0.0);
  Vec<unsigned int> result(numBins, 0);
  int bin = 0; 
  for (unsigned int i = 0; i < data.size(); ++i) {
    bin = static_cast<int>((data[i] - min) / delta);
    if ((bin >= 0) && (bin < static_cast<int>(result.size()) ) ) {
      ++result[bin];
    }
  }
  return result;
}

/** return historgramm: vector of bins filled with y values belonging
 to this bin */
Vec<Vec<double> > computeScatterHistogram(const Vec<double>& datax,
					  const Vec<double>& datay,
					  double min,
					  double delta, 
					  unsigned int numBins)
{
  PRECOND(delta > 0.0);
  PRECOND(datax.size() == datay.size());
  Vec<Vec<double> > result(numBins);
//    cout << "datax: " << datax << endl;
//    cout << "datay: " << datay << endl;
  int bin = 0; 
  for (unsigned int i = 0; i < datax.size(); ++i) {
    bin = static_cast<int>((datax[i] - min) / delta);
    if ((bin >= 0) && (bin < static_cast<int>(result.size()) ) ) {
      result[bin].push_back(datay[i]);
    }
  }
  return result;
}

/** return historgramm: vector of bins filled with counts */
Vec<double> computeDoubleHistogram(const Vec<double>& data,
				   double min,
				   double delta, 
				   unsigned int numBins)
{
  PRECOND(delta > 0.0);
  Vec<double> result(numBins, 0.0);
  int bin = 0; 
  for (unsigned int i = 0; i < data.size(); ++i) {
    bin = static_cast<int>((data[i] - min) / delta);
    if ((bin >= 0) && (bin < static_cast<int>(result.size()) ) ) {
      result[bin] = result[bin] + 1.0;
    }
  }
  return result;
}

/** return historgramm: vector of bins filled with counts */
Vec<double> computeFrequency(const Vec<double>& data,
			     double min,
			     double delta, 
			     unsigned int numBins)
{
  PRECOND(delta > 0.0);
  Vec<double> result(numBins, 0.0);
  int bin = 0;
  double count = 0;
  for (unsigned int i = 0; i < data.size(); ++i) {
    bin = static_cast<int>((data[i] - min) / delta);
    if ((bin >= 0) && (bin < static_cast<int>(result.size()) ) ) {
      ++result[bin];
      ++count;
    }
  }
  if (count > 0.0) {
    for (unsigned int i = 0; i < result.size(); ++i) {
      result[i] = result[i] / count;
    }
  }
  return result;
}


/** compute uniqe set from original set */
bool
containsDuplicates(const Vec<unsigned int>& vOrig)
{
  if (vOrig.size() == 0) {
    return false;
  }
  Vec<unsigned int> v = vOrig; // safety copy
  sort(v.begin(), v.end());
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] == v[i-1]) {
      return true;
    }
  }
  return false;
}


/* compute common subset of two sets.
   not unique list, if input lists are not unique! */
Vec<unsigned int>
commonSubset(const Vec<unsigned int>& lOrig1,
	     const Vec<unsigned int>& lOrig2 )
{
  Vec<unsigned int> l1 = lOrig1;
  Vec<unsigned int> l2 = lOrig2;
  Vec<unsigned int> result;
  sort(l1.begin(), l1.end());
  sort(l2.begin(), l2.end());
  for (unsigned int i = 0; i < l1.size(); ++i) {
    for (unsigned int j = 0; j < l2.size(); ++j) {
      if (l1[i] == l2[j]) {
	result.push_back(l1[i]);
	break;
      }
    }
  }
  return result;
}

/* compute unique union of two sets */
Vec<unsigned int>
uniqueUnion(const Vec<unsigned int>& l1,
	    const Vec<unsigned int>& l2)
{
  Vec<unsigned int> result = l1;
  for (unsigned int i = 0; i < l2.size(); ++i) {
    bool found = false;
    for (unsigned int j = 0; j < result.size(); ++j) {
      if (result[j] == l2[i]) {
	found = true;
	break;
      }
    }
    if (!found) {
      unsigned int tmp = l2[i];
      result.push_back(tmp);
    }
  }
  return result;
}

/** vector with probabilities of different entities.
    values must be between 0 and 1
*/
double
conservation(const Vec<double>& v)
{
  double sum = elementSum(v);
  if (sum <= 0.0) {
    return 0.0;
  }
  double x = *(max_element(v.begin(), v.end()));
  return x / sum;
}

/* return 3D matrix from Euler angles
   rotation around x-axis: w
   rotation around y-axis: a
   rotation around z-axis: k
   taken from: Ronald Azuma 1991 (collinear.pdf). Check again from other source!!!
*/
Matrix3D
computeEulerRotationMatrix(double w, double a, double k)
{
  Matrix3D m(
  cos(k)*cos(a),-sin(k)*cos(w)+cos(k)*sin(a)*sin(w),sin(k)*sin(w)+cos(w)*cos(k)*sin(a),
  sin(k)*cos(a),cos(k)*cos(w)+sin(w)*sin(k)*sin(a),-sin(w)*cos(k)+cos(w)*sin(k)*sin(a),
  -sin(a),sin(w)*cos(a),cos(w)*cos(a) );
  return m;
}

/* find minimum x, y, z coordinates */
Vector3D
findMinCoordinates(const Vec<Vector3D>& vv)
{
  PRECOND(vv.size() > 0);
  double xmin = vv[0].x();
  double ymin = vv[0].y();
  double zmin = vv[0].z();
  for (unsigned int i = 1; i < vv.size(); ++i) {
    if (vv[i].x() < xmin) {
      xmin = vv[i].x();
    }
    if (vv[i].y() < ymin) {
      ymin = vv[i].y();
    }
    if (vv[i].z() < zmin) {
      zmin = vv[i].z();
    }
  }
  Vector3D result(xmin, ymin, zmin);
  return result;
}

/* find maximum x, y, z coordinates */
Vector3D
findMaxCoordinates(const Vec<Vector3D>& vv)
{
  PRECOND(vv.size());
  double xmax = vv[0].x();
  double ymax = vv[0].y();
  double zmax = vv[0].z();
  for (unsigned int i = 1; i < vv.size(); ++i) {
    if (vv[i].x() > xmax) {
      xmax = vv[i].x();
    }
    if (vv[i].y() > ymax) {
      ymax = vv[i].y();
    }
    if (vv[i].z() > zmax) {
      zmax = vv[i].z();
    }
  }
  Vector3D result(xmax, ymax, zmax);
  return result;
}

unsigned int
findMaxDistanceIndex(const Vector3D& pos, const Vec<Vector3D>& v) {
  PRECOND(v.size() > 0);
  unsigned int bestIndex = 0;
  double dBest = vecDistanceSquare(pos, v[0]);
  for (unsigned int i = 1; i < v.size(); ++i) {
    double d = vecDistanceSquare(pos, v[i]);
    if (d > dBest) {
      dBest = d;
      bestIndex = i;
    }
  }
  return bestIndex;
}

unsigned int
findMinDistanceIndex(const Vector3D& pos,
		     const Vec<Vector3D>& v) {
  PRECOND(v.size() > 0);
  unsigned int bestIndex = 0;
  double dBest = vecDistance(pos, v[0]);
  for (unsigned int i = 1; i < v.size(); ++i) {
    double d = vecDistance(pos, v[i]);
    if (d < dBest) {
      dBest = d;
      bestIndex = i;
    }
  }
  POSTCOND(bestIndex < v.size());
  return bestIndex;
}

/* return distance in indices of pair of vectors with shortest distance */
double
findSmallestDistance(const Vec<Vector3D>& v1,
		    const Vec<Vector3D>& v2,
		    unsigned int& n1,
		    unsigned int& n2)
{
  PRECOND((v1.size() > 0) && (v2.size() > 0));
  unsigned int v1Size = v1.size();
  unsigned int v2Size = v2.size();
  double d;
  double dBest = vecDistanceSquare(v1[0], v2[0]);
  n1 = 0;
  n2 = 0;
  for (unsigned int i = 0; i < v1Size; ++i) {
    for (unsigned int j = 0; j < v2Size; ++j) {
      d = vecDistanceSquare(v1[i], v2[j]);
      if (d < dBest) {
	dBest = d;
	n1 = i;
	n2 = j;
      }
    }
  }
  return sqrt(dBest);
}

/** return coordinates which are in the middle of
    min and max values for each dimension */
Vector3D
computeCenterOfCoordinates(const Vec<Vector3D>& v)
{
  return ( findMaxCoordinates(v)
	 + findMinCoordinates(v) ) * 0.5;
}

/** return center of mass. Do not take weights into account! */
Vector3D
computeCenterOfMass(const Vec<Vector3D>& v)
{
  PRECOND(v.size() > 0);
  Vector3D sum(0.0,0.0,0.0);
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += v[i];
  }
  sum *= (1.0/static_cast<double>(v.size()));
  return sum;
}

/** return center of mass. */
Vector3D
computeCenterOfMass(const Vec<Vector3D>& v,
		    const Vec<double> weights)
{
  PRECOND((v.size() > 0) && (v.size() == weights.size()));
  Vector3D sum(0.0,0.0,0.0);
  double weightSum = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += (v[i] * weights[i]);
    weightSum += weights[i];
  }
  ASSERT(weightSum > 0.0);
  return sum * (1.0/weightSum);
}

/* return radius of gyration of set of vectors */
double
computeRadiusOfGyration(const Vec<Vector3D>& v)
{
  double result = 0;
  Vector3D center = computeCenterOfMass(v);
  for (unsigned int i = 0; i < v.size(); ++i) {
    double r = vecDistance(v[i],center);
    result += r * r;
  }
  result = sqrt(result);
  return result;
}

/** compute root mean square deviation of two sets of coordinate vectors */
double
rms(const Vec<Vector3D>& v1, const Vec<Vector3D>& v2)
{
  PRECOND((v1.size() > 0) && (v1.size() == v2.size()));
  double sum = 0.0;
  for (unsigned int i = 0; i < v1.size(); ++i) {
    sum += vecDistanceSquare(v1[i],v2[i]);
  }
  return sqrt(sum/v1.size());
}

/* generate set of n points with radius r around origin vector on a ring defined by theta */
Vec<Vector3D>
generateRingPoints(unsigned int n, double r, double theta, const Vector3D& origin)
{
  if (n == 0) {
    return Vec<Vector3D>();
  }
  Vec<Vector3D> ray(n);
  double phi;
  double phiDelta = (PI * 2) / static_cast<double>(n);
  for (unsigned int i = 0; i < n; ++i) {
    phi = i * phiDelta;
    ray[i].x(cos(phi) * sin(theta));
    ray[i].y(sin(phi)*sin(theta));
    ray[i].z(cos(theta));
  }
  for (unsigned int i = 0; i < ray.size(); ++i) {
    ray[i] = (ray[i] * r) + origin;
  }
  POSTCOND(ray.size() == n);
  return ray;
}

/* generate set of n points with radius r around origin vector on a ring defined by theta */
Vec<Vector3D>
generateRingPoints(unsigned int n, double r, double theta, const Vector3D& origin, 
		   Vector3D direction)
{
  if (n == 0) {
    return Vec<Vector3D>();
  }
  Vector3D zeroVec(0.0, 0.0, 0.0);
  Vector3D zAxis(0.0, 0.0, 1.0);
  Vec<Vector3D> ray = generateRingPoints(n, r, theta, zeroVec);
  if (direction.length() > 0.0) {
    direction.normalize();
    if (direction != zAxis) {
      double ang = angle(direction, zAxis);
      Vector3D rotAxis = cross(zAxis, direction);
      rotAxis.normalize();
      for (unsigned int i = 0; i < ray.size(); ++i) {
	ray[i] = Matrix3D::rotate(ray[i], rotAxis, ang);
      }
    }
  }
  for (unsigned int i = 0; i < ray.size(); ++i) {
    ray[i] = ray[i] + origin;
  }
  POSTCOND(ray.size() == n);
  return ray;
}


/* generate set of n points with radius r around origin vector */
Vec<Vector3D>
generateSpherePoints(unsigned int n, double r, const Vector3D& origin)
{
  if (n == 0) {
    return Vec<Vector3D>();
  }
  Vec<Vector3D> ray(n);
  double h,theta,phi;
  double oldPhi = 0.0;
  for (unsigned int i = 1; i < n-1; ++i)
    {
      h = -1.0 + 2.0 * i / (n-1.0);
      ASSERT((h>=-1.0)&&(h<=1.0));
      theta = acos(h);
      phi = (oldPhi + 3.6/sqrt(n*(1-h*h)));
      ray[i].x(cos(phi) * sin(theta));
      ray[i].y(sin(phi)*sin(theta));
      ray[i].z(cos(theta));
      oldPhi = phi;
    }
  ray[0] = Vector3D(0.0,0.0,-1.0);
  ray[n-1] = Vector3D(0.0,0.0,1.0);
  for (unsigned int i = 0; i < ray.size(); ++i) {
    ray[i] = (ray[i] * r) + origin;
  }
  return ray;
}

/* generate surface points of point set */
void
generateSurfacePoints(Vec<Vector3D>& mol, const Vec<Vector3D>& molProt, 
		      double r, unsigned int tessel, bool rawMode)
{
  mol.clear();
  Vec<Vec<Vector3D> > allv(molProt.size());
  for (unsigned int i = 0; i < molProt.size(); ++i) {
    Vec<Vector3D> sset = generateSpherePoints(tessel, r, molProt[i]);
    // add points to set of all generated points:
    allv[i] = sset;
  }
  // only keep those, that are not closer to point from other set:
  for (unsigned int i = 0; i < allv.size(); ++i) {
    for (unsigned int j = 0; j < allv[i].size(); ++j) {
      bool found = false;
      for (unsigned int k = 0; k < molProt.size(); ++k) {
	if (k == i) {
	  continue; // skip if same sphere
	}
	double d = vecDistance(allv[i][j], molProt[k]);
	if (d < r) {
	  found = true;
	  break;
	}
      }
      if (rawMode || (!found)) {
	mol.push_back(allv[i][j]);
      }
    }
  }
}

/* generate surface points of point set
   result: vector of size molProt, each subvector
   containing surface vectors belonging to that atom */
Vec<Vec<Vector3D> >
generateSurfacePoints(const Vec<Vector3D>& molProt, 
		      double r, unsigned int tessel, bool rawMode)
{
  PRECOND(molProt.size() > 0);
  Vec<Vec<Vector3D> > allv(molProt.size()), mol(molProt.size()) ;
  for (unsigned int i = 0; i < molProt.size(); ++i) {
    Vec<Vector3D> sset = generateSpherePoints(tessel, r, molProt[i]);
    // add points to set of all generated points:
    allv[i] = sset;
  }
  // only keep those, that are not closer to point from other set:
  for (unsigned int i = 0; i < allv.size(); ++i) {
    for (unsigned int j = 0; j < allv[i].size(); ++j) {
      bool found = false;
      for (unsigned int k = 0; k < molProt.size(); ++k) {
	if (k == i) {
	  continue; // skip if same sphere
	}
	double d = vecDistance(allv[i][j], molProt[k]);
	if (d < r) {
	  found = true;
	  break;
	}
      }
      if (rawMode || (!found)) {
	mol[i].push_back(allv[i][j]);
      }
    }
  }
  return mol;
}


/* generate surface points of point set */
/*
void
generateSurfacePoints(Vec<Vector3D>& mol,
     Vec<int>& newIdVec, const Vec<Vector3D>& molProt, 
     const Vec<int>& idVec, double r, unsigned int tessel, bool rawMode)
{
  PRECOND(molProt.size() == idVec.size());
  mol.clear();
  newIdVec.clear();
  Vec<Vec<Vector3D> > allv(molProt.size());
  for (unsigned int i = 0; i < molProt.size(); ++i) {
    Vec<Vector3D> sset = generateSpherePoints(tessel, r, molProt[i]);
    // add points to set of all generated points:
    allv[i] = sset;
  }
  // only keep those, that are not closer to point from other set:
  for (unsigned int i = 0; i < allv.size(); ++i) {
    for (unsigned int j = 0; j < allv[i].size(); ++j) {
      bool found = false;
      for (unsigned int k = 0; k < molProt.size(); ++k) {
	if (k == i) {
	  continue; // skip if same sphere
	}
	double d = vecDistance(allv[i][j], molProt[k]);
	if (d < r) {
	  found = true;
	  break;
	}
      }
      if (rawMode || (!found)) {
	mol.push_back(allv[i][j]);
	newIdVec.push_back(idVec[i]);	
      }
    }
  }
}
*/

/* generate surface points of point set */
void
generateSurfacePoints(Vec<Vector3D>& mol,
     Vec<int>& newIdVec, Vec<Vector3D>& directions,
     const Vec<Vector3D>& molProt, 
     const Vec<int>& idVec,
     double r, unsigned int tessel, bool rawMode)
{
  mol.clear();
  directions.clear();
  newIdVec.clear();
  Vec<Vec<Vector3D> > allv(molProt.size());
  for (unsigned int i = 0; i < molProt.size(); ++i) {
    Vec<Vector3D> sset = generateSpherePoints(tessel, r, molProt[i]);
    // add points to set of all generated points:
    allv[i] = sset;
  }
  // only keep those, that are not closer to point from other set:
  for (unsigned int i = 0; i < allv.size(); ++i) {
    for (unsigned int j = 0; j < allv[i].size(); ++j) {
      bool found = false;
      for (unsigned int k = 0; k < molProt.size(); ++k) {
	if (k == i) {
	  continue; // skip if same sphere
	}
	double d = vecDistance(allv[i][j], molProt[k]);
	if (d < r) {
	  found = true;
	  break;
	}
      }
      if (rawMode || (!found)) {
	mol.push_back(allv[i][j]);
	if (i < idVec.size()) {
	  newIdVec.push_back(idVec[i]);	
	}
	else {
	  newIdVec.push_back(i);
	}
	// vector from surf point to its nearest atom center is surface normal
	directions.push_back((allv[i][j] - molProt[i]));
      }
    }
  }
  POSTCOND((mol.size() == newIdVec.size())
	   && (mol.size() == directions.size()));
}

/* generate surface points of point set */
void
generateSurfacePointsAccessibleSurface(Vec<Vector3D>& mol,
     Vec<int>& newIdVec, Vec<Vector3D>& directions,
     Vec<double>& accessSurface,				       
     const Vec<Vector3D>& molProt, 
     const Vec<int>& idVec,
     double r, unsigned int tessel, bool rawMode)
{
  mol.clear();
  directions.clear();
  newIdVec.clear();
  accessSurface.clear();
  accessSurface = Vec<double>(molProt.size(), 0.0);
  Vec<Vec<Vector3D> > allv(molProt.size());
  for (unsigned int i = 0; i < molProt.size(); ++i) {
    Vec<Vector3D> sset = generateSpherePoints(tessel, r, molProt[i]);
    // add points to set of all generated points:
    allv[i] = sset;
  }
  // only keep those, that are not closer to point from other set:
  for (unsigned int i = 0; i < allv.size(); ++i) {
    for (unsigned int j = 0; j < allv[i].size(); ++j) {
      bool found = false;
      for (unsigned int k = 0; k < molProt.size(); ++k) {
	if (k == i) {
	  continue; // skip if same sphere
	}
	double d = vecDistance(allv[i][j], molProt[k]);
	if (d < r) {
	  found = true;
	  break;
	}
      }
      if (rawMode || (!found)) {
	mol.push_back(allv[i][j]);
	if (i < idVec.size()) {
	  newIdVec.push_back(idVec[i]);	
	}
	else {
	  newIdVec.push_back(i);
	}
	accessSurface[i] = accessSurface[i] + 1.0;
	// vector from surf point to its nearest atom center is surface normal
	directions.push_back((allv[i][j] - molProt[i]));
      }
    }
  }
  double totAtomSurface = 4.0 * PI * r * r;
  for (unsigned int i = 0; i < accessSurface.size(); ++i) {
    accessSurface[i] = accessSurface[i] / static_cast<double>(tessel) 
      * totAtomSurface;
    ASSERT((accessSurface[i] <= totAtomSurface));
  }
  POSTCOND((mol.size() == newIdVec.size())
	   && (mol.size() == directions.size())
	   && (accessSurface.size() == molProt.size()));
}

/* generate compute accessible surface for point set */
void
computeAccessibleSurface(const Vec<Vector3D>& molProt, 
			  Vec<double>& accessSurface,				       
			  double r, unsigned int tessel)
{
  Vec<Vector3D> surfaceMol;
  Vec<Vector3D> directions;
  Vec<int> newIdVec;
  Vec<int> idVec(molProt.size(), 0);
  generateSurfacePointsAccessibleSurface(surfaceMol,
     newIdVec, directions,
     accessSurface, molProt, idVec, r, tessel, false);
  POSTCOND(molProt.size() == accessSurface.size());
}

/* generate simplex of n+1 dimensions */
Vec<Vec<double> > generateSimplex(const Vec<double>& v, double step)
{
  Vec<Vec<double> > result(v.size() +1, v);
  // first vector is start vector, others are one step away
  for (unsigned int i = 1; i < result.size(); ++i) {
    result[i][i-1] += step;
  }
  POSTCOND(result.size() == v.size() + 1);
  return result;
}

/* generate simplex of n+1 dimensions */
Vec<Vec<double> > generateSimplex(const Vec<double>& v, const Vec<double>& stepSizes)
{
  PRECOND(stepSizes.size() == v.size());
  Vec<Vec<double> > result(v.size() +1, v);
  // first vector is start vector, others are one step away
  for (unsigned int i = 1; i < result.size(); ++i) {
    result[i][i-1] += stepSizes[i-1];
  }
  POSTCOND(result.size() == v.size() + 1);
  return result;
}


/*
template <class T>
Vec<unsigned int>
annotatedSort(Vec<T>& v)
{
  bool sorted = false;
  Vec<unsigned int> annotation(v.size(), 0);
  for (unsigned int i = 0; i < annotation.size(); ++i) {
    annotation[i] = i;
  }
  while (!sorted) {
    sorted = true;
    for (unsigned int i = 1; i < v.size(); ++i) {
      if (v[i] < v[i-1]) {
	swap(v[i-i], v[i]);
	swap(annotation[i-1], annotation[i]);
	sorted = false;
      }
    }
  }
  return annotation;
}
*/

// get k nearest neighours of query vector closer than maxDist 
Vec<unsigned int>
getKNearestNeighbours(const Vec<Vector3D>& v, 
		      unsigned int query,
		      unsigned int maxN,
		      double maxDist)
{
  PRECOND(query < v.size());
  Vec<unsigned int> bestIndices;
  Vec<double> distances;  
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (i == query) {
      continue;
    }
    double d = vecDistance(v[i], v[query]);      
    if (d <= maxDist) {
      distances.push_back(d);
      bestIndices.push_back(i);
    }
  }
  // sort:
  Vec<unsigned int> formerOrder = documentedSort(distances);
  unsigned int maxI = formerOrder.size();
  if (maxI > maxN) {
    maxI = maxN;
  }
  Vec<unsigned int> result;
  for (unsigned int i = 0; i < maxI; ++i) {
    result.push_back(bestIndices[formerOrder[i]]);
  }
  return result;
}

// get k nearest neighours of query vector closer than maxDist 
Vec<Vec<unsigned int> >
getKNearestNeighbours(const Vec<Vector3D>& v, 
		      unsigned int maxN,
		      double maxDist)
{
  Vec<Vec<unsigned int> > result(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    result[i] = getKNearestNeighbours(v,i,maxN,maxDist);
  }
  return result;
}

/* return non-linear weighting of value x */
double
nonLinearWeighting(double x, double maxY, double x0, double scale, double skew)
{
  PRECOND(scale > 0.0);
  return ( 0.5 * maxY * (tanh(scale*(x-x0)) + 1) ) + skew;
}

/* return non-linear weighting of value x */
double
nonLinearWeightingDouble(double x, double maxY1, double x01, double scale1, 
			 double maxY2, double x02, double scale2)
{
  PRECOND((scale1 > 0.0) && (scale2 > 0.0));
  return nonLinearWeighting(x, fabs(maxY1), x01, scale1, - fabs(maxY1)) 
       + nonLinearWeighting(x, fabs(maxY2), x02, scale2);
}

/* expects N SORTED x-values and N+1 y values defining a step function. */
double
stepFunction(double x, const Vec<double>& xVec, const Vec<double>& yVec)
{
  PRECOND(xVec.size() + 1 == yVec.size());
  double y = 0.0;
  if ((xVec.size() == 0) || (x < xVec[0])) {
    y = yVec[0];
  }
  else if (x > xVec[xVec.size()-1]) {
    y = yVec[xVec.size()];
  }
  else  {
    for (unsigned int i = 0; i < xVec.size(); ++i) {
      if (x <= xVec[i]) {
	y = yVec[i];
	break;
      }
    }
  }
  return y;
}

/* compute true/false positives/negatives of scores, which if greater
   than cutOff, are predicted to belong to class classNum */
double
computeMathews(const Vec<double>& scores, 
		const Vec<unsigned int>& trueClass, 
		unsigned int classNum, 
		double cutOff,
		unsigned int& truePos, 
		unsigned int& falsePos, 
		unsigned int& trueNeg, 
		unsigned int& falseNeg)
{
  PRECOND(scores.size() == trueClass.size());
  for (unsigned int i = 0; i < scores.size(); ++i) {
    if (scores[i] >= cutOff) {
      if (trueClass[i] == classNum) {
	++truePos;
      }
      else {
	++falsePos;
      }
    }
    else {
      if (trueClass[i] == classNum) {
	++falseNeg;
      }
      else {
	++trueNeg;
      }
    }
  }
  ASSERT(truePos + falsePos + falseNeg + trueNeg == scores.size());
  return computeMathews(truePos, falsePos, trueNeg, falseNeg);
}


/* compute true/false positives/negatives of scores, which if greater
   than cutOff, are predicted to belong to class classNum */
double
computeMathews(const Vec<Vec<double> > & scores, 
	       const Vec<Vec<unsigned int> > & trueClass, 
	       unsigned int classNum, 
	       double cutOff,
	       unsigned int& truePos, 
	       unsigned int& falsePos, 
	       unsigned int& trueNeg, 
	       unsigned int& falseNeg)
{
  PRECOND(scores.size() == trueClass.size());
  truePos = 0;
  falsePos = 0;
  trueNeg = 0;
  falseNeg = 0;
  for (unsigned int i = 0; i < scores.size(); ++i) {
    for (unsigned int j = 0; j < scores[i].size(); ++j) {
      if (scores[i][j] >= cutOff) {
	if (trueClass[i][j] == classNum) {
	  ++truePos;
	}
	else {
	  ++falsePos;
	}
      }
      else {
	if (trueClass[i][j] == classNum) {
	  ++falseNeg;
	}
	else {
	  ++trueNeg;
	}
      }
    }
  }
  ASSERT(truePos + falsePos + falseNeg + trueNeg == (scores.size()*scores[0].size()));
  return computeMathews(truePos, falsePos, trueNeg, falseNeg);
}


/* compute true/false positives/negatives of scores, which if greater
   than cutOff, are predicted to belong to class classNum */
double
computeMathews(	unsigned int truePos, unsigned int falsePos, 
		unsigned int trueNeg, unsigned int falseNeg)
{
  double head = static_cast<double>((truePos * trueNeg)) 
              - static_cast<double>((falsePos * falseNeg));
  double denom = static_cast<double>(trueNeg + falseNeg) 
               * static_cast<double>(trueNeg + falsePos)
               * static_cast<double>(truePos + falseNeg) 
               * static_cast<double>(truePos + falsePos);
  if (denom <= 0.0) {
    return -1000.0;
  }
  return head / sqrt(denom);
}

/* after computation the values between the two indices are
   interpolated */
void
vecInterpolate(Vec<double>& v, unsigned int startidx, unsigned int endidx)
{
  PRECOND((endidx > startidx) && (endidx < v.size()));
  unsigned int deltax = endidx - startidx;
  double dy = (v[endidx] - v[startidx]) / deltax;
  for (unsigned int i = startidx + 1; i < endidx; ++i) {
    v[i] = ( (i - startidx) * dy ) + v[startidx];
  }
}

/* fill in missing values if borders by defined values.
   undefined is every v[i] with fabs(v[i] - empty) <= diff.
*/
void
vecInterpolateMissing(Vec<double>& v, double empty, double diff)
{
  unsigned int i = 0;
  unsigned status = 0;
  unsigned int lastidx = 0;
  while (i < v.size()) {
    bool isdef = (fabs(v[i] - empty) > diff);
    switch (status)
      {
      case 0: // no possible start point defined yet. 
	if (isdef) {
	  status = 1; // possible startpoint
	  lastidx = i;
	}
	break;
      case 1: // possible start point defined, no undefined region found
	if (isdef) {
	  status = 1; // keep status unchanged move starting point
	  lastidx = i;
	}
	else if ( (i + 1) == v.size()) { // close to end
	  for (unsigned int j = lastidx + 1; j < v.size(); ++j) {
	    v[j] = v[lastidx]; // set right side to last seen value
	  }
	}
	else {
	  status = 2; // region suddenly undefined
	}
	break;
      case 2: // region was undefined
	if (isdef) { // suddenly defined again
	  vecInterpolate(v, lastidx, i);
	  status = 1; // possible startpoint
	  lastidx = i;
	}
	else if ( (i + 1) == v.size()) {
	  for (unsigned int j = lastidx + 1; j < v.size(); ++j) {
	    v[j] = v[lastidx]; // set right side to last seen value
	  }
	}
	break;
      default:
	ERROR("Internal error in line 2233!");
      }
    ++i;
  }

}

// compute gaussian at point n for vector v, with relaxation constant d,
// and maximum width nMax. If nMax == 0, ignore nMax
double
gaussValue(int n, const Vec<double>& v, double d, unsigned int nMax)
{
  //  const double INV_SQRT_2PI = 1.0 / sqrt((3.0/2.0) * PI);
  int nv = static_cast<int>(v.size() / 2);
  if ((nMax > 0) && (static_cast<int>(nMax) < nv)) {
    nv = nMax;
  }
  int vsize = v.size();
  double result = 0;
  double dsq = d * d;
  double val = 0;
  for (int i = -nv; i <= nv; ++i) {
    int j = n + i;
    // cout << "j: " << j << endl;
    if (j < 0) {
      val = v[0];
    }
    else if (j >= vsize) {
      val = v[vsize - 1];
    }
    else {
      val = v[j];
    }
    result += exp(- (i * i) / dsq) * val; 
  }
  //   result *= INV_SQRT_2PI; // NO NORMALIZATION ! DONE LATER!
  return result;
}

double
gaussValue(int n, const Vec<double>& v, double d)
{
  //  const double INV_SQRT_2PI = 1.0 / sqrt((3.0/2.0) * PI);
  int nv = static_cast<int>(v.size() / 2);
  int vsize = v.size();
  double result = 0;
  double dsq = d * d;
  double val = 0;
  for (int i = -nv; i <= nv; ++i) {
    int j = n - i;
    if (j < 0) {
      val = v[0];
    }
    else if (j >= vsize) {
      val = v[vsize - 1];
    }
    else {
      val = v[j];
    }
    result += exp(- (i * i) / dsq) * val; 
  }
  //   result *= INV_SQRT_2PI; // NO NORMALIZATION ! DONE LATER!
  return result;
}


Vec<double>
smoothGaussian(const Vec<double>& v, double d)
{
  Vec<double> w(v);
  if (d <= 0.0) {
    return w;
  }
  double sum = elementSum(v);
  for (unsigned int i = 0; i < v.size(); ++i) {
    w[i] = gaussValue(i, v, d);
  }
  double newSum = elementSum(w);
  if (newSum > 0.0) {
    scalarMul(w, sum / newSum); // normalize to old sum
  }
  return w;
}

Vec<double>
smoothGaussian(const Vec<double>& v, double d, unsigned int nMax)
{
  Vec<double> w(v);
  if (d <= 0.0) {
    return w;
  }
  double sum = elementSum(v);
  for (unsigned int i = 0; i < v.size(); ++i) {
    // cout << "Working on " << i << " " << v[i] << endl;
    w[i] = gaussValue(static_cast<int>(i), v, d, nMax );
  }
  double newSum = elementSum(w);
  if (newSum > 0.0) {
    scalarMul(w, sum / newSum); // normalize to old sum
  }
  return w;
}

// smoothing of 3D cube. Can be done as subsequent 1d smoothings
// see: http://www.mrc-cbu.cam.ac.uk/Imaging/smoothing.html
void
smoothGaussian(Vec<Vec<Vec<double> > >& cube, double d, unsigned int nMax)
{
  unsigned int dimX = cube.size();
  unsigned int dimY = cube[0].size();
  unsigned int dimZ = cube[0][0].size();
  Vec<double> xRow(dimX, 0.0);
  Vec<double> yRow(dimY, 0.0);
  Vec<double> zRow(dimZ, 0.0);
  // first smooth in x - direction:  
  for (unsigned int j = 0; j < dimY; ++j) {
    for (unsigned int k = 0; k < dimZ; ++k) {
      for (unsigned int i = 0; i < dimX; ++i) {
	xRow[i] = cube[i][j][k];
      }
      xRow = smoothGaussian(xRow, d, nMax);
      for (unsigned int i = 0; i < dimX; ++i) {
	cube[i][j][k] = xRow[i];
      }
    }
  }  
  // smooth in y - direction:  
  for (unsigned int i = 0; i < dimX; ++i) {
    for (unsigned int k = 0; k < dimZ; ++k) {
      for (unsigned int j = 0; j < dimY; ++j) {
	yRow[j] = cube[i][j][k];
      }
      yRow = smoothGaussian(yRow, d, nMax);
      for (unsigned int j = 0; j < dimY; ++j) {
	cube[i][j][k] = yRow[j];
      }
    }
  }  
  // smooth in z - direction:  
  for (unsigned int i = 0; i < dimX; ++i) {
    for (unsigned int j = 0; j < dimY; ++j) {
      for (unsigned int k = 0; k < dimZ; ++k) {
	zRow[k] = cube[i][j][k];
      }
      zRow = smoothGaussian(zRow, d, nMax);
      for (unsigned int k = 0; k < dimZ; ++k) {
	cube[i][j][k] = zRow[k];
      }
    }
  }  
}

// increase number to a certain basis.
void
incBaseNumber(unsigned int& n, unsigned int base, bool& overflow)
{
  ASSERT(n < base);
  ++n;
  if (n >= base) {
    n = 0;
    overflow = true;
  }
  else {
    overflow = false;
  }
}

// increase number to a certain basis. v[0] corresponds to lowest digit!
void
incBaseNumber(Vec<unsigned int>& v, unsigned int base)
{
  unsigned int i = 0;
  bool overflow = false;
  do {
    incBaseNumber(v[i++], base, overflow);
  }
  while (overflow && (i < v.size() ) );
}

// return maximum number which can be expressed by number with so many digits 
// to a certain base
unsigned int
maxBaseNumber(unsigned int digits, unsigned int base)
{
  double maxNum = pow(static_cast<double>(base), static_cast<int>(digits));
  if (maxNum > UINT_MAX){
    return UINT_MAX;
  }
  return static_cast<unsigned int>(maxNum);
}

/** makes matrix more similar to distance matrix: positive definite,
    and zero in diagonals */
void
distify(Vec<Vec<double> >& m)
{
  for (unsigned int i = 0; i < m.size(); ++i) {
    double ddi = m[i][i];
    for (unsigned int j = 0; j < i; ++j) {
      double ddj = m[j][j];
      // first symmetrize:
      double mavg = 0.5 * (m[i][j] + m[j][i]);
      m[i][j] = 0.5 * (fabs(ddi - mavg) + fabs(ddj - mavg));
      m[j][i] = m[i][j];
    }
  }
  for (unsigned int i = 0; i < m.size(); ++i) {
    m[i][i] = 0.0;
  }
}

/** returns index n,n from point m,n */
double
diagonalProjection(int row, int col, int size)
{
  if (col > row) {
    int help = row;
    row = col;
    col = help;
  }
  double x = static_cast<double>(row - col)/2.0;
  return row - x;
}

/** returns row index of projection to antidiagonal */
double
antiDiagonalProjection(int row, int col, int size)
{
  double x = static_cast<double>(row - col)/2.0;
  double cd = 0.5 * size - x; // anti-diagonal column
  ASSERT(cd >= 0.0);
  //   if (cd < 0.0) {
  //     x = static_cast<double>(col - row)/2.0;
  //     cd = 0.5 * size - x;
  //   }
  double rd = 0.5 * static_cast<double>(size) + x; // diagonal column
  return rd;
}

Vec<double>
getAntiDiagonalAverages(const Vec<Vec<double> >& mat)
{
  unsigned int maxDiagSize = 2 * mat.size() - 1;
  Vec<double> result(maxDiagSize);
  for (unsigned int i = 0; i < maxDiagSize; ++i) {
    result[i] = vecMean(getAntiDiagonal(mat, i));
  }
  return result;
}

Vec<double>
getDiagonalAverages(const Vec<Vec<double> >& mat)
{
  int maxDiagSize = 2 * static_cast<int>(mat.size()) - 1;
  Vec<double> result(maxDiagSize);
  int istart = - static_cast<int>(mat.size()) + 1;
  int ioffs = - istart;
  for (int i = istart; i < static_cast<int>(mat.size()); ++i) {
    Vec<double> diag = getDiagonal(mat, i);
    // cout << "Getting diagonal " << i << "  " << diag << endl;
    ASSERT(i + ioffs < static_cast<int>(result.size()));
    result[i + ioffs] = vecMean(diag);
  }
  return result;
}

/** returns correlation coefficient (Pearson correlation).
    @see Peter Dalgaard, Introductory Statistics with R
*/
double
correlationCoefficient(const Vec<double>& x, const Vec<double>& y)
{
  PRECOND(x.size() == y.size());
  double xm = vecMean(x);
  double ym = vecMean(y);
//   double xvar = vecVariance(x);
//   double yvar = vecVariance(y);
  double sum = 0.0;
  double denom1 = 0.0;
  double denom2 = 0.0;
  for (unsigned int i = 0; i < x.size(); ++i) {
    sum += (x[i] - xm) * (y[i] - ym);
    denom1 += (x[i] - xm) * (x[i] - xm);
    denom2 += (y[i] - ym) * (y[i] - ym);
  }
  double denom = sqrt(denom1 * denom2);  
  if (denom == 0.0) {
    return 0.0;
  }
  return sum / denom;
}

/** what would be that z-score of a perfectly predicted row
    in a contact matrix? Assume vector with all zeros and one "1" */
double
contactMatrixRowMaxScore(unsigned int n)
{
  PRECOND(n > 0);
  double mean = 1.0 / static_cast<double>(n);
  double sum = 1.0 * 1.0;
  double std = sqrt((sum/n) - (mean * mean)); 
  if (std == 0.0) {
    cerr << "Warning: zero standard deviation in contactMatrixRowMaxScore detected" << endl;
    return 0.0;
  }
  // ASSERT((std > 0.0));
  double result = (1.0 - mean) / std;
  POSTCOND(result > 0.0);
  return result;
}

double
contactMatrixRowScore(const Vec<double>& v, 
		      unsigned int pos, unsigned int diagLim)
{
  double maxVal = v[0];
  for (unsigned int i = 0; i + diagLim < pos + 1; ++i) {
    if (v[i] > maxVal) {
      maxVal = v[i];
    }
  }
  for (unsigned int i = pos + diagLim; i < v.size(); ++i) {
    if (v[i] > maxVal) {
      maxVal = v[i];
    }
  }
  double mean = vecMean(v);
  double maxStd = sqrt(vecVariance(v));
  if (maxStd < 1e-10) {
    return 0.0;
  }
  double zScore = (maxVal-mean) / maxStd;
  return zScore;
}

/** returns "1.0" for perfectly "sharp" predictions
    (each row with all zeros and one "1"),
    final result is smaller one otherwise
*/
double
contactMatrixQuality(const Vec<Vec<double> >& matrix, unsigned int diagLim)
{
  if (matrix.size() == 0) {
    return 0.0;
  }
  double maxZScore = contactMatrixRowMaxScore(matrix.size());
  double sum = 0.0;
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    sum += contactMatrixRowScore(matrix[i], i, diagLim);
  }
  sum /= maxZScore; // renormalize each row to a maximum score of one
  return (sum / matrix.size()); // renormalize to total score maximal "1"
}

/** in each row, leave only the highest scoring element,
 also delete diagonal */
void
winnerTakesAll(Vec<Vec<double> >& matrix, int diagBorder, double cutoff)
{
  // store all promising candidates:
  Vec<RankedSolution5<int, int> > positions;
  int nn = matrix.size();
  for (int i = 0; i < nn; ++i) {
    for (int j = 0; j + diagBorder <= i; ++j) {
      if (matrix[i][j] >= cutoff) {
	positions.push_back(RankedSolution5<int, int>(matrix[i][j], i, j));
      }
    }
  }
  // sort:
  sort(positions.begin(), positions.end());
  reverse(positions.begin(), positions.end()); // highest score first:
  int n, m;
  while (positions.size() > 0) {
    n = positions[0].second;
    m = positions[0].third;
    // delete all competing interactions:
    for (int i = 0; i < nn; ++i) {
      if (i == n) {
	continue;
      }
      matrix[i][m] = 0.0;
      matrix[m][i] = 0.0;
      // delete competing positions:
      for (int k = static_cast<int>(positions.size())-1; k >= 0; --k) {
	if (((positions[k].second == i) && (positions[k].third == m)) 
	    || ((positions[k].second == m) && (positions[k].third == i))) {
	  positions.erase(positions.begin() + k);
	  break; // list is unique, we can quite loop here
	}
      }
    }
    for (int i = 0; i < nn; ++i) {
      if (i == m) {
	continue;
      }
      matrix[n][i] = 0.0;
      matrix[i][n] = 0.0;
      // delete competing positions:
      for (int k = static_cast<int>(positions.size())-1; k >= 0; --k) {
	if (((positions[k].second == i) && (positions[k].third == n)) 
	    || ((positions[k].second == n) && (positions[k].third == i))) {
	  positions.erase(positions.begin() + k);
	  break; // list is unique, we can quite loop here
	}
      }
    }
    // delete first element
    if (positions.size() > 0) {
      positions.erase(positions.begin());
    }
  }
}




/** sets all elements i,i-diagBorder to i,i+diagBorder to specified value. */
void
fillDiagonal(Vec<Vec<double> >& matrix, int diagBorder, double value) {
  for (int i = 0; i <  static_cast<int>(matrix.size()); ++i) {
    for (int j = i-diagBorder; j <= i+diagBorder; ++j) {
      if ((j < 0) || (j >= static_cast<int>(matrix.size()))) {
	continue;
      }
      matrix[i][j] = value;
      matrix[j][i] = value;
    }
  }
}






