#include <iostream>
#include <fstream>
#include <debug.h>
#include <Vec.h>
#include <GetArg.h>
#include <set>

#define OVERLAP_VERSION "0.6.4"

/** Version history
 * 0.6.0  First production version for generating genomic histograms 
 * 0.6.1  Second algorithm (finding HOLCs). Option -a hist|holc
 * 0.6.2  Multi-region mode: option -a multi and -r <col1> <col2> ...
 * 0.6.3  Multi-region mode using one column with a region name: option -a multi and -t <col> ...
 * 0.6.4  Change in verbosity of output.
 */

using namespace std;

typedef int int_type;
typedef string::size_type size_type;

/** Central function that performs binning
 * starts and ends define intervals; starts are zero-based, ends are one-based
 */
Vec<Vec<int_type> > 
overlapHist(const Vec<int_type>& starts,
	    const Vec<int_type>& ends,
	    int_type discr,
	    int verbose) {
  ASSERT(starts.size() == ends.size());
  if (verbose > 1) {
    cout << "# Starting overlapHist" << endl;
  }
  int_type chromMax = 0;
  for (Vec<int_type>::size_type i = 0; i < ends.size(); ++i) {
    if (ends[i] > chromMax) {
      chromMax = ends[i];
    }
  }
  ++chromMax; // one larger
  Vec<int_type> v(chromMax, 0); // array of counters for each genomic position
  for (Vec<int_type>::size_type i = 0; i < starts.size(); ++i) {
    int_type start = starts[i];
    int_type end = ends[i];
    ASSERT(start >= 0);
    for (int_type j = start; j < end; ++j) {  // increase count for each position of interval
      ASSERT(j < static_cast<int_type>(v.size()));
      v[j] = v[j] + 1; 
    }
  }
  int_type firstNonZero = 0;
  for (firstNonZero = 0; firstNonZero < chromMax; ++firstNonZero) {
    if (v[firstNonZero] != 0) {
      break;
    }
  }
  ERROR_IF ( firstNonZero == chromMax, "No intervals found!");
  Vec<int_type> starts2(1,firstNonZero);
  Vec<int_type> counts2(1,v[firstNonZero]);
  Vec<int_type> counts2Min(1,v[firstNonZero]);
  Vec<int_type> counts2Max(1,v[firstNonZero]);
  Vec<int_type> counts2Sum(1,v[firstNonZero]);
  Vec<int_type> counts2SumSq(1,v[firstNonZero] * v[firstNonZero]);
  Vec<int_type> ends2;
  ASSERT(starts2.size() == counts2.size());
  if (verbose > 1) {
    cout << "# Finished parsing intervals. Counting overlaps...\n" << endl;
  }
  int_type lastOpen = 0; // id of last opened interval
  for (int_type i = firstNonZero + 1; i < chromMax; ++i) {
    ASSERT(i < static_cast<int_type>(v.size()));
    if ((abs(v[i] - counts2Min[lastOpen]) >= discr) 
	|| (abs(v[i] - counts2Max[lastOpen]) >= discr) 
	|| ((v[i] == 0) && (v[i-1]!= 0)) || ((v[i]!= 0) && (v[i-1] == 0))) { // start new interval
      if (starts2.size() != ends2.size()) {
	ends2.push_back(i); // close interval
      }
      //     ASSERT(length(starts2) == length(ends2))
      // ASSERT(length(starts2) == length(counts2))
      if ((v[i] != 0) && ((i+1) != chromMax)) { // start new interval
	starts2.push_back(i);
	counts2.push_back(v[i]);
	counts2Min.push_back(v[i]);
	counts2Max.push_back(v[i]);
	counts2Sum.push_back(v[i]);
	counts2SumSq.push_back(v[i] * v[i]);
	++lastOpen; // increase id of last open interval
      }
    } else { // no new interval started. Update min and max values for that interval
      if (starts2.size() != ends2.size()) { // there is currently an interval open
	ASSERT(v[i] != 0);
	if (v[i] < counts2Min[lastOpen]) {
	  counts2Min[lastOpen] = v[i];
	}
	if (v[i] > counts2Max[lastOpen]) {
	  counts2Max[lastOpen] = v[i];
	}
	counts2Sum[lastOpen] += v[i];
	counts2SumSq[lastOpen] += v[i] * v[i];
      }
    }
  }
  if (ends2.size() < starts2.size()) {
    ends2.push_back(chromMax);
  }
  ASSERT(starts2.size() == ends2.size());
  // ASSERT(length(starts2) == length(ends2))
  if (verbose > 1) {
    cout << "# Finished overlapHist_internal...\n" << endl;
  }
  Vec<Vec<int_type> > result;
  result.push_back(starts2);
  result.push_back(ends2);
  result.push_back(counts2);
  result.push_back(counts2Min);
  result.push_back(counts2Max);
  result.push_back(counts2Sum);
  result.push_back(counts2SumSq);
  if (verbose > 1) {
    cout << "# Finished overlapHist" << endl;
  }
  return result;
}

Vec<size_type> 
computeRegionDescriptor(const Vec<double>& v, double cutoff) {
  Vec<size_type> result(v.size(), 0U);
  for (size_type i = 0; i < v.size(); ++i) {
    if (v[i] >= cutoff) {
      result[i] = 1;
    }
  }
  return result;
}

bool
canStartIntervall(const Vec<Vec<double> >& regionFractions, double cutoff,size_type pos) {
  for (size_type k = 0; k < regionFractions[pos].size(); ++k) {
    if (regionFractions[pos][k] >= cutoff) {
      return true;
    }
  }
  return false;
}

bool
canStartIntervall(const Vec<Vec<int_type> >& regionCounts, size_type pos) {
  for (size_type k = 0; k < regionCounts[pos].size(); ++k) {
    if (regionCounts[pos][k] >= 0) {
      return true;
    }
  }
  return false;
}

bool
shouldFinishIntervall(const Vec<Vec<int_type> >& regionCounts,size_type pos) {
  if (pos == 0) {
    return false;
  }
  for (size_type k = 0; k < regionCounts[pos].size(); ++k) {
    if (regionCounts[pos-1][k] != regionCounts[pos][k]) {
      return true;
    }
  }
  return false;
}

/** Central function that performs binning
 * starts and ends define intervals; starts are zero-based, ends are one-based
 * The result format is somewhat strange: a 3D vector of integers;
 * The v[0][0] : all start positions; v[0][1]: all end positions; v[1] : a 2d vector with all counts for all regions
 */
Vec<Vec<Vec<int_type> > >
multiOverlapHist(const Vec<int_type>& starts,
		 const Vec<int_type>& ends,
		 const Vec<Vec<double> >& regionFractions, // designation of what type of region the current region is interacting with regionFractions[position][type]
		 double regionCutoff, // what fraction garantuees region membership?
		 int_type discr,
		 int verbose) {
  ASSERT(starts.size() == ends.size());
  if (verbose > 1) {
    cout << "# Starting overlapHist" << endl;
  }
  int_type chromMax = 0;
  size_type nReg = regionFractions[0].size(); // number of different region types
  for (Vec<int_type>::size_type i = 0; i < ends.size(); ++i) {
    if (ends[i] > chromMax) {
      chromMax = ends[i];
    }
  }
  ++chromMax; // one larger
  Vec<Vec<int_type> > v(chromMax, Vec<int_type>(nReg, 0)); // array of counters for each region and each genomic position 
  bool found = false;
  int_type firstNonZero = chromMax;
  for (size_type k = 0; k < nReg; ++k) {
    for (Vec<int_type>::size_type i = 0; i < starts.size(); ++i) {
      int_type start = starts[i];
      int_type end = ends[i];
      ASSERT(start >= 0);
      if (regionFractions[i][k] >= regionCutoff) {
	for (int_type j = start; j < end; ++j) {  // increase count for each region cutoff and each position of interval
	  ASSERT(j < static_cast<int_type>(v.size()));
	  ++v[j][k];
	}
	if (!found) {
	  firstNonZero = start;
	  found = true;
	}
      }
    }
  }

  ERROR_IF ( firstNonZero == chromMax, "No intervals found!");
  Vec<int_type> starts2(1,firstNonZero);
  Vec<Vec<int_type> > counts2(1,v[firstNonZero]);
  Vec<int_type> ends2;
  ASSERT(starts2.size() == counts2.size());
  if (verbose > 1) {
    cout << "# Finished parsing intervals. Counting overlaps...\n" << endl;
  }
  int_type lastOpen = 0; // id of last opened interval
  for (int_type i = firstNonZero + 1; i < chromMax; ++i) {
    ASSERT(i < static_cast<int_type>(v.size()));
    // if ((abs(v[i] - counts2Min[lastOpen]) >= discr) 
    // || (abs(v[i] - counts2Max[lastOpen]) >= discr) 
    // || ((v[i] == 0) && (v[i-1]!= 0)) || ((v[i]!= 0) && (v[i-1] == 0))) { // start new interval
    if (shouldFinishIntervall(v, i)) {
      if (starts2.size() != ends2.size()) {
	ends2.push_back(i); // close interval
      }
      //     ASSERT(length(starts2) == length(ends2))
      // ASSERT(length(starts2) == length(counts2))
      // if ((v[i] != 0) && ((i+1) != chromMax)) { // start new interval
      if (canStartIntervall(v, i) && ((i+1) != chromMax)) {
	starts2.push_back(i);
	counts2.push_back(v[i]);
	++lastOpen; // increase id of last open interval
      }
    } else { // no new interval started. Update min and max values for that interval
      if (starts2.size() != ends2.size()) { // there is currently an interval open
	// ASSERT(v[i] != 0);
	// update sum, min, max, avg ... not implemented yet
      }
    }
  }
  if (ends2.size() < starts2.size()) {
    ends2.push_back(chromMax);
  }
  ASSERT(starts2.size() == ends2.size());
  // ASSERT(length(starts2) == length(ends2))
  if (verbose > 1) {
    cout << "# Finished overlapHist_internal...\n" << endl;
  }
  Vec<Vec<Vec<int_type> > > result;
  Vec<Vec<int_type> > startsAndEnds;
  startsAndEnds.push_back(starts2);
  startsAndEnds.push_back(ends2);
  // result.push_back(starts2);
  // result.push_back(ends2);
  result.push_back(startsAndEnds);
  result.push_back(counts2);
  if (verbose > 1) {
    cout << "# Finished overlapHist" << endl;
  }
  return result;
}


/** Central function that performs binning
 * starts and ends define intervals; starts are zero-based, ends are one-based
 * discr has different interpretation: minimum number of overlapping intervals
 */
Vec<Vec<int_type> > 
findHotspots(const Vec<int_type>& starts,
	     const Vec<int_type>& ends,
	     int_type discr,
	     int verbose) {
  ASSERT(starts.size() == ends.size());
  if (verbose > 1) {
    cout << "# Starting overlapHist" << endl;
  }
  int_type chromMax = 0;
  for (Vec<int_type>::size_type i = 0; i < ends.size(); ++i) {
    if (ends[i] > chromMax) {
      chromMax = ends[i];
    }
  }
  ++chromMax; // one larger
  Vec<int_type> v(chromMax, 0); // array of counters for each genomic position
  for (Vec<int_type>::size_type i = 0; i < starts.size(); ++i) {
    int_type start = starts[i];
    int_type end = ends[i];
    ASSERT(start >= 0);
    for (int_type j = start; j < end; ++j) {  // increase count for each position of interval
      ASSERT(j < static_cast<int_type>(v.size()));
      v[j] = v[j] + 1; // v cannot start at zero
    }
  }
  int_type firstNonZero = 0;
  for (firstNonZero = 0; firstNonZero < chromMax; ++firstNonZero) {
    if (v[firstNonZero] >= discr) {
      break;
    }
  }
  ERROR_IF ( firstNonZero == chromMax, "No hotspots of long-range covariation found!");
  Vec<int_type> starts2(1,firstNonZero);
  Vec<int_type> counts2(1,v[firstNonZero]);
  Vec<int_type> counts2Min(1,v[firstNonZero]);
  Vec<int_type> counts2Max(1,v[firstNonZero]);
  Vec<int_type> counts2Sum(1,v[firstNonZero]);
  Vec<int_type> counts2SumSq(1,v[firstNonZero] * v[firstNonZero]);
  Vec<int_type> ends2;
  ASSERT(starts2.size() == counts2.size());
  if (verbose > 1) {
    cout << "# Finished parsing intervals. Counting overlaps...\n" << endl;
  }
  size_type lastOpen = 0; // id of last opened interval
  bool isOpen = true; // one interval is currently open
  ASSERT(counts2.size() == (lastOpen+1));
  for (int_type i = firstNonZero + 1; i < chromMax; ++i) {
    ASSERT(i < static_cast<int_type>(v.size()));
    ASSERT(counts2.size() == (lastOpen+1));
    if (isOpen) {
      ASSERT(starts2.size() == (ends2.size() +1 ));
      if (v[i] >= discr) { // remain open, update stats
	ASSERT(v[i] != 0);
	if (v[i] < counts2Min[lastOpen]) {
	  counts2Min[lastOpen] = v[i];
	}
	if (v[i] > counts2Max[lastOpen]) {
	  counts2Max[lastOpen] = v[i];
	}
	counts2Sum[lastOpen] += v[i];
	counts2SumSq[lastOpen] += v[i] * v[i];
	
      } else { // close interval; store stats
	ends2.push_back(i); // close interval
	isOpen = false;
      }
    } else {
      ASSERT(starts2.size() == ends2.size());
      if ((v[i] >= discr) && ((i+1) < chromMax)) { // open new 
	starts2.push_back(i);
	counts2.push_back(v[i]);
	counts2Min.push_back(v[i]);
	counts2Max.push_back(v[i]);
	counts2Sum.push_back(v[i]);
	counts2SumSq.push_back(v[i] * v[i]);
	++lastOpen; // increase id of last open interval
	isOpen = true;
      } else {
	// do nothing, remain closed
	ASSERT(!isOpen);
      }
    }
  }
  if (ends2.size() < starts2.size()) {
    ends2.push_back(chromMax);
  }
  ASSERT(starts2.size() == ends2.size());
  if (verbose > 1) {
    cout << "# Finished overlapHist_internal...\n" << endl;
  }
  Vec<Vec<int_type> > result;
  result.push_back(starts2);
  result.push_back(ends2);
  result.push_back(counts2);
  result.push_back(counts2Min);
  result.push_back(counts2Max);
  result.push_back(counts2Sum);
  result.push_back(counts2SumSq);
  if (verbose > 1) {
    cout << "# Finished findHotspots" << endl;
  }
  return result;
}


/*
void overlapHist(starts, ends, chroms = NULL, chromLengths, verbose=0) {
  result = NULL
  if (is.null(chroms)) {
   result = overlapHist_interna(starts, ends, verbose=(verbose-1))
  } else {
   result = list()
   chromFac = as.factor(chroms)
   chromFacNames = levels(chromFac)
   chromNames = chromFacNames[chromFac]
   chromLengthNames = chromLengths[,1]
   if (is.factor(chromLengthNames)) {
     chromLengthNames = levels(chromLengthNames)[chromLengthNames]
   }
   for (i in 1:length(chromFacNames)) {
    if (verbose > 0) {
     cat("Working on", i,chromFacNames[i], "\n")
    }
    chromStarts = starts[chromNames == chromFacNames[i]]
    chromEnds = ends[chromNames == chromFacNames[i]]
    chromLen = chromLengths[chromLengthNames == chromFacNames[i],2] 
    if (verbose > 1) {
     cat("Length of chromosome", chromFacNames[i], chromLen, "\n")
    }
    ASSERT(!is.null(chromLen))
    ASSERT(!is.na(chromLen))
    ASSERT(length(chromLen) == 1)
    result[[chromFacNames[i]]] = overlapHist_internal(chromStarts, chromEnds, chromLen, verbose=(verbose-1))
   }
  }
  ASSERT(!is.null(result))
  result
}
*/

/*
write.list2bed = function(data, file="", startCol=1, endCol=2) {
  chromNames = names(data)
  append = FALSE
  for (i in 1:length(data)) {
    dat = data[[i]]
    for (j in 1:nrow(dat)) {
      cat(chromNames[i], dat[j, startCol], dat[j,endCol], "\n",
	append=append, file=file)
      append=TRUE
    }  
  }
}
*/

/*
test.overlapHist.small = function() {
 Start = c(10,30)
 End   = c(20,40)
 Chrom = c("chr1", "chr1")
 chromVec = c("chr1")
 chromLens = c(50)
 result = overlapHist(Start, End, Chrom, cbind(chromVec,chromLens))
 write.list2bed(result)
 checkEquals(length(result), 1)
 checkEquals(nrow(result[[1]]), 2)
 result
}

test.overlapHist.small2 = function() {
 Start = c(10,15,30)
 End   = c(20,25,40)
 Chrom = c("chr1", "chr1")
 chromVec = c("chr1")
 chromLens = c(50)
 result = overlapHist(Start, End, Chrom, cbind(chromVec,chromLens))
 write.list2bed(result)
 checkEquals(length(result), 1)
 checkEquals(nrow(result[[1]]), 4)
 result
}
*/

void helpOutput(ostream& os) {
  os << "-i FILENAME  : input file" << endl;
  os << "-a hist|holc|multi : defines algorithm (generate histogram or find HOLCs)" << endl;
  os << "-c CHROM  : chromosome name (like chrX or chr4)" << endl;
  os << "-d 1..N   : minimum count difference between reported histogram regions" << endl;
  os << "-r <col1> <col2> ... <colN>  : specifies columns with fractional overlap information of N region types. Alternative: option -t <col> specifies one column with a region name." << endl;
  os << "-t <col> specifies one column with a region name." << endl;
  os << "--tf <filename> specifies a file containing region names. Use in conjunction with option -t. Format: first word contains number of regions; subsequent words are the region names to be recognized." << endl;
  // os << "m SIZE: length of chromosome (maximum defined position in assembly)" << endl;
  os << "-v 0|1|2 : verbose level" << endl;
}

void
parameterOutput(ostream& os, int argc, char** argv) {
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

/** Main program */
int main(int argc, char ** argv) {
  string algorithm = "hist"; // defined algorithm; defined: "hist" or "holc"
  string filename;
  Vec<int_type> starts;
  Vec<int_type> ends;
  Vec<string> regionTypeNames;
  string typeFileName;
  size_type chromCol = 1;
  size_type startCol = 2;
  size_type endCol = 3; // use one-based column counting!
  vector<unsigned int> regionCols;
  double regionCutoff = 0.5;
  Vec<Vec<double> > regionFractions;
  set<string> regionTypeNameSet;
  // int_type chromMax = 0;
  int_type discr = 1;
  size_type typeCol = 0; // id column containing region type like "tRNA", "rRNA", "utr3" etc; 0: unspecified
  int verbose = 1;
  string chromName;
  if (verbose > 0) {
    cout << "# Analyzing overlaps between genomic regions with OVERLAP version " << OVERLAP_VERSION
	 << endl;
    cout << "# ";
    parameterOutput(cout, argc, argv);
  }
  if (argc < 2) {
    helpOutput(cout);
    exit(0);
  }
  getArg("a", algorithm, argc, argv, algorithm);
  getArg("i", filename, argc, argv, filename);
  getArg("c", chromName, argc, argv,chromName);
  getArg("d", discr, argc, argv, discr);
  // getArg("m", chromMax, argc, argv,chromMax);
  getArg("r", regionCols, argc, argv);
  getArg("t", typeCol, argc, argv, typeCol);  // id of column containing type like "utr5" or "tRNA"
  getArg("-tf", typeFileName, argc, argv, typeFileName);
  getArg("v", verbose, argc, argv, verbose);
  // string name;
  // read region names
  if (typeFileName.size() > 0) {
    ifstream typeFile(typeFileName.c_str());
    ERROR_IF(!typeFile, "Error opening region type file name (option --tf)" + typeFileName);
    size_type tn = 0;
    typeFile >> tn; // number of different region types
    for (size_type i = 0; i < tn; ++i) {
      string regName;
      ERROR_IF(!typeFile, "Error reading line " + uitos(i) + " in region type file " + typeFileName);
      typeFile >> regName;
      regionTypeNameSet.insert(regName);
    }
  }
  if (verbose > 1) {
    cout << "# Analyzing chromosome " << chromName << endl;
  }
  ERROR_IF(chromName.size() == 0,
	   "No chromosome name specified. Use option -c");
  ifstream ifile(filename.c_str());
  ERROR_IF(!ifile,"Error opening input file (option -i)");
  size_type lineCount = 0;
  while (ifile) {
    string line = getLine(ifile);
    ++lineCount;
    if (verbose > 3) {
      cout << "Parsing line: " << lineCount << " : " << line << endl;
    }
    if (line.size() == 0) {
      continue;
    }
    vector<string> words = getTokens(line);
    ERROR_IF(chromCol > words.size(),
	     "Chromosome column index larger than number of detected words in line " + uitos(lineCount) + " : "
	     + line);
    string chrom = words[chromCol-1];
    if (chrom == chromName) {
      ERROR_IF(startCol > words.size(),
	       "Start column index larger than number of detected words in line " + uitos(lineCount) + " : "
	       + line);
      int_type start = stoi(words[startCol-1]);
      ERROR_IF(endCol > words.size(),
	       "End column index larger than number of detected words in line " + uitos(lineCount) + " : "
	       + line);
      int_type end = stoi(words[endCol-1]);
      ERROR_IF(chromCol > words.size(),
	       "Chromosome column index larger than number of detected words in line " + uitos(lineCount) + " : "
	       + line);
      starts.push_back(start);
      ends.push_back(end);
      if (verbose > 2) {
	cout << "Added start-stop pair: " << start << "\t" << end << " sizes: " << starts.size() << " " << ends.size() << endl;
      }
      if (regionCols.size() > 0) {
	Vec<double> fractions(regionCols.size());
	for (size_type i = 0; i < regionCols.size(); ++i) {
	  ERROR_IF((regionCols[i]-1) >= words.size(), "Insufficient number of columns in line " + uitos(lineCount));
	  fractions[i] = stod(words[regionCols[i]-1]);
	  regionFractions.push_back(fractions);
	}
      }
      if (typeCol > 0) {
	ERROR_IF(typeCol > (words.size()),
		 "Insufficient number of words (region type column missing, inspect option -t) in line " + uitos(lineCount));
	regionTypeNames.push_back(words[typeCol-1]);
	if (typeFileName.size() == 0) {
	  regionTypeNameSet.insert(words[typeCol-1]);
	}
      }
    } else {
      if (verbose > 2) {
	cout << "Chromosome name does not match " << chromName << " : " << chrom << endl;
      }
    }
  }
  if (regionTypeNameSet.size() > 0) {
    ERROR_IF(regionTypeNames.size() != starts.size(),
	     "Internal error: number region type names must match number of region starts.");
    size_type regionTypeCount = regionTypeNameSet.size();
    regionFractions = Vec<Vec<double> >(regionTypeNames.size(), Vec<double>(regionTypeCount, 0.0));
    for (size_type i = 0; i < regionFractions.size(); ++i) {
      set<string>::const_iterator it = regionTypeNameSet.find(regionTypeNames[i]);
      ERROR_IF((it == regionTypeNameSet.end()),
	       "Could not find region name in line " + uitos(i)); // + regionTypeNames[i] + " 
      ERROR_IF(static_cast<size_type>(distance(regionTypeNameSet.begin(), it)) >= regionFractions[i].size(),
	       "Internal error in line " + uitos(i) + " set iterator distance too large.");
      regionFractions[i][distance(regionTypeNameSet.begin(), it)] += 1.0;
    }
    cout << "# Defined region types: " << regionTypeNameSet.size();
    for (set<string>::const_iterator it = regionTypeNameSet.begin(); it != regionTypeNameSet.end(); ++it) {
      cout << " " << (*it);
    }
    cout << endl;
  }
  if (verbose > 0) {
    cout << "# Number of read regions: " << starts.size() << endl;;
  }
  ERROR_IF(starts.size() != ends.size(), "Number of starts and ends do not match!");
  // ERROR_IF(chromMax == 0, "Chromosome size has to be specified with option -m");
  ERROR_IF(starts.size() == 0, "No regions have been read from the input file!");
  Vec<Vec<int_type> > result;
  if (algorithm == "hist") {
    result = overlapHist(starts, ends, discr, verbose);
  } else if (algorithm == "holc") {
    result = findHotspots(starts, ends, discr, verbose);
  } else if (algorithm == "multi") {
    ERROR_IF(regionFractions.size() == 0 || regionFractions[0].size() == 0,
	     "No region fractions were specified. Use option -r <col1> <col2> ... to specify columns.");
    Vec<Vec<Vec<int_type> > > result2 = multiOverlapHist(starts, ends, regionFractions, regionCutoff, discr, verbose);
    result.push_back(result2[0][0]);
    result.push_back(result2[0][1]);
    const Vec<Vec<int_type> >& regionCounts = result2[1];
    cout << "# chrom\tstart\tend";
    for (set<string>::const_iterator it = regionTypeNameSet.begin(); it != regionTypeNameSet.end(); ++it) {
      cout << "\t" << (*it);
    }
    for (Vec<int_type>::size_type i = 0; i < result[0].size(); ++i) {
      if (chromName.size() > 0) {
	cout << chromName << "\t";
      }
      // int_type len = result[1][i] - result[0][i]; // length of interval
      cout << result[0][i] << "\t" << result[1][i];
      for (size_type k = 0; k < regionCounts[i].size(); ++k) {
	cout << "\t" << regionCounts[i][k];
      }
      cout << endl;
    }
    exit(0);
  } else {
    ERROR("Unknown algorithm (defined: hist|holc): " + algorithm);
  }
  for (Vec<int_type>::size_type i = 0; i < result[0].size(); ++i) {
    if (result[2][i] > 0) {
      if (chromName.size() > 0) {
	cout << chromName << "\t";
      }
      int_type len = result[1][i] - result[0][i]; // length of interval
      cout << result[0][i] << "\t" << result[1][i] << "\t" 
	   << static_cast<double>(result[5][i])/len << "\t" // average value of interval
	// << result[2][i] << "\t" 
	   << result[3][i] << "\t" << result[4][i] << endl; // min and max of interval
    }
  }
  if (verbose > 1) {
    cout << "# Good bye!" << endl;
  }
  return 0;
  }


