#ifndef _SORTED_COMPRESSED_SEQUENCE2_
#define _SORTED_COMPRESSED_SEQUENCE2_

#include <iostream>
#include <algorithm>
#include <Vec.h>
#include <debug.h>
#include <Random.h>
#include <Timer.h>
#include <limits.h>

using namespace std;

// maximum usable span for encoding number (subtracts number of escape characters):
#define BYTE_D 256
#define CHAR_D ( BYTE_D - 4)
#define CHAR_D2 ( CHAR_D * CHAR_D )
#define CHAR_D3 ( CHAR_D * CHAR_D * CHAR_D )

class SortedCompressedSequence2ConstIterator;

  enum { STORE_BYTE2 = CHAR_MAX, STORE_BYTE3 = (CHAR_MAX-1), STORE_BYTE4 = (CHAR_MAX-2), STORE_HASH=(CHAR_MAX-3), 
	 STORE_BYTE1MAX = (CHAR_MAX - 4), 
	 STORE_BYTE1MAXDIFF = CHAR_D, STORE_BYTE2MAXDIFF = (CHAR_D * CHAR_D), STORE_BYTE3MAXDIFF = (CHAR_D * CHAR_D * CHAR_D),
	 STORE_INTERVALL = 100 }; // not more than this many differences but store absolute value in hash


class SortedCompressedSequence2 {

 public:


  typedef SortedCompressedSequence2ConstIterator const_iterator;

  typedef SortedCompressedSequence2ConstIterator iterator;

  typedef string compressed_type;

  typedef int value_type;

  typedef Vec<int> set_type;

  typedef compressed_type::size_type size_type;

  typedef compressed_type::difference_type difference_type;

  typedef map<size_type, value_type> map_type; // used to store too large values

 private:

  compressed_type s; // this string encode numeric data!

  map_type additional;

  size_type sz;

 public:

  /** Default constructor */
  SortedCompressedSequence2() : sz(0) { } 

  /** Construction using un-compressed set */
  SortedCompressedSequence2(const set_type& content) : sz(0) {
    add(content);
    // ASSERT((*this) == content);
  }

  /** Copy constructor */
  SortedCompressedSequence2(const SortedCompressedSequence2& other) : s(other.s), additional(other.additional), sz(other.sz) {
  }

  /** Virtual destructor */
  virtual ~SortedCompressedSequence2() { }

  /** Assignment operator */
  virtual SortedCompressedSequence2& operator = (const SortedCompressedSequence2& other) {
    if (this != &other) {
      copy(other);
    }
    return *this;
  }

  /** Concatenates content to end */
  virtual void add(const set_type& content);
  
  /** Returns iterator to beginning of container */
  virtual SortedCompressedSequence2ConstIterator begin() const;
  
  /** Removes all content */
  virtual void clear() {
    s = "";
    additional.clear();
    sz = 0;
  }

  /** Copy method , being used by copy constructor and assignment operator */
  virtual void copy(const  SortedCompressedSequence2& other) {
    s = other.s;
    additional = other.additional; 
    sz = other.sz;
  }

  /** Returns iterator to end of container */ 
  virtual SortedCompressedSequence2ConstIterator end() const;

  /** Returns internal representation */
  virtual const compressed_type& internal() const { return s; }

  /** Returns internal representation */
  virtual const map_type& internal2() const { return additional; }

  /** Returns number of stored items */
  virtual size_type size() const { return sz; }

  virtual set_type toVector() const;

  friend class SortedCompressedSequence2ConstIterator;

};

/** Comparison operator */
bool operator == (const SortedCompressedSequence2& left,  const SortedCompressedSequence2& right);

/** Comparison operator */
bool operator == (const SortedCompressedSequence2& left,  const SortedCompressedSequence2::set_type& right);

/** Comparison operator */
bool operator == (const SortedCompressedSequence2::set_type& left, const SortedCompressedSequence2& right);




/** Iterator for compressed class. Trying: random_access_iterator: so far too slow because of linear inc() operator */
class SortedCompressedSequence2ConstIterator : public std::iterator<std::input_iterator_tag,
					      SortedCompressedSequence2::value_type>{

 public:

  typedef string _Alloc;

  typedef SortedCompressedSequence2::compressed_type compressed_type; 

  typedef SortedCompressedSequence2::set_type set_type; 

  typedef SortedCompressedSequence2::size_type size_type; 

  typedef SortedCompressedSequence2::value_type value_type;

  typedef SortedCompressedSequence2 vector_type;

  typedef compressed_type::difference_type difference_type;

  typedef int ptrdiff_t;

 private:

  const vector_type *container;

  size_type lastPos; // cached last position

  value_type lastVal; // cached last value

  size_type pos; // current internal position on string 

  value_type val; // current content value

 public:

  /** Standard constructor */
  SortedCompressedSequence2ConstIterator(const vector_type *_container) : container(_container), lastPos(0), lastVal(0), pos(0), val(0) {
    interpretPosition(); // sets value and next jump
  }

  /** Default constructor */
  SortedCompressedSequence2ConstIterator() : container(0), lastPos(0), lastVal(0), pos(0), val(0) {
  }

  /** Constructor using an internal position. Should be private FIXIT */
  SortedCompressedSequence2ConstIterator(const vector_type *_container, size_type _pos) : container(_container), 
    lastPos(0), lastVal(0), pos(_pos), val(0) {
    interpretPosition(); // sets value and next jump
  }

  /** Copy constructor */
  SortedCompressedSequence2ConstIterator(const SortedCompressedSequence2ConstIterator& other) 
    : container(other.container), lastPos(other.lastPos), lastVal(other.lastVal), pos(other.pos), val(other.val) { }
  
  /** Virtual destructor */
  virtual ~SortedCompressedSequence2ConstIterator() { }

  /** Assignment operator */
  SortedCompressedSequence2ConstIterator& operator = (const SortedCompressedSequence2ConstIterator& other) {
    if (&other != this) {
      copy(other);
    }
    return *this;
  }

  /** Copy method */
  virtual void copy(const SortedCompressedSequence2ConstIterator& other) {
    container = other.container;
    lastPos = other.lastPos;
    lastVal = other.lastVal;
    pos = other.pos;
    val = other.val;
  }

  /** Returns position iterator is pointing to */ 
  virtual size_type getPos() const { return pos; }

  /** Increment */
  virtual void inc() {
    ASSERT(false);
    ASSERT(*this != container->end());
    lastPos = pos;
    lastVal = val;
    ++pos;
    interpretPosition(); 
  }

  /** Increment */
  virtual void dec() {
    ASSERT(false) ; // not supported anymore
    ASSERT(*this != container->begin());
    lastPos = pos;
    lastVal = val;
    --pos;
    interpretPosition(); 
  }

  /** Access to element that iterator is pointing to */
  value_type operator * ( ) {
    return val;
  }

  /** prefix ++ operator */
  void operator ++ () {
    ASSERT(*this != container->end());
    const compressed_type& set = container->s;
    ASSERT(pos < set.size());
    // lastPos = pos;
    // lastVal = val;
    ++pos; // before ++ operation, ++ operator points to last byte of previous element
    if (pos == set.size()) {
      val = 0;
      return;
    }
    ASSERT(pos < set.size());
    int c = set[pos];
    if (c <= STORE_BYTE1MAX) {
      val += (c - CHAR_MIN);
    } else {
      switch(c) {
      case STORE_BYTE2:
	ASSERT((pos+2) <set.size());
	val += (STORE_BYTE1MAXDIFF * (set[pos+1] - CHAR_MIN)) + (set[pos+2] - CHAR_MIN);;
	pos += 2;
	break;
      case STORE_BYTE3:
	ASSERT((pos+3) <set.size());
	val += (STORE_BYTE2MAXDIFF * (set[pos+1] - CHAR_MIN)) + (STORE_BYTE1MAXDIFF * (set[pos+2] - CHAR_MIN)) + (set[pos+3] - CHAR_MIN);
	pos += 3;
	break;
      case STORE_BYTE4:
	ERROR("Internal error for storing 4-byte number!");
	ASSERT(pos+4 <set.size());
	val += (STORE_BYTE3MAXDIFF * (set[pos+1] - CHAR_MIN)) + (STORE_BYTE2MAXDIFF * (set[pos+2] - CHAR_MIN)) 
	  + (STORE_BYTE1MAXDIFF * (set[pos+3] - CHAR_MIN)) + (set[pos+4] - CHAR_MIN);
	pos += 4;
	break;
      case STORE_HASH:
	val = (container->additional).find(pos)->second;
	break;
      }
    }
  }

  /** postfix ++ operator.  "int" indicates postfix, bizarre C++ idiom... */
  void operator ++ (int) {
    ASSERT(*this != container->end());
    const compressed_type& set = container->s;
    ASSERT(pos < set.size());
    ++pos; // before ++ operation, ++ operator points to last byte of previous element
    if (pos == set.size()) {
      val = 0;
      return;
    }
    ASSERT(pos < set.size());
    int c = set[pos];
    if (c <= STORE_BYTE1MAX) {
      val += (set[pos] - CHAR_MIN);      
    } else {
      switch(c) {
      case STORE_BYTE2:
	ASSERT(pos+2 <set.size());
	val += (STORE_BYTE1MAXDIFF * (set[pos+1] - CHAR_MIN)) + (set[pos+2] - CHAR_MIN);;
	pos += 2;
	break;
      case STORE_BYTE3:
	ASSERT(pos+3 <set.size());
	val += (STORE_BYTE2MAXDIFF * (set[pos+1] - CHAR_MIN)) + (STORE_BYTE1MAXDIFF * (set[pos+2] - CHAR_MIN)) + (set[pos+3] - CHAR_MIN);
	pos += 3;
	break;
      case STORE_BYTE4:
	ERROR("Internal error for storing 4-byte number!");
	ASSERT(pos+4 <set.size());
	val += (STORE_BYTE3MAXDIFF * (set[pos+1] - CHAR_MIN)) + (STORE_BYTE2MAXDIFF * (set[pos+2] - CHAR_MIN)) 
	  + (STORE_BYTE1MAXDIFF * (set[pos+3] - CHAR_MIN)) + (set[pos+4] - CHAR_MIN);
	pos += 4;
	break;
      case STORE_HASH:
	val = (container->additional).find(pos)->second;
	break;
      }
    }
  }

  /** prefix -- operator */
  void operator -- () {
    ASSERT(*this != container->begin());
    dec();
  }
  
  /** postfix -- operator */
  void operator -- (int) {
    ASSERT(*this != container->begin());
    dec();
  }
  
  /** += operator. */
  void operator += (size_type n) {
    ASSERT(false);
    ASSERT(*this != container->end());
    lastPos = pos;
    lastVal = val;
    pos += n;
    interpretPosition();
  }
  
  /** -= operator. */
  void operator -= (size_type n) {
    ASSERT(false);
    ASSERT(*this != container->begin());
    lastPos = pos;
    lastVal = val;
    pos -= n;
    interpretPosition();
  }
  
  /** Comparison operator */
  bool operator == (const SortedCompressedSequence2ConstIterator& other) const {
    return (container == other.container) && (pos == other.pos);
  }
  
  /** Negated equality */
  bool operator != (const SortedCompressedSequence2ConstIterator& other) const {
    return !(*this == other);
  }
  
					      private:
  
  /** Interprets compressed string at current position and set val member */
  void interpretPosition();
  
};
  
/** Interprets compressed string at current position and sets val members */
inline
void
SortedCompressedSequence2ConstIterator::interpretPosition() {
  // cout << "Starting interPretPosition with " << pos << " " << val << endl;
  const compressed_type& set = container->s;
  val = 0;
  if ((set.size() == 0) || (pos >= set.size())) {
    return;
  }
  ERROR_IF(pos != 0, "Internal error in SortedCompressedSequence2ConstIterator::interpretPosition");
    char c = set[pos];
    switch(c) {
    case STORE_BYTE2:
      ASSERT(pos+2 <set.size());
      val += (STORE_BYTE1MAXDIFF * (set[pos+1] - CHAR_MIN)) + (set[pos+2] - CHAR_MIN);;
      pos += 2;
      break;
    case STORE_BYTE3:
      ASSERT(pos+3 <set.size());
      val += (STORE_BYTE2MAXDIFF * (set[pos+1] - CHAR_MIN)) + (STORE_BYTE1MAXDIFF * (set[pos+2] - CHAR_MIN)) + (set[pos+3] - CHAR_MIN);
      pos += 3;
      break;
    case STORE_BYTE4:
      ASSERT(pos+4 <set.size());
      val += (STORE_BYTE3MAXDIFF * (set[pos+1] - CHAR_MIN)) + (STORE_BYTE2MAXDIFF * (set[pos+2] - CHAR_MIN)) 
	+ (STORE_BYTE1MAXDIFF * (set[pos+3] - CHAR_MIN)) + (set[pos+4] - CHAR_MIN);
      pos += 4;
      break;
    case STORE_HASH:
      val = (container->additional).find(pos)->second;
      break;
    default:
      val += (set[pos] - CHAR_MIN);
    }

//   int lastPosi = static_cast<int>(lastPos);
//   for (int i = static_cast<int>(pos); i >= 0; i--) {
//     if ((i == lastPosi) && (lastPosi > 0)) {
//       val += lastVal;
//       break;
//     }
//     if (set[i] > CHAR_MIN) {
//       val += (set[i] - CHAR_MIN);
//     } else {
//       // ASSERT((container->additional).find(i) != (container->additional).end());
//       val += (container->additional).find(i)->second;
//       break;
//     }
//   }
}

inline
SortedCompressedSequence2ConstIterator
SortedCompressedSequence2::begin() const { return SortedCompressedSequence2ConstIterator(this, 0); }
  
inline
SortedCompressedSequence2ConstIterator
SortedCompressedSequence2::end() const { return SortedCompressedSequence2ConstIterator(this, s.size()); }

inline
SortedCompressedSequence2::set_type
SortedCompressedSequence2::toVector() const {
  // cout << "Started SortedCompressedSequence2.toVector(): " << size() << endl;
  if (size() == 0) {
    return set_type();
  }
  set_type result;// (size());
  result.reserve(size());
  for (const_iterator it = begin(); it != end(); it++) {
    result.push_back(*it);
  } 
  ASSERT(result.size() == size());
  // cout << "Finished SortedCompressedSequence2.toVector()..." << endl;
  return result;
}

inline
ostream& 
operator << (ostream& os, const SortedCompressedSequence2& seq) {
  os << seq.size() << "  ";
  SortedCompressedSequence2::size_type count = 0;
  for (SortedCompressedSequence2::const_iterator it = seq.begin(); it != seq.end(); it++) {
    os << (*it) << " ";
    ++count;
  }
  //  ASSERT(count == seq.size()); // must be this many entries
  return os;
}

inline
SortedCompressedSequence2ConstIterator::ptrdiff_t 
operator - (const SortedCompressedSequence2ConstIterator& left, const SortedCompressedSequence2ConstIterator& right) {
  return (static_cast<ptrdiff_t>(left.getPos()) - static_cast<ptrdiff_t>(right.getPos()));
}
     
/** Tests compression and uncompression */
class SortedCompressedSequence2Test {

 public:

  typedef SortedCompressedSequence2::value_type value_type;
  typedef SortedCompressedSequence2::set_type set_type;
  typedef SortedCompressedSequence2::size_type size_type;

  static void testASet(const set_type& testSet) {
    cout << "Starting SortedCompressedSequence2::testASet" << endl;
    cout << "Original set: " << testSet << endl;
    SortedCompressedSequence2 compressed(testSet);
    ASSERT(compressed.size() == testSet.size());
    //    compressed_type compressed = compressSet(testSet);
    cout << "Compressed version: " << compressed.size() << " " << compressed  << endl;
    SortedCompressedSequence2::compressed_type internal = compressed.internal(); // internal representation
    cout << "Internal representation: " << internal << endl;
    for (SortedCompressedSequence2::compressed_type::const_iterator it = internal.begin(); it != internal.end(); it++) {
      cout << static_cast<int>(*it) << " " << (static_cast<int>(*it) - CHAR_MIN) << " " << (*it) << endl;
    }
    set_type uncompressed = compressed.toVector();
    cout << "Uncompressed set: " << uncompressed << endl;
    ASSERT(testSet == uncompressed);
    cout << "Testing iterator: " << endl;
    size_type count = 0;
    for (SortedCompressedSequence2::const_iterator it = compressed.begin(); it != compressed.end(); it++) {
      cout << ++count << " " << (*it) << endl;
    }
    ASSERT(count == testSet.size());

    cout << "Finished SortedCompressedSequence2::testASet" << endl;

  }

  static void test1() {
    set_type testSet;
    testSet.push_back(2);
    testSet.push_back(4);
    testSet.push_back(8);
    testSet.push_back(16);
    testASet(testSet);
  }

  static void test2() {
    set_type testSet;
    value_type val = 1;
    for (size_type i = 1; i < CHAR_D; i *= 2) {
      testSet.push_back(val);
      val += i;
    }
    testASet(testSet);
  }

  static void test3() {
    set_type testSet;
    value_type val = 1;
    for (size_type i = 1; i < CHAR_D2; i *= 2) {
      testSet.push_back(val);
      val += i;
    }
    testASet(testSet);
  }

  static void test4() {
    set_type testSet;
    value_type val = 1;
    for (size_type i = 1; i < CHAR_D3; i *= 2) {
      testSet.push_back(val);
      val += i;
    }
    testASet(testSet);
  }

  static set_type generateRandomSortedVec(value_type min, value_type max, size_type count) {
    set_type result;
    result.reserve(count);
    Random rnd = Random::getInstance();
    value_type diff = max - min;
    result.push_back(rnd.getRand(diff) + min);
    for (size_type i = 0; i < count; ++i) {
      result.push_back(result[result.size()-1] + rnd.getRand(diff) + 1);
    }
    return result;
  }

  /** Tests, if set_intersection algorithm works */
  static void testIntersection(int verbose) {
    cout << "Starting testIntersection..." << endl;
    set_type set1 = generateRandomSortedVec(1,300, 100);
    set_type set2 = generateRandomSortedVec(1,300, 100);
    cout << "Testing intersection between sets " << endl << set1.size() << endl << "and" << endl << set2.size() << endl;
    if (verbose > 1) {
      cout << "Set 1: " << set1 << endl;
      cout << "Set 2: " << set2 << endl;
    }
    set_type set3;
    set_type cset3;
    Timer timer1, timer2;
    timer1.start();
    for (int i = 0; i < 200; i++) {
      set3.clear();
      set_intersection(set1.begin(), set1.end(), set2.begin(), set2.end(), back_inserter(set3));
    }
    timer1.stop();
    cout << "intersection of vector set " << set3 << endl;
    cout << "Timer: " << timer1 << endl;
    SortedCompressedSequence2 cset1(set1);
    SortedCompressedSequence2 cset2(set2);
    ASSERT(cset1 == set1);
    ASSERT(cset2 == set2);
    if (verbose > 1) {
      cout << "Compressed Set 1: " << set1 << endl;
      cout << "Compressed Set 2: " << set2 << endl;
    }
    timer2.start();
    for (int i = 0; i < 200; i++) {
      cset3.clear();
      set_intersection(cset1.begin(), cset1.end(), cset2.begin(), cset2.end(), back_inserter(cset3));
    }
    timer2.stop();
    cout << "intersection of compressed set " << cset3 << endl;
    cout << "Timer: " << timer2 << endl;
    ASSERT(cset3 == set3);
    cout << "Finished testIntersection..." << endl;
 }

  /** Tests, if set_intersection algorithm works */
  static void testLowerBound() {
    cout << "Starting testLowerBound..." << endl;
    set_type set1 = generateRandomSortedVec(1,300, 100);
    set_type set2 = generateRandomSortedVec(1,300, 100);
    cout << "Testing lower bound between sets " << endl << set1.size() << endl << "and" << endl << set2.size() << endl;
    set_type::iterator lb;
    Timer timer1, timer2;
    timer1.start();
    for (int i = 0; i < 200; i++) {
      lb = lower_bound(set1.begin(), set1.end(), *(set2.begin()));
    }
    timer1.stop();
    cout << "Lower bound of vector set " << (*lb) << endl;
    cout << "Timer: " << timer1 << endl;
    SortedCompressedSequence2 cset1(set1);
    SortedCompressedSequence2 cset2(set2);
    SortedCompressedSequence2::iterator lb2;
    timer2.start();
    for (int i = 0; i < 200; i++) {
      lb2 = lower_bound(cset1.begin(), cset1.end(), *(cset2.begin()));
    }
    timer2.stop();
    cout << "Lower bound of compressed set " << (*lb2) << endl;
    cout << "Timer: " << timer2 << endl;
    ASSERT((*lb) == (*lb2));
    cout << "Finished testLowerBound..." << endl;
 }


};


/** Comparison operator */
inline
bool operator == (const SortedCompressedSequence2& left,  const SortedCompressedSequence2& right) {
  return (left.internal() == right.internal()) && (left.internal2() == right.internal2());
}

/** Comparison operator */
inline
bool operator == (const SortedCompressedSequence2& left,  const SortedCompressedSequence2::set_type& right) {
  if (left.size() != right.size()) {
    return false;
  }
  SortedCompressedSequence2::set_type::const_iterator it2 = right.begin();
  for (SortedCompressedSequence2::const_iterator it = left.begin(); it != left.end(); it++) {
    if (*it != *it2) {
      return false;
    }
    it2++;
  }
  return true;
}


/** Comparison operator */
inline
bool operator == (const SortedCompressedSequence2::set_type& left, const SortedCompressedSequence2& right) {
  return (right == left);
}



inline
void SortedCompressedSequence2::add(const set_type& set) {
  ASSERT(size() == 0);
  // cout << "Starting add " << set << endl;
  if (set.size() == 0) {
    return; // do nothing
  }
  string::size_type n = set.size();
  string::size_type resultLen = static_cast<string::size_type>(set.size());  // plus 1: separater for size string
  for (string::size_type i = 0; i < n; ++i) { 
    value_type diff = set[i];
    if (i > 0) {
      diff -= set[i-1];
      ASSERT(diff > 0);
    }
    if ((i == 0) || (( i % STORE_INTERVALL) == 0))  {
    }
    else if (diff < STORE_BYTE1MAXDIFF) {
    } else if (diff < STORE_BYTE2MAXDIFF) {
      resultLen += 2;
    } else if (diff <= STORE_BYTE3MAXDIFF) {
      resultLen += 3;
    } else { //  if (diff <= STORE_BYTE4MAXDIFF) {
      // resultLen += 4; // uses hash again
    }
    //     } else {
    // ERROR("Number too large!");
    // }
  }

  string result(resultLen,' ');
  set_type::size_type pc = 0;
  for(set_type::size_type i = 0; i < n; ++i) {
    value_type diff = set[i];
    if (i > 0) {
      diff -= set[i-1];
      ASSERT(diff > 0);
    }
    if ((i == 0) || ((i % STORE_INTERVALL) == 0))  {
      additional[pc] = set[i];
      result[pc++] = STORE_HASH;
    }
    else if (diff < STORE_BYTE1MAXDIFF) {
      ASSERT(pc < result.size());
      result[pc++] = diff + CHAR_MIN;
    } else if (diff < STORE_BYTE2MAXDIFF) {
      ASSERT((pc+2) < result.size());
      ASSERT(diff/STORE_BYTE1MAXDIFF < CHAR_D);
      result[pc++] = STORE_BYTE2;
      result[pc++] = (diff/STORE_BYTE1MAXDIFF) + CHAR_MIN;
      ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
      result[pc++] = (diff % STORE_BYTE1MAXDIFF) + CHAR_MIN;
      ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
    } else if (diff <= STORE_BYTE3MAXDIFF) {
      ASSERT((pc+3) < result.size());
      result[pc++] = STORE_BYTE3;
      ASSERT(diff/STORE_BYTE2MAXDIFF < CHAR_D);
      result[pc++] = (diff/STORE_BYTE2MAXDIFF) + CHAR_MIN;
      ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
      result[pc++] = ((diff/STORE_BYTE1MAXDIFF) % STORE_BYTE1MAXDIFF )+ CHAR_MIN;
      ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
      result[pc++] = (diff % STORE_BYTE1MAXDIFF) + CHAR_MIN;
      ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
    } else { // if (diff <= STORE_BYTE4MAXDIFF) {
      additional[pc] = set[i];
      result[pc++] = STORE_HASH;

//       ASSERT((pc+4) < result.size());
//       result[pc++] = STORE_BYTE4;
//       ASSERT(diff/STORE_BYTE3MAXDIFF < CHAR_D);
//       result[pc++] = (diff/STORE_BYTE3MAXDIFF) + CHAR_MIN;
//       ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
//       result[pc++] = ((diff/STORE_BYTE2MAXDIFF) % STORE_BYTE1MAXDIFF ) + CHAR_MIN;
//       ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
//       result[pc++] = ((diff/STORE_BYTE1MAXDIFF) % STORE_BYTE1MAXDIFF ) + CHAR_MIN;
//       ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
//       result[pc++] = (diff % STORE_BYTE1MAXDIFF) + CHAR_MIN;
//       ASSERT(result[pc-1] <= STORE_BYTE1MAX); 
    } 
    // else
    // ERROR("Number too large!"); 
    //     }
    
  }
  if (pc != result.size()) {
    cout << "Strange: " << pc << " " << result.size() << endl;
  }
  ASSERT(pc == result.size());
  s = s + result;
  sz += set.size();
  if (!((*this) == set)) {
    cout << "Internal error compressing set: " << set << endl;
    cout << (*this) << endl;
  }
  ASSERT((*this) == set); // Assumes that this was empty before FIXIT
}


#endif
