// --*- C++ -*------x---------------------------------------------------
#ifndef _ANNOTATION_TOOLS_H_
#define _ANNOTATION_TOOLS_H_

#include <iostream>
#include <CorrelationFinder.h>
#include <string>
#include <vector>
#include <StringTools.h>
#include <MAFAlignment.h>

class AnnotationTools {

public:

  typedef CorrelationFinder::result_container result_container;
  typedef string::size_type size_type;
  typedef MAFAlignment::length_type length_type;  

  AnnotationTools() {
    init();
  }

  /** Red region pairs in extended BED (-like) format. Count number of "hits" in results that correspond to each region.
    * Write results in simmilar format to output file. */
  void annotateRegions(istream& is, ostream& os, const result_container& results, const string& refChrom1, const string& refChrom2,
		       length_type border) {
    REMARK << "Starting annotate regions!" << endl;
    while (is) {
      string line = getLine(is);
      if (line.size() == 0) {
	continue;
      } else if (line[0] == '#') {
        os << line << endl; // copy line
      } else {
	vector<string> words = getTokens(line);
	// REMARK << "Defined columns: " << (start1Col+1) << " " << (stop1Col+1) << " " << (start2Col + 1) << " " << (stop2Col + 1) << endl;
	if (words.size() <= stop2Col) {
	  REMARK << "AnnotateTools: Ignoring weird line in BED format file: " << line << endl;
	  continue;
	}
	string chrom1 = words[chrom1Col];
	string chrom2 = words[chrom2Col];
	ERROR_IF(start1Col >= words.size(), "Error parsing BED format file: start column index 1 is greater than number of words.");
        length_type start1 = stoi(words[start1Col]);
	ERROR_IF(stop1Col >= words.size(), "Error parsing BED format file: stop column index 1 is greater than number of words.");
        length_type stop1 = stoi(words[stop1Col]);
	ERROR_IF(start2Col >= words.size(), "Error parsing BED format file: start column index 2 is greater than number of words.");
        length_type start2 = stoi(words[start2Col]);
	ERROR_IF(stop2Col >= words.size(), "Error parsing BED format file: stop column index 2 is greater than number of words.");
        length_type stop2 = stoi(words[stop2Col]);
	// REMARK << "Found regions for annotation: " << start1 << " " << stop1 << " " << start2 << " " << stop2 << endl;
        // REMARK << "Now checking how many of " << results.size() << " covarying column pairs fall into this region:" << endl;
	size_type count = 0;
	for (result_container::size_type i = 0; i < results.size(); ++i) {
	  // REMARK << results[i];
	  if (isInRegion(refChrom1, results[i].getStart(), refChrom2, results[i].getStop(),
			 chrom1, start1, stop1, chrom2, start2, stop2, border)) {
	    ++count;
	    // os << " Yes!";
	  } else {
	    // os << " Nope!";
	  }
	}
	os << line << " " << count << endl;
      }
    }
    REMARK << "Finished annotate regions!" << endl;
  }

  /** Returns true, if x-y point (pos1,pos2) is in region defined by reg1Start-reg1Stop x reg2Start-reg2Stop
   *  not that pos1 and pos2 are zero-based, 
   *  region starts are zero based, region stops are one-based.
   * the region is increased by the size of "border". The rational is that actually the count for a larger 
   * region should be used, the size of border should be the size of the cluster cutoff.
   */
  bool isInRegion(const string& chrom1, 
		  length_type pos1,
		  const string& chrom2,
		  length_type pos2,
		  const string& regChrom1,
		  length_type reg1Start,
		  length_type reg1Stop,
		  const string& regChrom2,
		  length_type reg2Start,
		  length_type reg2Stop,
		  length_type border) {
    if (border >= reg1Start) {
      reg1Start = 0;
    } else {
      reg1Start = reg1Start - border;
    }
    if (border >= reg2Start) {
      reg2Start = 0;
    } else {
      reg2Start = reg2Start - border;
    }
    reg1Stop += border;
    reg2Stop += border;
    if ((chrom1 == regChrom1) && (pos1 >= reg1Start) && (pos1 < reg1Stop)
	&& (chrom2 == regChrom2) && (pos2 >= reg2Start) && (pos2 < reg2Stop) ) {
      return true;
    }
    if ((chrom1 == regChrom2) && (pos1 >= reg2Start) && (pos1 < reg2Stop)
	&& (chrom2 == regChrom1) && (pos2 >= reg1Start) && (pos2 < reg1Stop) ) {
      return true;
    }
    return false;
  }

  void init() {
    chrom1Col = 0;
    start1Col = 1;
    stop1Col = 2;
    chrom2Col = 4;
    start2Col = 5;
    stop2Col = 6;
  }

private:

  /** Column indices in zero-based counting: */
  size_t chrom1Col;
  size_t start1Col;
  size_t stop1Col;
  size_t chrom2Col;
  size_t start2Col;
  size_t stop2Col;

};

#endif
