#include <string.h>
#include <MAFAlignment.h>
#include <BEDRegions.h>
#include <MainTools.h>

#define SHUFFLE_MAF_VERSION "0.2.2"

/** Version history:
 * version 0.9.0: first version that allows several different kinds of shuffling.
 * version 0.10.0: version that allows different kinds of shuffling applied one after the other
 * version 0.11.0: important fix of bug in horizontal shuffling
 * version 0.2.0 : vertical shuffling now preserves gap-patterns.
 * version 0.2.1 : comments are written now at end, in order to not break MAF file parsing by Galaxy system.
 * version 0.2.2 : added option "n" (NO shuffling for this option; used as control)
 */

MAFAlignment::length_type MAFAlignment::unloadCharLimit = 10000;

void helpOutput(ostream& os) {
  os << "Welcome to shufflemaf, a program for shuffling genomic alignments in MAF format. Version: " << SHUFFLE_MAF_VERSION << endl;
  os << "usage: shufflemaf a|d|h|v MAFfile ASSEMBLY" << endl;
}

/** version history:
  version 0.8.0: production version: shuffles vertically, leaves reference genome unchanged. Known issues: reference genome has to be first sequence
                 in every MAF block.    
  version 0.9.0: implemented different shuffle modes (a|d|h|v)
*/

int main(int argc, char ** argv) {
  if (argc != 4) {
    helpOutput(cout);
    exit(0);
  }
  double normLimit = 0.1;
  int verbose = 1;
  MAFAlignment maf;
  string currAssembly(argv[3]);
  maf.setRemovePropertiesMode(false);
  maf.setRefAssembly(currAssembly);
  maf.setVerbose(0);
  char * filename = argv[2];
  string shuffleMode(argv[1]);
  cout << "# Shuffled MAF format alignment: shufflemaf version " << SHUFFLE_MAF_VERSION << endl; 
  cout << "# shufflemaf called with parameters: ";

  cout << endl;
  ifstream ifs(filename);
  ERROR_IF(!ifs, "Error opening MAF file!");
  maf.read(ifs);
  if (verbose > 1) {
    REMARK << "Read " << maf.size() << " alignment blocks!" << endl;
  }
  maf.setVerbose(verbose);
  for (string::size_type i = 0; i < shuffleMode.size(); ++i) {
    cout << "# Shuffling MAF format blocks with mode " << shuffleMode[i] << endl;
    switch (shuffleMode[i]) {
    case 'a': maf.shuffleRows();
      break;
    case 'd': maf.dinucleotideShuffle(normLimit, true);
      break;
    case 'h': maf.shuffleHorizontal();
      break;
    case 'n': // n : stands for do "nothing" (used as control)
      break;
    case 'v': maf.shuffleVertical();
      break;
    default:
      ERROR("Undefined shuffle mode. Allowed: a (all); h (horizontal), v (vertical), d (dinucleotide)");
    }
  }
  maf.writeMAF(cout);
  ifs.close();
  MainTools::parameterOutput(cout, argc, argv); // output of parameters
}

