// --*- C++ -*------x---------------------------------------------------------
// $Id: Gaussian1D.h,v 1.1.1.1 2006/07/03 14:43:20 bindewae Exp $
//
// Class:           Gaussian1D
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     Implements simple storing of sum and sum of squares to compute average and 
//                  standard deviation
// Reviewed by:     -
// -----------------x-------------------x-------------------x-----------------

#ifndef __GAUSSIAN_H__
#define __GAUSSIAN_H__

// Includes
#include <math.h>
#include <iostream>

#include <debug.h>

/** Stores mean and standard deviation (equiv: sum, squaresum and num elements) of dataset 
    @author Eckart Bindewald
    @see    -
    @review - */
class Gaussian1D {
public:
  Gaussian1D() : numElements(0), sum(0.0), squareSum(0.0), avg(0.0), deviation(0.0),
  highest(0.0), lowest(0.0) { }

  Gaussian1D(unsigned int _num, double _avg, double _stddev) : numElements(_num),
    avg(_avg), deviation(_stddev), highest(0.0), lowest(0.0) { updateSumsFromAvg(); }
  
  Gaussian1D(const Gaussian1D& orig) { 
    copy(orig);
  }
  
  virtual ~Gaussian1D() { }
  
  /* OPERATORS */
  
  /** Assigment operator. */
  Gaussian1D& operator = (const Gaussian1D& orig) {
    if (this != &orig) {
      copy(orig);
    }
    return *this;
  }
  
  friend ostream& operator << (ostream& os, const Gaussian1D& rval) {
    if (rval.getNumElements() == 0) {
      os << "0 0.0 0.0" << endl;
    }
    else {
      os << rval.getNumElements() << " " << rval.getAverage() << " " << rval.getDeviation();
    }
    return os;
  }
  
  friend istream& operator >> (istream& is, Gaussian1D& rval) {
    unsigned int num;
    double avg, std;
    is >> num >> avg >> std;
    rval = Gaussian1D(num, avg, std); // needs variance instead of standard deviation
    return is;
  }

  /* PREDICATES */

  /** Is current state valid? */
  virtual bool isValid() const { 
    return numElements > 0; 
  }
  
  virtual double getAverage() const {
    return avg;
  }
  
  virtual double getDeviation() const {
    return deviation;
  }

  virtual double getVariance() const {
    return deviation * deviation;
  }

  virtual double getDeviationOfAverage() const {
    if (numElements == 0) {
      return getDeviation();
    }
    return deviation / sqrt(static_cast<double>(numElements));
  }


  virtual double getHighest() const {
    return highest;
  }

  virtual double getLowest() const {
    return lowest;
  }

  virtual unsigned int getNumElements() const {
    return numElements;
  }
  
  virtual double getSum() const {
    return sum;
  }

  virtual double getSquareSum() const {
    return squareSum;
  }
  
  virtual double isZScoreDefined() const {
    return ((numElements > 1) && (deviation > 0.0));
  }

  virtual double getZScore(double val) const {
    if (!isZScoreDefined()) {
      return 0.0;
    }
    return (val - avg) / deviation;
  }

  /* MODIFIERS */

  /**
   * adds single datapoint to gaussian distribution
   */
  void add(double val) {
    if (numElements == 0) {
      highest = val;
      lowest = val;
    }
    else {
      if (val > highest) {
	highest = val;
      }
      if (val < lowest) {
	lowest = val;
      }
    }
    ++numElements;
    sum += val;
    squareSum += val * val;
    updateAvgFromSums();
  }

  /**
   * adds info from other distribution to this one
   */
  void add(const Gaussian1D& g) {
    numElements += g.numElements;
    sum += g.sum;
    squareSum += g.squareSum;
    updateAvgFromSums();
  }

  /** 
   * resets content
   */
  void clear() {
    numElements = 0;
    sum = 0.0;
    squareSum = 0.0;
    avg = 0.0;
    deviation = 0.0;
  }


protected:

  /* OPERATORS  */

  /* PREDICATES */

  /* MODIFIERS  */

  void copy(const Gaussian1D& other) {
    numElements = other.numElements;
    sum = other.sum;
    squareSum = other.squareSum;
    deviation = other.deviation;
    avg = other.avg;
    highest = other.highest;
    lowest = other.lowest;
  }

private:
  /* OPERATORS  */

  /* PREDICATES */

  /* MODIFIERS  */

  void updateAvgFromSums() {
    if (numElements == 0) {
      avg = 0.0;
      deviation = 0.0;
    }
    else {
      avg = sum / numElements;;
      deviation = (squareSum/numElements) - (avg * avg);
      // deal with numerical inaccuracies:
      if (deviation < 0.0) {
        deviation = 0.0;
      }
      deviation = sqrt(deviation); // otherwise it would be variance
    }
  }

  void updateSumsFromAvg() {
    sum = avg * numElements;
    squareSum = numElements * ( (deviation * deviation) + (avg * avg));
  }

private:
  /* PRIVATE ATTRIBUTES */
  
  unsigned int numElements;

  double sum;
  
  double squareSum;

  double avg;

  double deviation;

  double highest;

  double lowest;

};

#endif /* __ACLASS_H__ */

