// --*- C++ -*------x---------------------------------------------------------
// $Id: editmatrix.cc,v 1.20 2011-10-12 11:24:54 eckart Exp $
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2011-10-12 11:24:54 $
//
// Description:     
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <debug.h>
#include <GetArg.h>
#include <FileName.h>
#include <vectornumerics.h>
#include <Random.h>
#include <MatrixTools.h>

#define EDITMATRIX_VERSION "0.8.0"

void
helpOutput(ostream& os)
{
  os << "editmatrix version " << EDITMATRIX_VERSION << endl;
  os << "usage: editmatrix -i inputfile --of outputformat --subset vector --verbose n" << endl;
  os << "other options: " << endl
     << "--diagonal value" << endl
     << "--error value : error tolerance (0.01 is default)" << endl
     << "--l1 value    : low theshold (used for output format 16)" << endl
     << "--l2 value    : high theshold (used for output format 16)" << endl
     << "--monotonize 1|0 : if 1, run monotonization algorithm" << endl
     << "--modulate filename : filename with modulation values (list of index value pairs)" << endl
     << "--norm mode  : 0: nothing, 1: probability normalize" << endl
     << "--pairs id1x id1y id2x id2y ..." << endl
     << "--rescale filename" << endl
     << "--rescale2 filename" << endl
     << "--reverse-columns 0|1 " << endl
     << "--reverse-rows 0|1 " << endl
     << "--size n " << endl
     << "--step n" << endl
     << "--smooth gauss-stddev" << endl
     << "--transpose 0|1" << endl
     << "--winner 0|1" << endl;
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}


void
generateKnnPoints(ostream& os, 
		  int knnNum, 
		  double prob,
		  double posx, 
		  double posy, 
		  double deltax,
		  double deltay)
{
  Random& rnd = Random::getInstance();
  double px, py;
  for (int i = 0; i < knnNum; ++i) {
    px = posx + (rnd.getRandf()-0.5)*deltax;
    py = posy + (rnd.getRandf()-0.5)*deltay;
    os << px << " " << py << " ";
    if (rnd.getRandf() < prob) {
      os << "1";
    }
    else {
      os << "0";
    }
    os << endl;
  }
}

void
generateKnnPoints(ostream& os, 
		  int knnNum, 
		  const Vec<Vec<double> >& matrix, 
		  double minx, 
		  double miny, 
		  double deltax,
		  double deltay)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      generateKnnPoints(os, knnNum, matrix[i][j],  i * deltax + minx, j * deltay + miny, deltax, deltay);
    }
  }
}

Vec<Vec<double> >
readListMatrix(istream& is)
{
  string line = getLine(is);
  Vec<string> words = getTokens(line);
  ERROR_IF(words.size() != 2,
	   "Matrix dimenstions expected in first line!");
  unsigned dx = stoui(words[0]);
  unsigned dy = stoui(words[1]);
  ERROR_IF((dx == 0) || (dy == 0),
	   "Matrix dimensions have to be larger zero!");
  Vec<Vec<double> > result(dx, Vec<double>(dy, 0.0));
  while (is) {
    line = getLine(is);
    words = getTokens(line);
    if (words.size() == 0) {
      continue;
    }
    ERROR_IF(words.size() != 3,
	     "Bad data line, 3 entries expected!");
    unsigned int ix = stoui(words[0]);
    unsigned int iy = stoui(words[1]);
    ERROR_IF((ix == 0) || (ix > result.size()), "Bad data line, x value out of range!");
    --ix;
    ERROR_IF((iy == 0) || (iy > result[ix].size()), "Bad data line, x value out of range!");
    --iy;
    double val = stod(words[2]);
    result[ix][iy] = val;
  }
  return result;
}

Vec<Vec<double> >
readListMatrix2(istream& is, double step, unsigned int size)
{
  ERROR_IF(size < 1,
	   "Matrix dimensions have to be larger zero!");
  ERROR_IF(step <= 0.0,
	   "Matrix step size must be larger zero!");
  Vec<Vec<double> > result(size, Vec<double>(size, 0.0));
  while (is) {
    string line = getLine(is);
    Vec<string> words = getTokens(line);
    if (words.size() == 0) {
      continue;
    }
    ERROR_IF(words.size() != 3, "Bad data line, 3 entries expected!");
    unsigned int ix = static_cast<unsigned int>(stod(words[0])/step + 0.01);
    unsigned int iy = static_cast<unsigned int>(stod(words[1])/step + 0.01);
    double val = stod(words[2]);
    if ((ix < result.size()) && (iy < result[ix].size())) {
      result[ix][iy] = val;
    }
  }
  return result;
}

/** parses postscript file generated by RNAfold with option -p */
Vec<Vec<double> >
readRNAfoldPostscript(istream& is, bool squareMode) {
  // find sequence, deduce size:
  string line;
  Vec<string> words;
  do {
    line = getLine(is);
  }
  while (is && (line.substr(0, 9).compare("/sequence") != 0));
  ERROR_IF(!is, "Could not find sequence header!");
  string sequence = getLine(is);
  while (sequence[sequence.size()-1] == '\\') { // check for trailing '\' character
    sequence = sequence.substr(0, sequence.size()-1) + getLine(is); // delete last character and add new line
  }
  // find "\" character:
  for (int i = 0; i < static_cast<int>(sequence.size()); ++i) {
    char c = sequence[i];
    if ((c == '\\') || (c == ')')) {
      sequence = sequence.substr(0, i);
    }
  }
  // cerr << "Scanned sequence: " << sequence << " with " << sequence.size() << " characters!" << endl;
  unsigned int numRes = sequence.size(); // subtract 1 because last character is "\"
  Vec<Vec<double> >  result(numRes, Vec<double>(numRes, 0.0));
  while (is) {
    line = getLine(is);
    words = getTokens(line);
    if ((words.size() == 4) && (words[3].compare("ubox") == 0)) {
      unsigned int id1 = stoui(words[0])-1;
      unsigned int id2 = stoui(words[1])-1;
      result[id1][id2] = stod(words[2]);
      if (squareMode) {
	result[id1][id2] *= result[id1][id2]; // Postscript file contains square roots of probabilities!
      }
      result[id2][id1] = result[id1][id2];
    }
  }
  return result;
}

void
monotomizeFunction(Vec<double>& v)
{
  double last = v[0];
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] < last) {
      v[i] = last;
    }
    else {
      last = v[i];
    }
  }
}
void
monotomizeFunctionDown(Vec<double>& v)
{
  if (v.size() < 2) {
    return;
  }
  double last = v[v.size()-1];
  for (int i = v.size()-2; i >= 0; --i) {
    if (v[i] > last) {
      v[i] = last;
    }
    else {
      last = v[i];
    }
  }
}

void
monotomizeMatrixRows(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    monotomizeFunction(matrix[i]);
  }
}

void
monotomizeMatrixRowsDown(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    monotomizeFunctionDown(matrix[i]);
  }
}

void
monotomizeMatrixColumns(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix[0].size(); ++i) {
    Vec<double> col = getColumn(matrix, i);
    monotomizeFunction(col);
    setColumn(matrix, col, i);
  }
}

void
monotomizeMatrixColumnsDown(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix[0].size(); ++i) {
    Vec<double> col = getColumn(matrix, i);
    monotomizeFunctionDown(col);
    setColumn(matrix, col, i);
  }
}

bool
isMonoton(const Vec<double>& v, double err)
{
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] + err < v[i-1]) {
      return false;
    }
  }
  return true;
}

bool
isMonoton(const Vec<Vec<double> >& v, double err)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (!isMonoton(v[i], err)) {
      return false;
    }
  }
  for (unsigned int i = 0; i < v[0].size(); ++i) {
    if (!isMonoton(getColumn(v,i), err)) {
      return false;
    }
  }
  return true;
}


void
monotomizeMatrix(Vec<Vec<double> >& matrix, double err)
{
  int iter = 0;
  do {
    Vec<Vec<Vec<double> > > matrices(4, matrix);
    double norm = 1.0 / static_cast<double>(matrices.size()); 
    monotomizeMatrixRows(matrices[0]);
    monotomizeMatrixRowsDown(matrices[1]);
    monotomizeMatrixColumns(matrices[2]);
    monotomizeMatrixColumnsDown(matrices[3]);
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	matrix[i][j] = 0.0;
	for (unsigned int k = 0; k < 4; ++k) {
	  matrix[i][j] += matrices[k][i][j];
	}
	matrix[i][j] *= norm;
      }
    }
    if (iter > 1000000) {
      cout << "Could not find monoton matrix!" << endl;
      break;
    }
  }
  while (!isMonoton(matrix, err));
}

Vec<Vec<double> >
smoothMatrixGaussian(const Vec<Vec<double> >& matrix, double d, 
		     unsigned int nMax)
{
  Vec<Vec<double> > result = matrix;
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    result[i] = smoothGaussian(result[i], d, nMax);
  }
  for (unsigned int i = 0; i < matrix[0].size(); ++i) {
    Vec<double> col = getColumn(result, i);
    col = smoothGaussian(col, d, nMax);
    setColumn(result, col, i);
  }
  return result;
}

/** rescales value v.
 * assumes pairs of x,y values, such that closest x values larger than v is used to return y[x]
 */
double
rescale(double v,
	const Vec<double>& rescaleX,
	const Vec<double>& rescaleY)
{
  // find x value greater than v:
  for (unsigned int i = 0; i < rescaleX.size(); ++i) {
    if (rescaleX[i] >= v) {
      return rescaleY[i];
    }
  }
  // v is larger than all supplied x -values: use last y value
  return rescaleY[rescaleY.size()-1];
}

void
rescaleMatrix(Vec<Vec<double> >& matrix, 
	      const Vec<double>& rescaleX,
	      const Vec<double>& rescaleY)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      matrix[i][j] = rescale(matrix[i][j], rescaleX, rescaleY);
    }
  }
}

int
findStemStretchLen(const Vec<Vec<double> >& matrix,
		   int xOrig, int yOrig, double thresh, int diagLim)
{
  int x = xOrig;
  int y = yOrig;
  if (matrix[x][y] < thresh) {
    return 0;
  }
  int len1 = 1;
  for (len1 = 1; ; ++len1) {
    x = xOrig + len1;
    y = yOrig - len1;
    if ((abs(x-y) < diagLim) || (x >= static_cast<int>(matrix.size())) 
	|| (y < 0)) {
      break;
    }
    if (matrix[x][y] < thresh) {
      break;
    }
  }
  --len1;
  int len2 = 1;
  for (len2 = 1; ; ++len2) {
    x = xOrig - len2;
    y = yOrig + len2;
    if ((abs(x-y) < diagLim) || (y >= static_cast<int>(matrix.size())) 
	|| (x < 0)) {
      break;
    }
    if (matrix[x][y] < thresh) {
      break;
    }
  }
  --len2;
  return 1 + len1 + len2;
}

void
rescaleMatrix2(Vec<Vec<double> >& matrix, 
	       const Vec<Vec<double> >& rescaleLenX,
	       const Vec<Vec<double> >& rescaleLenY,
	       int diagLim)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      // find stem length:
      int len = findStemStretchLen(matrix, i,j, matrix[i][j], diagLim);
      if (len >= static_cast<int>(rescaleLenX.size())) {
	len = rescaleLenX.size()-1;
      }
      matrix[i][j] = rescale(matrix[i][j], rescaleLenX[len], rescaleLenY[len]);
    }
  }
}

void
setMatrixDiagonal(Vec<Vec<double> >& matrix, int diagZero, double val)
{
  for (int i = 0; i < static_cast<int>(matrix.size()); ++i) {
    for (int j = i-diagZero; (j <= i+diagZero) ; ++j) {
      if ((j < 0) || (j >= static_cast<int>(matrix.size()))) {
	continue;
      }
      matrix[i][j] = val;
      matrix[j][i] = val;
    }
  }
}

/** returns largest element */
double
elementMaximum(const Vec<Vec<double> >& matrix) {
  double result = matrix[0][0];
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      if (matrix[i][j] > result) {
	result = matrix[i][j];
      }
    }
  }
  return result;
}

/** returns smalles element */
double
elementMinimum(const Vec<Vec<double> >& matrix) {
  double result = matrix[0][0];
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      if (matrix[i][j] < result) {
	result = matrix[i][j];
      }
    }
  }
  return result;
}

/** computes sum of row maxima */
double
computeRowMaximaSum(const Vec<Vec<double> >& matrix) {
  double s = 0.0;
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    s += matrix[i][maxElement(matrix[i])];
  }
  return s;
}

/** computes fraction of high and low confidence base pairs */
void
writeAverageProperties(ostream& os, const Vec<Vec<double> >& matrix,
		       double threshLow,  double threshHigh)
{
  unsigned int lowCount = 0;
  unsigned int highCount = 0;
  double avgRowSum = elementSum(matrix) / matrix.size();
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    unsigned int idx = findMaxIndex(matrix[i]);
    double val = matrix[i][idx];
    if (val > threshLow) {
      ++lowCount;
    }
    if (val > threshHigh) {
      ++highCount;
    } 
  }
  double maximaSum = computeRowMaximaSum(matrix);
  double lowFrac = lowCount / static_cast<double>(matrix.size());
  double highFrac = highCount / static_cast<double>(matrix.size());
  os << avgRowSum << " " << lowFrac << " " << highFrac << " "
     << maximaSum << " " << (maximaSum/matrix.size()) << endl;
}

void
writeWMatchFormatVertex(ostream& os, 
			MatrixTools::size_type n,
			const MatrixTools::matrix_type& matrix,
			double limitVal) {
  // count number of connected elements:
  MatrixTools::size_type numConn = MatrixTools::countThresholdElements(matrix, limitVal, n);
  os << numConn << " A 0 0" << endl; // degree label coordx coordy
  for (MatrixTools::size_type i = 0; i < matrix[n].size(); ++i) {
    if ((i != n) && (matrix[n][i] >= limitVal)) {
      os << (i+1) << " " << static_cast<int>(1000.0 * matrix[n][i]) << endl;
    }
  }
}

void
writeWMatchFormat(ostream& os, 
		  const Vec<Vec<double> >& matrix,
		  double limitVal)
{
  MatrixTools::size_type numEdges = MatrixTools::countThresholdElements(matrix, limitVal) / 2;
  MatrixTools::size_type numVertices = matrix.size();
  MatrixTools::size_type numVerticesUsed = numVertices;
  bool unevenFlag = false;
  if ((numVertices % 2) != 0) {
    unevenFlag = true;
    numVerticesUsed++; // workaround: wmatch can handle only even number of vertices
  }
  os << numVerticesUsed << " " << numEdges << " U" << endl;
  for (MatrixTools::size_type i = 0; i < numVertices; ++i) {
    writeWMatchFormatVertex(os, i, matrix, limitVal);
  }
  if (unevenFlag) {
    os << "0 A 0 0" << endl; // add dummy
  }
}

/** Reads modulation values in form
 * matrixsize
 * numEntries
 * id (starting from 1) value
 * id value
 * ...
 * id value
 */
Vec<double> 
readModulateValues(istream& is) {
  int n;
  int numEntries;
  is >> n >> numEntries;
  Vec<double> result(n, 1.0); // by default use factor 1.0
  for (int i = 0; i < numEntries; ++i) {
    Vec<double>::size_type index;
    is >> index;
    --index;
    ERROR_IF(!is, "Error reading modulate values!");
    ERROR_IF(index < 0 || index >= result.size(),
	     "Illegal column index in modulation values!");
    is >> result[index];
    // cout << "Result of line: " << (i+1) << " : " << index << " " << result[index] << endl;
  }
  return result;
}

unsigned long
analyzeTriangleInequality(ostream& os, const Vec<Vec<double> >& matrix) {
  Vec<Vec<double> >::size_type nRows = matrix.size();
  Vec<Vec<double> >::size_type nCols = matrix[0].size();
  ERROR_IF(nRows != nCols, "Error while analyzing triangular inequality. Matrix has to be square and symmetric!");
  unsigned long violationCount = 0;
  for (Vec<Vec<double> >::size_type i = 0; i < nRows; ++i) {
    cout << "# Working on first item " << (i + 1) << endl;
    for (Vec<Vec<double> >::size_type j = (i+1); j < nRows; ++j) {
      double dij = matrix[i][j];
      for (Vec<Vec<double> >::size_type k = (j+1); k < nRows; ++k) {
	double dik = matrix[i][k];
	double djk = matrix[j][k];
        if ( ((dij + djk) < dik) || ((djk + dik) < dij) || ((dij + dik) < djk)) {
	  os << "Triangular inequality violated for triple " << (i+1) << " " << (j+1) << " " << (k + 1) << " ; distances dij, dik, djk: " << dij << " " << dik << " " << djk << endl;
	  ++violationCount;
	  ERROR_IF(violationCount > 100000, "More than 100000 triangle inequality violations found. Quitting prematurely.");
	} 
      }      
    }
  }
  return violationCount;
}

int
main(int argc, char ** argv)
{

  bool helpMode;
  int argcFile = 0;
  char ** argvFile = 0;
  int diagLim = 3;
  int diagZero = 0;
  int inputFormat = 3;
  int knnNum = 0;
  int monotomizeMode = 0;
  int normMode = 0;
  int outputFormat = 3;
  int reverseColumnMode = 0;
  int reverseRowMode = 0;
  int transposeMode = 0;
  int winnerDiagBorder = 1;
  int winnerMode = 0;
  unsigned int matrixSize = 0;
  double deltax = 0.1;
  double deltay = 0.1;
  double diagonalVal = 0.0;
  double err = 0.01;
  double g = 0.0; //gaussian smoothing constant
  double winnerCutoff = 0.01; // used for speedup of winner takes all 

  double minx = 0.0;
  double miny = 0.0;
  double stepSize = 0;
  double threshLow = 0.1;
  double threshHigh = 0.7;
  unsigned int verboseLevel = 0;
  string commandFileName;
  string subsetString;
  string logFileName; //  = "mainprogramtemplate.log";
  string rescaleFileName;
  string modulateFileName;
  string rescaleFileName2;
  string rootDir = ".";
  string inputFileName;
  double limitVal = 0.0;
  Vec<double> rescaleX, rescaleY;
  Vec<Vec<double> > rescaleLenX, rescaleLenY;
  Vec<unsigned int> pairs;
  Vec<unsigned int> subset;

  getArg("-help", helpMode, argc, argv);

  if ((argc < 2) || helpMode)  {
    helpOutput(cout);
    exit(0);
  }

  getArg("-root", rootDir, argc, argv, rootDir);
  addSlash(rootDir);

  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);

  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.");
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }
  
  getArg("-diagonal", diagonalVal, argcFile, argvFile,  diagonalVal);
  getArg("-diagonal", diagonalVal, argc, argv,  diagonalVal);
  getArg("-diag-zero", diagZero, argcFile, argvFile,  diagZero);
  getArg("-diag-zero", diagZero, argc, argv,  diagZero);
  getArg("-error", err, argcFile, argvFile, err);
  getArg("-error", err, argc, argv, err);
  getArg("i", inputFileName, argc, argv);
  getArg("l", limitVal, argc, argv, limitVal);
  getArg("-l1", threshLow, argcFile, argvFile, threshLow);
  getArg("-l1", threshLow, argc, argv, threshLow);
  getArg("-l2", threshHigh, argcFile, argvFile, threshHigh);
  getArg("-l2", threshHigh, argc, argv, threshHigh);
  getArg("-if", inputFormat, argcFile, argvFile, inputFormat);
  getArg("-if", inputFormat, argc, argv, inputFormat);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("-knn", knnNum, argc, argv, knnNum);
  getArg("-monotonize", monotomizeMode, argc, argv, monotomizeMode);
  getArg("-modulate", modulateFileName, argc, argv, modulateFileName);
  getArg("-minx", minx, argc, argv, minx);
  getArg("-miny", miny, argc, argv, miny);
  getArg("-deltax", deltax, argc, argv, deltax);
  getArg("-deltay", deltay, argc, argv, deltay);
  getArg("-norm", normMode, argcFile, argvFile, normMode);
  getArg("-norm", normMode, argc, argv, normMode);
  getArg("-of", outputFormat, argcFile, argvFile, outputFormat);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("-pairs", pairs, argc, argv);
  convert2InternalCounting(pairs);
  getArg("-rescale", rescaleFileName, argc, argv, rescaleFileName);
  getArg("-rescale2", rescaleFileName2, argc, argv, rescaleFileName2);
  getArg("-reverse-columns", reverseColumnMode, argc, argv, reverseColumnMode);
  getArg("-reverse-rows", reverseRowMode, argc, argv, reverseRowMode);
  getArg("-size", matrixSize, argc, argv, matrixSize);
  getArg("-step", stepSize, argc, argv, stepSize);
  getArg("-subset", subsetString, argc, argv, subsetString);
  subset = parseStringToVector(subsetString);
  convert2InternalCounting(subset);
  sort(subset.begin(), subset.end());
  getArg("-smooth", g, argcFile, argvFile, g);
  getArg("-smooth", g, argc, argv, g);
  getArg("-transpose", transposeMode, argc, argv, transposeMode);
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);
  getArg("-winner", winnerMode, argcFile, argvFile, winnerMode);
  getArg("-winner", winnerMode, argc, argv, winnerMode);

  if (logFileName.size() > 0) {
    ofstream logFile(logFileName.c_str(), ios::app);
    parameterOutput(logFile, argc, argv);
    if (argcFile > 1) {
      logFile << "Parameters from command file: ";
      parameterOutput(logFile, argcFile, argvFile);
    }
    logFile.close();
  }

  /***************** MAIN PROGRAM *****************************/

  if (rescaleFileName.size() > 0) {
    ifstream rescaleFile(rescaleFileName.c_str());
    ERROR_IF(!rescaleFile, "Error opening rescale file!");
    unsigned int numEntries;
    rescaleFile >> numEntries;
    rescaleX = Vec<double>(numEntries, 0.0);
    rescaleY = Vec<double>(numEntries, 0.0);
    for (unsigned int i = 0; i < numEntries; ++i) {
      rescaleFile >> rescaleX[i] >> rescaleY[i];
    }
    rescaleFile.close();
  }

  if (rescaleFileName2.size() > 0) {
    ifstream rescaleFile2(rescaleFileName2.c_str());
    ERROR_IF(!rescaleFile2, "Error opening rescale file2!");
    unsigned int numEntries, numLen;
    rescaleFile2 >> numLen >> numEntries;
    rescaleLenX = Vec<Vec<double> >(numLen, Vec<double>(numEntries, 0.0));
    rescaleLenY = Vec<Vec<double> >(numLen, Vec<double>(numEntries, 0.0));
    for (unsigned int j = 0; j < numLen; ++j) {
      for (unsigned int i = 0; i < numEntries; ++i) {
	rescaleFile2 >> rescaleLenX[j][i] >> rescaleLenY[j][i];
      }
    }
    rescaleFile2.close();
  }

  Vec<Vec<double> > matrix;
  ifstream inputFile(inputFileName.c_str());
  ERROR_IF(!inputFile, "Error opening input file!");
  switch (inputFormat) {
  case 1: 
    matrix = readListMatrix(inputFile);
    break;
  case 3:
    matrix = readPlainMatrix(inputFile);
    break;
  case 4: 
    matrix = readListMatrix2(inputFile, stepSize, matrixSize);
    break;
  case 5:
    ERROR("Input mode 5 (unsquared RNAfold postscript probability matrix) not supported anymore! Use --if 6 instead.");
    matrix = readRNAfoldPostscript(inputFile, false); // backwards compatibility: use square root of probabilities
    break;
  case 6:
    matrix = readRNAfoldPostscript(inputFile, true); // use probabilities
    break;
  default: ERROR("Unknown matrix read mode!");
  }
  inputFile.close();

  ERROR_IF(matrix.size() == 0, "No matrix defined!");

  if (diagZero > 0) {
    setMatrixDiagonal(matrix, diagZero, 0.0);
  }

  // sets diagonal to certain value
  if (isPresent("-diagonal", argc, argv)) {
    setMatrixDiagonal(matrix, 0, diagonalVal);
  }

  if (transposeMode) {
    cout << "Transposing matrix!" << endl;
    matrix = matrixTranspose2(matrix);
  }

  if (modulateFileName.size() > 0) {
    ifstream mFile(modulateFileName.c_str());
    ERROR_IF(!mFile, "Error opening modulating file: " + modulateFileName);
    Vec<double> modulateValues = readModulateValues(mFile);
    ERROR_IF(modulateValues.size() != matrix.size(),
	     "Matrix values have to have same size as modulation vector!");
    mFile.close();
    modulateMatrix(matrix, modulateValues);
  }

  if (subset.size() > 0) {
    matrix = getMatrixSubset(matrix, subset);
    ERROR_IF(matrix.size() != matrix[0].size(),
	     "Result matrix of subset is not quadratic!");
  }

  if (reverseRowMode) {
    reverse(matrix.begin(), matrix.end());
  }
  if (reverseColumnMode) {
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      reverse(matrix[i].begin(), matrix[i].end());
    }
  }
  
  if (rescaleX.size() > 0) {
    if (verboseLevel > 1) {
      cout << "Rescaling matrix!" << endl;
    }
    rescaleMatrix(matrix, rescaleX, rescaleY);
  }

  if (rescaleLenX.size() > 0) {
    if (verboseLevel > 1) {
      cout << "Rescaling matrix!" << endl;
    }
    rescaleMatrix2(matrix, rescaleLenX, rescaleLenY, diagLim);
  }

  if (monotomizeMode) {
    if (verboseLevel > 0) {
      cout << "Monotimizing matrix!" << endl;
    }
    monotomizeMatrix(matrix, err);
  }

  if (g > 0.0) {
    matrix = smoothMatrixGaussian(matrix,g, 5);
  }

  switch (normMode) {
  case 0: // do nothing
    break;
  case 1:
    cout << "Normalizing such that elements of each row sum up to one!" << endl;
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      probabilityNormalize(matrix[i]); // normalize such that elements of each row sum to one
    }
    break;
  case 2: // convert to z-score
    transformToZScores(matrix);
    break;
  default: ERROR("Unknown normalization mode!");
  }

  // possible winner takes all filter (corresponds to each position having a contact with only one other position
    // optional winner-takes-all operation:
  if (winnerMode) {
    if (verboseLevel > 0) {
      cout << "Applying winner-takes-all filter to matrix!" << endl;
    }
    winnerTakesAll(matrix, winnerDiagBorder, winnerCutoff);
  }


  switch (outputFormat) {
  case 0:
    break; // do nothing
  case 1:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	if (matrix[i][j] > limitVal) {
	  cout << i + 1 << " " << j + 1 << " " << matrix[i][j] << endl;
	}
      }
    }
    break;
  case 3:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	cout << matrix[i][j] << " ";
      }
      cout << endl;
    }
    break;
  case 4:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j <= i; ++j) {
	if (matrix[i][j] > limitVal) {
	  cout << i + 1 << " " << j + 1 << " " << matrix[i][j] << endl;
	}
      }
    }
    break;
  case 5:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = i; j < matrix[i].size(); ++j) {
	if (matrix[i][j] > limitVal) {
	  cout << i + 1 << " " << j + 1 << " " << matrix[i][j] << endl;
	}
      }
    }
    break;
  case 6:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	// if (matrix[i][j] > limitVal) {
	cout << (deltax*i  + minx) << " " << (deltay*j + miny) << " " << matrix[i][j] << endl;
	  // }
      }
    }
    break;
  case 7: 
    cout << matrix << endl;
    break;
  case 8:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	// if (matrix[i][j] > limitVal) {
	cout << i * stepSize << " " << j * stepSize << " " << matrix[i][j] << endl;
	// }
      }
    }
    break;
  case 9:
    for (unsigned int i = 0; i + 1 < pairs.size(); i+=2) {
      cout << i + 1 << " " << pairs[i] + 1 << " " << pairs[i+1] + 1 << " "
	   << matrix[pairs[i]][pairs[i+1]] << endl;
    }
    break;
  case 10:
    // row - sums
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      cout << (i + 1) << " " << elementSum(matrix[i]) << endl;
    }
    break;
  case 11: // sum
    cout << elementSum(matrix) << endl;
    break;
  case 12: // average row sum:
    cout << elementSum(matrix)/matrix.size() << endl;
    break;
  case 13: // write element maximum:
    cout << elementMaximum(matrix) << endl;
    break;
  case 14: // write element minimum:
    cout << elementMinimum(matrix) << endl;
    break;
  case 15: // write difference between element maximum and minimum:
    cout << (elementMaximum(matrix)-elementMinimum(matrix)) << endl;
    break;
  case 16: 
    writeAverageProperties(cout, matrix, threshLow, threshHigh);
    cout << endl;
    break;
  case 17:
    // row - maxima
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      cout << (i + 1) << " " << matrix[i][maxElement(matrix[i])] << endl;
    }
    break;
  case 18:
    // sum of row - maxima
    cout << computeRowMaximaSum(matrix) << endl;
    break;
  case 19:
    // average sum of row - maxima
    cout << (computeRowMaximaSum(matrix)/matrix.size()) << endl;
    break;
  case 20:
    // average sum of row - maxima
    cout << elementSquareSum(matrix) << endl;
    break;
  case 21:
    writeWMatchFormat(cout, matrix, limitVal);
    cout << endl;
    break;
  case 22:
    // square root of average square sum of elements
    cout << sqrt(elementSquareSum(matrix)) << endl;
    break;
  case 23:
    // square root of average square sum of elements
    cout << sqrt(elementSquareSum(matrix))/matrix.size() << endl;
    break;
  case 24:
    // square root of square sum of elements divided by one dimension
    // best norm for RNA structure similarities. Flory theory?
    cout << sqrt(elementSquareSum(matrix)/matrix.size()) << endl;
    break;
  case 25:
    // row - sums in one row
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      cout << elementSum(matrix[i]) << " ";
    }
    cout << endl;
    break;
  case 26:
    {
      cout << "# Searching for triplets that violate the triangle-inequality..." << endl;
      unsigned long violCount = analyzeTriangleInequality(cout, matrix);
      cout << "Number of violations to triangle inequality: " << violCount << endl;
      cout << "# ...done" << endl;
    }
    break;
  default:
    ERROR("Unknown matrix output format!");
  }
  if (knnNum > 0) {
    generateKnnPoints(cout, knnNum, matrix, minx, miny, deltax, deltay);    
  }
    
  return 0;
}
