#include <entropyfunctions.h>
#include <vectornumerics.h> // my utility functions

// #define ENTROPY_DEBUG

/** vector with probabilities of different entities.
    values must be between 0 and 1
*/
double
entropyShannon(const Vec<double>& v) 
{
  const double LOG2 = log(2.0);
  double sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    ASSERT((v[i] >= 0.0) && (v[i] <= 1.0));
    if (v[i] > 0) { // ignore if equal zero
      sum += - v[i] * log(v[i]) / LOG2;
    }
  }
  return sum;
}


/** 
 * @param v: arbitrary vector with double values.
 * from that the routine computes a normalized histogram and using that the Shannon entropy
*/
double
entropyOfHistogram(const Vec<double>& v) 
{
  // find minimum element:
  double minVal = v[minElement(v)];
  double maxVal = v[maxElement(v)];
  if (maxVal <= minVal) {
    return 0.0;
  }
  unsigned int numBins = v.size();
  ASSERT(numBins > 0);
  double delta = (maxVal - minVal) / numBins;
  Vec<unsigned int> histogram = computeHistogram(v, minVal, delta, numBins);
  Vec<double> probs(histogram.size(), 0.0);
  for (unsigned int i = 0; i < probs.size(); ++i) {
    probs[i] = static_cast<double>(histogram[i]) / v.size();
  }
  return entropyShannon(probs);
}

/**
 * returns true if probability p
 */
bool
chooseWithProbability(double p, Random& rnd) {
  return (rnd.getRandf() < p);
}

/**
 * use with chooseBoltzmannMax
 */
double
computeBaseProbabilityMax(const Vec<double>& scores, double kt) {
  PRECOND(kt > 0.0);
  unsigned int bestIndex = findMaxIndex(scores) ;
  double bestScore = scores[bestIndex];
  double sum = 0.0;
  for (unsigned int i = 0; i < scores.size(); ++i) {
    ASSERT(bestScore - scores[i] >= 0.0);
    sum += exp( - (bestScore - scores[i])/kt);
  }
  ASSERT(sum >= 1.0);
  double result = 1.0 / sum;
  if (result > 1.0) {
    result = 1.0;
  }
  POSTCOND((result  >= 0.0) && (result <= 1.0));
  return result;
}

/**
 * use with chooseBoltzmannMax
 */
double
computeBaseProbabilityMin(const Vec<double>& scores, double kt) {
  PRECOND(kt > 0.0);
  unsigned int bestIndex = findMinIndex(scores) ;
  double bestScore = scores[bestIndex];
  double sum = 0.0;
  for (unsigned int i = 0; i < scores.size(); ++i) {
    ASSERT(scores[i] - bestScore >= 0.0);
    sum += exp( - (scores[i] - bestScore)/kt);
  }
  ASSERT(sum >= 1.0);
  double result = 1.0 / sum;
  if (result > 1.0) {
    result = 1.0;
  }
  POSTCOND((result >= 0.0) && (result <= 1.0));
  return result;
}

/**
 * returns id of element according to Boltzmann statistics
 * assumes that fragment vec is ordered such that best (HIGHEST!) score is element zero
 * assumes NOT that the score is store in fragments[i].first
 */
/*
unsigned int
chooseBoltzmannElementHighestBest(const Vec<double>& scores, double kt)
{
  PRECOND(kt > 0.0);
  unsigned int nSize = scores.size();
  Vec<unsigned int> randomIndices = generateRandomIndexSubset(nSize, nSize, 0);
  unsigned int bestIndex = findMaxIndex(scores) ;
  double bestScore = scores[bestIndex];
  double newScore;
  for (unsigned int i = 0; i  < nSize; ++i) {
    newScore = scores[randomIndices[i]];
    if (rnd.getRandf() <= exp(- (bestScore - newScore) / kt)) {
      return i; // returns i'th element because Boltzmanm condition fulfilled
    }
  }
  return bestIndex; // return best element
}
*/

/**
 * returns id of element according to Boltzmann statistics
 * derived property of computeBaseProbabilityMax, totally correct implementation would use Roulette-Wheel
 * only approximate implementation, correct (but slower) solution would be to use chooseRouletteWheel 
 */
unsigned int
chooseBoltzmannElementHighestBest(const Vec<double>& scoresOrig, double kt)
{
  PRECOND(scoresOrig.size() > 0);
  PRECOND(kt > 0.0);
  Vec<double> scores = scoresOrig;
  // Vec<unsigned int> oldOrder = documentedSort2(scores);
  // reverse(scores.begin(), scores.end()); // highest element first
  // reverse(oldOrder.begin(), oldOrder.end());
  unsigned int bestIndex = findMaxIndex(scores) ;
  double bestScore = scores[bestIndex];
  // double p0 = computeBaseProbabilityMaxTransformProb(scores, kt); // scores contain unnormalized probabilities 
  for (unsigned int i = 0; i < scores.size(); ++i) {
    ASSERT(bestScore - scores[i] >= 0.0);
    scores[i] = exp( - (bestScore - scores[i])/kt);
  }
  unsigned int result = chooseRouletteWheel(scores);
#ifdef ENTROPY_DEBUG
  cout << "Result of chooseBoltzmannElementHighest best: " << kt << " " << scoresOrig.size() << " "
       << "best: " << bestIndex << " " << bestScore << " chosen: " << result << " " << scores[result] << " "
       << (bestScore + (kt * log(scores[result]))) << endl;
#endif
  POSTCOND(result < scoresOrig.size());
  return result;

//    if (chooseWithProbability(p0, rnd)) {
//      return oldOrder[0]; // return index of highest scoring element
//    }
//    for (unsigned int i = 1; i  < nSize; ++i) {
//      newScore = scores[i];
//      if (chooseWithProbability(p0 * exp(- (bestScore - newScore) / kt), rnd)) {
//        return oldOrder[i]; // returns i'th element because Boltzmanm condition fulfilled
//      }
//    }
//    return rnd.getRand() % nSize; // return random element
}

/**
 * returns id of element according to Boltzmann statistics
 * derived property of computeBaseProbabilityMax, totally correct implementation would use Roulette-Wheel
 * only approximate implementation, correct (but slower) solution would be to use chooseRouletteWheel 
 */
unsigned int
chooseBoltzmannElementLowestBest(const Vec<double>& scoresOrig, double kt)
{
  PRECOND(scoresOrig.size() > 0);
  PRECOND(kt > 0.0);
  Vec<double> scores = scoresOrig;
  // Vec<unsigned int> oldOrder = documentedSort2(scores);
  // reverse(scores.begin(), scores.end()); // highest element first
  // reverse(oldOrder.begin(), oldOrder.end());
  unsigned int bestIndex = findMinIndex(scores) ;
  double bestScore = scores[bestIndex];
  // double p0 = computeBaseProbabilityMaxTransformProb(scores, kt); // scores contain unnormalized probabilities 
  for (unsigned int i = 0; i < scores.size(); ++i) {
    ASSERT(bestScore - scores[i] >= 0.0);
    scores[i] = exp( - (scores[i] - bestScore)/kt);
  }
  unsigned int result = chooseRouletteWheel(scores);
  POSTCOND(result < scoresOrig.size());
  return result;

//    if (chooseWithProbability(p0, rnd)) {
//      return oldOrder[0]; // return index of highest scoring element
//    }
//    for (unsigned int i = 1; i  < nSize; ++i) {
//      newScore = scores[i];
//      if (chooseWithProbability(p0 * exp(- (bestScore - newScore) / kt), rnd)) {
//        return oldOrder[i]; // returns i'th element because Boltzmanm condition fulfilled
//      }
//    }
//    return rnd.getRand() % nSize; // return random element
}


/**
 * returns id of element according to Boltzmann statistics
 * assumes that fragment vec is ordered such that best (LOWEST!) score is element zero
 * assumes NOT that the score is store in fragments[i].first
 */
/*
unsigned int
chooseBoltzmannElementLowestBest(const Vec<double>& scores, double kt, Random& rnd)
{
  PRECOND(kt > 0.0);
  unsigned int nSize = scores.size();
  Vec<unsigned int> randomIndices = generateRandomIndexSubset(nSize, nSize, 0);
  unsigned int result = findMinIndex(scores);
  double bestScore = scores[result];
  double newScore;
  for (unsigned int i = 0; i  < nSize; ++i) {
    newScore = scores[randomIndices[i]];
    if (rnd.getRandf() <= exp(- (newScore - bestScore) / kt)) {
      result = i; // returns i'th element because Boltzmanm condition fulfilled
      break;
    }
  }
  POSTCOND(result < scores.size());
  return result; // return best element
}
*/
