#include <math.h>
// #include <Vector3D.h>
#include <geometry.h>
#include <debug.h>
#include <vectornumerics.h>
#include <clusterAlgorithms.h>

#ifndef PI 
#define PI 3.141592654
#endif

/* result is between 0 and 2PI */
double
phiFromXY(double x, double y)
{
  if (y == 0.0) {
    if (x >= 0.0) {
      return 0.0;
    }
    else {
      return PI;
    }
  }
  // programming bug !?? was atan(fabs(x) / fabs(y)) EB 2006
  double phi = atan(fabs(y) / fabs(x)); // treat as if quadrant I
  // phi is now in [0.. PI/2]
  // choose correct quadrant
  if (x < 0.0) { // quadrant II or III
    if (y < 0.0) { // quadrant III
      phi += PI;
    }
    else { // quadrant II
      phi = PI - phi;
    }
  }
  else if (y < 0.0) { // quadrant IV
    phi = (2 * PI) - phi;
  }
  POSTCOND((phi >= 0.0) && (phi <= 2*PI));
  return phi;
}

/* result is between -PI/2 and 3/2PI */
double
phiFromXYFast(double x, double y)
{
  if (y == 0.0) {
    return 0.0;
  }
  double phi = atan(x / y); 
  // phi is now in [-PI/2 .. PI/2]
  // choose correct quadrant
  if (x < 0.0) { // quadrant II or III
    phi = PI - phi;
  }
  return phi;
}

/* result is between -PI/2 and 3/2PI */
double
thetaFromXYZ(double x, double y, double z)
{
  if (z == 0.0) {
    return (0.5 * PI);
  }
  double theta;
  double rxy = sqrt((x*x) + (y*y));
  if (z > 0.0) {
    theta = atan( rxy / z );
  }
  else { // z < 0.0
    theta = PI - atan( rxy / fabs(z) );
  }
  POSTCOND( (theta >= 0.0) && (theta <= PI));
  return theta;
}

void
cartesianToPolar(double x, double y, double z,
		 double& r, double& theta, double& phi)
{
  // theta = acos(z / r); 
  //    if (x == 0.0) {
  //      phi = 0.0;
  //    }
  //    else {
  //      phi = atan(y/x); // not totally correct
  //    }
  // phi = acos(x / sqrt( ( r * r ) - ( z * z ) ) );
  //    if (z == 0.0) {
  //      theta = 0.0;
  //    }
  //    else {
  //      theta = atan(sqrt( (x*x) + (y*y) ) / z );
  //    }
  r = sqrt( (x * x) + (y * y) + (z * z) );
  if (r <= 0.0) {
    theta = 0.0;
    phi = 0.0;
    return;
  }
  phi = phiFromXY(x,y);
  theta = thetaFromXYZ(x, y, z);
}


/*
void
cartesianToPolar(const Vector3D& cart,
		 Vector3D& pol)
{
  cartesianToPolar(cart.x(), cart.y(), cart.z(), 
		   pol.x(), pol.y(), pol.z());
}
*/

/* transform polar coordinates to cartesion coordinates.
   as always, angles are measured in radians */
void
polarToCartesian(double &x, double& y, double& z,
		 double  r, double theta, double phi)
{
  x = r * sin(theta) * cos(phi);
  y = r * sin(theta) * sin(phi);
  z = r * cos(theta);
}

/*
  define rotation matrix M, such that
  M * u is parallel to v
*/
Matrix3D
generateAlignMatrix(const Vector3D& u, const Vector3D& v)
{
  if ((u.length() == 0.0) || (v.length() == 0.0)) {
    return Matrix3D(1.0, 1.0, 1.0); // nothing to do
  }
  Vector3D w = cross(u, v);
  double ang = angle(u, v);
  Matrix3D m = Matrix3D::rotationMatrix(w, ang);
  // result vector must be parallel
  POSTCOND( cross(v, m * u) < 0.01);
  return m;
}

// return volume of sphere
double
sphereVolume(double r)
{
  return (4.0 / 3.0) * PI * r * r * r;
}

// return intersecting volume (double-sided cap of sphere) of two spheres with radius r1, r2 and distance d
double
sphereIntersectionVolume(double r1, double r2, double d)
{
  double rDiff = r1 - r2;
  double rTot = r1 + r2;
  if (d > rTot) {
    return 0.0; // no intersection volume, spheres are too far apart
  }
  // one sphere fully contains the other:
  if ( ( r1 + d ) <= r2 ) {
    return sphereVolume(r1);
  }
  else if ( ( r2 + d ) <= r1 ) {
    return sphereVolume(r2);
  }
  double d2 = d * d;
  // radius of intersecting cap
  double rI = sqrt( ( (rTot * rTot) - d2) * (d2 - (rDiff * rDiff)) ) / ( 2.0 * d);
  double h1 = ( ( r2 * r2 ) - ( (r1 - d) * (r1 - d) ) ) / (2.0 * d);
  double h2 = ( ( r1 * r1 ) - ( (r2 - d) * (r2 - d) ) ) / (2.0 * d);

  double v1 = (PI/ 6.0) * ( 3.0 * rI * rI + h1 * h1) * h1; 
  double v2 = (PI/ 6.0) * ( 3.0 * rI * rI + h2 * h2) * h2; 
  double v = v1 + v2;
  POSTCOND(v >= 0.0);
  return v;
}


// return vector which contains n points belonging to line
// includes startt and end points a an b. n must be greater than one 
Vec<Vector3D>
linePoints(const Vector3D& a, const Vector3D& b, unsigned int n)
{
  //  cout << "strarting line point! ";
  if (n < 2) {
    n = 2;
  }
  Vec<Vector3D> result(n);
  Vector3D v = b - a;
  if (v.length() == 0.0) {
    return Vec<Vector3D>(n,a);
  }
  double divLen = 1.0 / static_cast<double>(n-1);
  v = v * divLen;
  for (unsigned int i = 0; i < n; ++i) {
    result[n] = a + v * static_cast<double>(i);
  }
  //  cout << "linepoints finished: " << result.size() << endl;
  return result;
}

Vec<Vector3D>
fillPoints(const Vec<Vector3D>& s, double grid)
{
  cout << "Starting fillPoints! " << s.size() << endl;
  if (s.size() < 2) {
    return Vec<Vector3D>();
  }
  unsigned int lineNum = (s.size() * (s.size() - 1)) / 2;
  cout << "generating " << lineNum << " lines."  << endl;
  Vec<Vec<Vector3D> > lines; // (lineNum);

  unsigned int lineCount = 0;
  for (unsigned int i = 1; i < s.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      // cout << "huhu!" << endl;
      unsigned n = static_cast<unsigned int>(vecDistance(s[i], s[j]) / grid);
      //  cout << "huhu2!" << endl;
      if (n < 2) { 
	n = 2;
      }
      // cout << "huhu3: " << n << endl;
      if (lineCount >= lines.size()) {
	break;
      }
      // cout << "huhu4: " << lineCount << endl;
      lines.push_back(linePoints(s[i], s[j], n));
      // cout << "huhu4b" << endl;
    }
  }
  cout << " flattening ... ";
  Vec<Vector3D> result = flatten(lines);
  cout << "end of fillPoints! " << result.size() << endl;
  return result;
}

// thin out dense points, not closer than grid
Vec<Vector3D>
thinPoints(const Vec<Vector3D>& s, double grid)
{
  cout << "Starting thinPOints! " << s.size() << endl;
  if (s.size() < 2) {
    return s;
  }
  Vec<Vector3D> work(s.size());
  work[0] = s[0];
  unsigned int pointCount = 1;  
  for (unsigned int i = 1; i < s.size(); ++i) {
    bool found = false;
    for (unsigned int j = 0; j < pointCount; ++j) {
      if (vecDistance(s[i], work[j]) < grid) {
	found = true;
	break;
      }
    }
    if (!found) {
      if (pointCount >= work.size()) {
	break;
      }
      else {
	work[pointCount++] = s[i];
      }
    }
  }
  cout << "ending thinPoints!" << endl;
  if (pointCount == 0) {
    return Vec<Vector3D>();
  }
  Vec<Vector3D> result =  vectorCut(work, pointCount);
  cout << "end of thinPoints! " << result.size() << endl;
  return result;
}

/* convert list of 3D vectors to distance matrix */
void
pointsToField(const Vec<Vector3D>& v, Vec<Vec<double> >& m)
{
  m.clear();
  unsigned int dim = v.size();
  Vec<double> mRow = Vec<double>(dim);
  m = Vec<Vec<double> > (dim, mRow);
  for (unsigned int i = 0; i < dim; ++i) {
    for (unsigned int j = 0; j <= i; ++j) {
      m[i][j] = vecDistance(v[i], v[j]);
      m[j][i] = m[i][j];
    }
  }
}

// use only surface points, which cannot be erased using eraser with radius r
Vec<Vector3D>
erasePoints(const Vec<Vector3D>& surfPoints, const Vec<Vector3D>& solidPoints, double eraserRadius)
{
  ERROR("This version of erasePoints is deprecated!");
  double radiusSquare = eraserRadius * eraserRadius;
  Vec<unsigned int> deleted(surfPoints.size(), 0U);
  unsigned int deleteCount = 0;
  for (unsigned int i = 0; i < surfPoints.size(); ++i) {
    // check if too close to solid point:
    bool found = false;
    for (unsigned int j = 0; j < solidPoints.size(); ++j) {
      if (vecDistanceSquare(surfPoints[i], solidPoints[j]) < radiusSquare) {
	found = true;
	break;
      }
    }
    // is a valid point for eraser:
    if (!found) {
      for (unsigned int j = 0; j < surfPoints.size(); ++j) {
	if (vecDistanceSquare(surfPoints[i], surfPoints[j]) < radiusSquare) {
	  if (deleted[j] == 0) {
	    ++deleteCount;
	    deleted[j] = 1;
	  }
	}
      }
    }
  }
  ERROR_IF(deleteCount > surfPoints.size(), 
	   "Internal error in line 305.");
  if (deleteCount == surfPoints.size()) {
    Vec<Vector3D> tmpResult;
    return tmpResult;
  }
  Vec<Vector3D> result(surfPoints.size() - deleteCount);
  unsigned int pc = 0;
  for (unsigned int i = 0; i < surfPoints.size(); ++i) {
    if (deleted[i] == 0) {
      ASSERT(pc < result.size());
      result[pc++] = surfPoints[i];
    }
  }
  return result;
}

Vec<Vector3D>
generateCubeGridPoint(const Vector3D& minVec, 
		      const Vector3D& maxVec, double step)
{
  PRECOND(step > 0.0);
  unsigned int nx = static_cast<int>((maxVec.x() - minVec.x()) / step) + 1;
  unsigned int ny = static_cast<int>((maxVec.y() - minVec.y()) / step) + 1;
  unsigned int nz = static_cast<int>((maxVec.z() - minVec.z()) / step) + 1;
  unsigned int totSize = nx * ny * nz;
  Vector3D zeroVec(0.0,0.0,0.0);
  Vec<Vector3D> points(totSize, zeroVec);
  double x,y,z;
  unsigned int count = 0;
  for (unsigned int i = 0; i < nx; ++i) {
    x = minVec.x() + ( i * step );
    for (unsigned int j = 0; j < ny; ++j) {
      y = minVec.y() + ( j * step );
      for (unsigned int k = 0; k < nz; ++k) {
	z = minVec.z() + ( k * step );
	points[count++] = Vector3D(x,y,z);
      } 
    }
  }
  return points;
}

Vec<Vector3D>
generateSphereGridPoint(Vector3D& origin, double radius, double step)
{
  PRECOND(step > 0.0);
  Vector3D  maxVec = origin;
  maxVec.x(maxVec.x() + radius);
  maxVec.y(maxVec.y() + radius);
  maxVec.z(maxVec.z() + radius);
  Vector3D  minVec = origin;
  minVec.x(minVec.x() - radius);
  minVec.y(minVec.y() - radius);
  minVec.z(minVec.z() - radius);
  cout << "r, minv max: " << origin << " " << minVec << " " << maxVec << endl;
  unsigned int nx = static_cast<int>((maxVec.x() - minVec.x()) / step) + 1;
  unsigned int ny = static_cast<int>((maxVec.y() - minVec.y()) / step) + 1;
  unsigned int nz = static_cast<int>((maxVec.z() - minVec.z()) / step) + 1;
  unsigned int totSize = nx * ny * nz;
  cout << "sphere1: " << nx << " " << ny << " " << nz << " " << totSize << endl;
  Vector3D zeroVec(0.0,0.0,0.0);
  Vec<Vector3D> points(totSize, zeroVec);
  double x,y,z;
  unsigned int count = 0;
  for (unsigned int i = 0; i < nx; ++i) {
    x = minVec.x() + ( i * step );
    for (unsigned int j = 0; j < ny; ++j) {
      y = minVec.y() + ( j * step );
      for (unsigned int k = 0; k < nz; ++k) {
	z = minVec.z() + ( k * step );
        Vector3D v(x,y,z);
	if (vecDistance(v, origin) <= radius) {
	  ASSERT(count < points.size());
	  points[count++] = v;
	  // cout << "adding : " << v << endl;
	}
      } 
    }
  }
  Vec<Vector3D> smallPoints(count);
  for (unsigned int i = 0; i < smallPoints.size(); ++i) {
    smallPoints[i] = points[i];
  }
  return smallPoints;
}

/* compute bouding box of fixPoints, such that docked varPoints
   have always space for docking */
void
computeBoundingBox(const Vec<Vector3D>& varPoints,
		   const Vec<Vector3D>& fixPoints,
		   double border,
		   Vector3D& minVec,
		   Vector3D& maxVec)
{
  PRECOND(border >= 0.0);
  Vector3D varMin = findMinCoordinates(varPoints);
  Vector3D varMax = findMaxCoordinates(varPoints);
  Vector3D fixMin = findMinCoordinates(fixPoints);
  Vector3D fixMax = findMaxCoordinates(fixPoints);
  Vector3D varDelta = varMax - varMin;
  Vector3D borderVec(border, border, border);
  // write result:
  minVec = fixMin - varDelta - borderVec;  // bounding box of fix + width of var
  maxVec = fixMax + varDelta + borderVec;
}

/* compute bouding box of fixPoints, such that docked varPoints
   have always space for docking */
void
computeBoundingBox(const Vec<Vector3D>& varPoints,
		   const Vec<Vector3D>& fixPoints,
		   double border,
		   double varReduction,
		   Vector3D& minVec,
		   Vector3D& maxVec)
{
  PRECOND(border >= 0.0);
  Vector3D varMin = findMinCoordinates(varPoints);
  Vector3D varMax = findMaxCoordinates(varPoints);
  Vector3D fixMin = findMinCoordinates(fixPoints);
  Vector3D fixMax = findMaxCoordinates(fixPoints);
  Vector3D varDelta = (varMax - varMin) * varReduction;
  Vector3D borderVec(border, border, border);
  // write result:
  minVec = fixMin - varDelta - borderVec;  // bounding box of fix + width of var
  maxVec = fixMax + varDelta + borderVec;
}


/* compute bouding sphere of fixPoints, such that docked varPoints
   have always space for docking */
void
computeBoundingSphere(const Vec<Vector3D>& varPoints,
		      const Vec<Vector3D>& fixPoints,
		      double border, Vector3D& origin, double& radius )
{
  Vector3D varMin = findMinCoordinates(varPoints);
  Vector3D varMax = findMaxCoordinates(varPoints);
  Vector3D fixMin = findMinCoordinates(fixPoints);
  Vector3D fixMax = findMaxCoordinates(fixPoints);
  Vector3D varDelta = varMax - varMin;
  Vector3D fixDelta = fixMax - fixMin;
  // write result:
  origin = fixMin + ((fixMax - fixMin) * 0.5 );
  radius = 0.5 * (fixDelta.length() + varDelta.length());
}


// use only surface points, which cannot be erased using eraser with radius r
Vec<Vector3D>
erasePoints(const Vec<Vector3D>& surfPoints, const Vec<Vector3D>& solidPoints, double eraserRadius,
	    double tooCloseRadius, double regrowRadius)
{
  cout << "This version of erasePoints is not safe. Avoid using it!" << endl;
  ERROR("Deprecated version of erasePoints used!");
  double radiusSquare = eraserRadius * eraserRadius;
  double regrowRadiusSqr = regrowRadius * regrowRadius;
  double tooCloseRadiusSqr = tooCloseRadius * tooCloseRadius;
  Vec<unsigned int> deleted(surfPoints.size(), 0U);
  Vec<unsigned int> tooClose(surfPoints.size(), 0U);
  unsigned int deleteCount = 0;
  double dSqr = 0.0;
  for (unsigned int i = 0; i < surfPoints.size(); ++i) {
    // check if too close to solid point:
    bool found = false;
    for (unsigned int j = 0; j < solidPoints.size(); ++j) {
      dSqr = vecDistanceSquare(surfPoints[i], solidPoints[j]);
      if (dSqr < tooCloseRadiusSqr) {
	tooClose[i] = 1;
      }
      if (dSqr < radiusSquare) {
	found = true;
	break;
      }
    }
    // is a valid point for eraser:
    if ((!found) && (tooClose[i] == 0))  {
      for (unsigned int j = 0; j < surfPoints.size(); ++j) {
	if ((deleted[j] == 0) && (vecDistanceSquare(surfPoints[i], surfPoints[j]) < radiusSquare)) {
	  ++deleteCount;
	  deleted[j] = 1;
	}
      }
    }
  }
  // "regrow" parts of points:
  if (regrowRadiusSqr > 0.0) {
    for (unsigned int i = 0; i < surfPoints.size(); ++i) {
      if ((deleted[i] == 0) && (tooClose[i] == 0)) {
	// "undelete" appropriate points: 
	for (unsigned j = 0; j < surfPoints.size(); ++j) {
	  if ((i!=j) && (deleted[j] == 1) && (tooClose[j] == 0)
	      && (vecDistanceSquare(surfPoints[i], surfPoints[j]) < regrowRadiusSqr) ) {
	    ASSERT(deleteCount > 0);
	    deleted[j] = 0;
	    --deleteCount;
	  }
	}
      }	  
    }
  }
  ERROR_IF(deleteCount > surfPoints.size(), 
	   "Internal error in line 525.");
  if (deleteCount == surfPoints.size()) {
    Vec<Vector3D> tmpResult;
    return tmpResult;
  }
  Vec<Vector3D> result(surfPoints.size() - deleteCount);
  unsigned int pc = 0;
  for (unsigned int i = 0; i < surfPoints.size(); ++i) {
    if (deleted[i] == 0) {
      ASSERT(pc < result.size());
      result[pc++] = surfPoints[i];
    }
  }
  return result;
}


// use only surface points, which cannot be erased using eraser with radius r
Vec<Vector3D>
erasePoints(const Vec<Vector3D>& surfPoints, const Vec<Vector3D>& solidPoints, double eraserRadius,
	    double eraserStep, double tooCloseRadius, double regrowRadius)
{
  Vector3D borderVec(eraserRadius, eraserRadius, eraserRadius);
  Vector3D minVec = findMinCoordinates(surfPoints) - borderVec;
  Vector3D maxVec = findMaxCoordinates(surfPoints) + borderVec;
  Vec<Vector3D> eraserPoints = generateCubeGridPoint(minVec, maxVec, eraserStep);
  double eraserRadiusSqr = eraserRadius * eraserRadius;
  double regrowRadiusSqr = regrowRadius * regrowRadius;
  double tooCloseRadiusSqr = tooCloseRadius * tooCloseRadius;
  Vec<unsigned int> deleted(surfPoints.size(), 0U);
  Vec<unsigned int> tooClose(surfPoints.size(), 0U);
  unsigned int deleteCount = 0;
  // double dSqr = 0.0;
  for (unsigned int i = 0; i < eraserPoints.size(); ++i) {
    // check if too close to solid point:
    bool found = false;
    for (unsigned int j = 0; j < solidPoints.size(); ++j) {
      if (vecDistanceSquare(eraserPoints[i], solidPoints[j]) < eraserRadiusSqr) {
	found = true;
	break;
      }
    }
    // is a valid point for eraser:
    if (!found)  {
      for (unsigned int j = 0; j < surfPoints.size(); ++j) {
	if ((deleted[j] == 0) && (vecDistanceSquare(eraserPoints[i], surfPoints[j]) < eraserRadiusSqr)) {
	  ++deleteCount;
	  deleted[j] = 1;
	}
      }
    }
  }
  // now delete all points which are too close to solid points:
  for (unsigned int i = 0; i < surfPoints.size(); ++i) {
    // check if too close to solid point:
    for (unsigned int j = 0; j < solidPoints.size(); ++j) {
      if (vecDistanceSquare(surfPoints[i], solidPoints[j]) < tooCloseRadiusSqr) {
	if (deleted[i] == 0) {
	  ++deleteCount;
	}
	deleted[i] = 1;
	tooClose[i] = 1;
	break;
      }
    }
  }
  // "regrow" parts of points:
  if ((regrowRadiusSqr > 0.0) && (deleteCount > 0)) {
    Vec<unsigned int> undelete(surfPoints.size(), 0U);
    for (unsigned int i = 0; i < surfPoints.size(); ++i) {
      if (deleted[i] == 0) {
	// "undelete" appropriate points: 
	for (unsigned j = 0; j < surfPoints.size(); ++j) {
	  if ((i!=j) && (deleted[j] == 1) && (tooClose[j] == 0)
              && (undelete[j] == 0)
	      && (vecDistanceSquare(surfPoints[i], surfPoints[j]) < regrowRadiusSqr) ) {
	    undelete[j] = 1;
	  }
	}
      }	  
    }
    // use undelete information:
    for (unsigned int i = 0; i < deleted.size(); ++i) {
      if (undelete[i] == 1) {
        ASSERT(deleted[i] == 1);
        ASSERT(deleteCount > 0);
        deleted[i] = 0;
        --deleteCount;
        if (deleteCount == 0) {
          break;
        }
      }
    }
  }
  ERROR_IF(deleteCount > surfPoints.size(), 
	   "Internal error in line 619.");
  if (deleteCount == surfPoints.size()) {
    Vec<Vector3D> tmpResult;
    return tmpResult;
  }
  Vec<Vector3D> result(surfPoints.size() - deleteCount);
  unsigned int pc = 0;
  for (unsigned int i = 0; i < surfPoints.size(); ++i) {
    if (deleted[i] == 0) {
      ASSERT(pc < result.size());
      result[pc++] = surfPoints[i];
    }
  }
  return result;
}

Vec<Vector3D>
eraseSmallClusterPoints(const Vec<Vector3D>& v, double step, unsigned int numLigAtoms, unsigned int maxClusterNumber,
			Vec<unsigned int>& clusterNumbers, double volRedFactor, unsigned int verboseLevel)
{
  // Vec<Vec<double> > m; // distance matrix
  // pointsToField(v, m);
  // perform single link
  // ERROR_IF(m.size() != v.size(), "Internal error in line 2673!");
  double cellVol = step * step * step;
  double ligVol = 4.0 * numLigAtoms; // very rough estimation 4 angstroem**3 per atom
  Vec<Vec<unsigned int> > clusters = singleLinkage(v, 1.1 * step);
  ERROR_IF(clusters.size() == 0, "No docking point clusters found!");
  if (verboseLevel > 0) {
    cout << "Found cluster sizes: " << endl;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      cout << clusters[i].size() << " ";
    }
    cout << endl;
  }
  unsigned int numPoints = 0;
  unsigned int maxClust = maxClusterNumber;
  if (maxClust > clusters.size()) {
    maxClust = clusters.size();
  }
  Vec<unsigned int> largestClusterIds;
  for (unsigned int k = 0; k < maxClust; ++k) {
    unsigned int bestThisRoundSize = 0;
    unsigned int bestThisRoundId = 0;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      // check if this cluster is already found:
      if (findFirstIndex(largestClusterIds, i) < largestClusterIds.size()) {
	continue; // this cluster has already been found
      }
      // find best cluster which has not been found so far:
      if (clusters[i].size() > bestThisRoundSize) {
	bestThisRoundSize = clusters[i].size();
	bestThisRoundId = i;
      }
    }
    // check if cluster volume too small:
    if ((k > 0) && (bestThisRoundSize * cellVol < volRedFactor * ligVol)) {
      // stop here, clusters too small
      break;
    }
    largestClusterIds.push_back(bestThisRoundId);
    numPoints += bestThisRoundSize;
  }
  if (numPoints == 0) {
    return Vec<Vector3D>();
  }
  Vec<Vector3D> result(numPoints);
  clusterNumbers = Vec<unsigned int>(numPoints);
  unsigned int pc = 0;
  if (verboseLevel > 1) {
    cout << "The used clusters are: " << largestClusterIds << endl;
  }
  for (unsigned int i = 0; i < largestClusterIds.size(); ++i) {
    for (unsigned int j = 0; j < clusters[largestClusterIds[i]].size(); ++j) {
      clusterNumbers[pc] = i;
      result[pc++] = v[clusters[largestClusterIds[i]][j]];
    }
  }
  if (verboseLevel > 0) {
    cout << "Found docking point clusters: " << clusters.size() << " keeping only " << result.size() 
	 << " points." << endl;
  }
  return result;
}

/* different version: only allow biggest cluster and all clusters whose volume
   is at least half as large as the volume of the ligand */
Vec<Vector3D>
eraseSmallClusterPoints2(const Vec<Vector3D>& v, double step, 
			 unsigned int numLigAtoms, 
			 Vec<unsigned int>& clusterNumbers, double volRedFactor, unsigned int verboseLevel)
{
  // Vec<Vec<double> > m; // distance matrix
  // pointsToField(v, m);
  // perform single link
  // ERROR_IF(m.size() != v.size(), "Internal error in line 2673!");
  Vec<Vec<unsigned int> > clusters = singleLinkage(v, 1.1 * step);
  double cellVol = step * step * step;
  double ligVol = 4.0 * numLigAtoms; // very rough estimation 4 angstroem**3 per atom
  ERROR_IF(clusters.size() == 0, "No docking point clusters found!");
  if (verboseLevel > 0) {
    cout << "Found cluster sizes: " << endl;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      cout << clusters[i].size() << " ";
    }
    cout << endl;
  }
  unsigned int numPoints = 0;
  // unsigned int maxClust = minimum(maxClusterNumber, clusters.size());
  Vec<unsigned int> largestClusterIds;
  for (unsigned int k = 0; k < clusters.size(); ++k) {
    unsigned int bestThisRoundSize = 0;
    unsigned int bestThisRoundId = 0;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      // check if this cluster is already found:
      if (findFirstIndex(largestClusterIds, i) < largestClusterIds.size()) {
	continue; // this cluster has already been found
      }
      // find best cluster which has not been found so far:
      if (clusters[i].size() > bestThisRoundSize) {
	bestThisRoundSize = clusters[i].size();
	bestThisRoundId = i;
      }
    }
    // check if cluster volume too small:
    if ((k > 0) && (bestThisRoundSize * cellVol < volRedFactor * ligVol)) {
      // stop here, clusters too small
      break;
    }
    largestClusterIds.push_back(bestThisRoundId);
    numPoints += bestThisRoundSize;
  }
  if (numPoints == 0) {
    return Vec<Vector3D>();
  }
  Vec<Vector3D> result(numPoints);
  clusterNumbers = Vec<unsigned int>(numPoints);
  unsigned int pc = 0;
  if (verboseLevel > 1) {
    cout << "The used clusters are: " << largestClusterIds << endl;
  }
  for (unsigned int i = 0; i < largestClusterIds.size(); ++i) {
    for (unsigned int j = 0; j < clusters[largestClusterIds[i]].size(); ++j) {
      clusterNumbers[pc] = i;
      result[pc++] = v[clusters[largestClusterIds[i]][j]];       
    }
  }
  if (verboseLevel > 0) {
    cout << "Found docking point clusters: " << clusters.size() << " keeping only " << result.size() 
	 << " points." << endl;
  }
   return result;
}

// return largest minus smallest distance from center of mass
double
getClusterDepth(const Vec<Vector3D>& v, const Vector3D& center)
{
  PRECOND(v.size() > 0);
  // find smallest and largest distance:
  double smallestDist = vecDistance(v[0], center);
  double largestDist = vecDistance(v[0], center);
  double d;
  for (unsigned int i = 0; i < v.size(); ++i) {
    d = vecDistance(v[i], center);
    if (d < smallestDist) {
      smallestDist = d;
    }
    if (d > largestDist) {
      largestDist = d;
    }
  }
  return largestDist - smallestDist;
}

/** returns cluster id of point closest to search point.
    If no point if closer than maxDist, return 999
*/
unsigned int
getClusterNumber(const Vector3D& x,
		 const Vec<Vector3D>& points,
		 const Vec<unsigned int>& clusterNumbers,
		 double maxDist)
{
  PRECOND(points.size() == clusterNumbers.size());
  unsigned int id = 999;
  if (points.size() == 0) {
    return id;
  }
  double d;
  double dBest = maxDist;
  for (unsigned int i = 1; i < points.size(); ++i) {
    d = vecDistance(x, points[i]);
    if (d < dBest) {
      dBest = d;
      id = i;
    }
  }
  if (id < clusterNumbers.size()) {
    return clusterNumbers[id];
  }
  return 999; // bad id
}

Vec<Vector3D>
eraseShallowClusterPoints(const Vec<Vector3D>& v, double step, 
			  unsigned int numLigAtoms, unsigned int maxClusterNumber,
			  const Vector3D& center, Vec<unsigned int>& clusterNumbers, 
			  double volRedFactor, unsigned int verboseLevel)
{
  // Vec<Vec<double> > m; // distance matrix
  // pointsToField(v, m);
  // perform single link
  // ERROR_IF(m.size() != v.size(), "Internal error in line 2673!");
  Vec<Vec<unsigned int> > clusters = singleLinkage(v, 1.1 * step);
  double cellVol = step * step * step;
  double ligVol = 4.0 * numLigAtoms; // very rough estimation 4 angstroem**3 per atom
  ERROR_IF(clusters.size() == 0, "No docking point clusters found!");
  if (verboseLevel > 0) {
    cout << "Found cluster sizes and depths: " << endl;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      cout << clusters[i].size() << " " << getClusterDepth(getSubset(v,clusters[i]), center) << " ";
    }
    cout << endl;
  }
  unsigned int numPoints = 0;
  unsigned int maxClust = maxClusterNumber;
  if (maxClust > clusters.size()) {
    maxClust = clusters.size();
  }
  Vec<unsigned int> largestClusterIds;
  double depth;
  for (unsigned int k = 0; k < maxClust; ++k) {
    double bestThisRoundSize = 0.0;
    unsigned int bestThisRoundId = 0;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      if (findFirstIndex(largestClusterIds, i) < largestClusterIds.size()) {
	continue; // this cluster has already been found
      }
      depth = getClusterDepth(getSubset(v, clusters[i]), center);
      // check if this cluster is already found:
      // find best cluster which has not been found so far:
      if (depth > bestThisRoundSize) {
	bestThisRoundSize = depth;
	bestThisRoundId = i;
      }
    }
    largestClusterIds.push_back(bestThisRoundId);
    // check if cluster volume too small:
    if ((k <= 0) || (clusters[bestThisRoundId].size() * cellVol >= volRedFactor * ligVol)) {
      // clusters large enough
      numPoints += clusters[bestThisRoundId].size();
    }
  }
  if (numPoints == 0) {
    return Vec<Vector3D>();
  }
  Vec<Vector3D> result(numPoints);
  clusterNumbers = Vec<unsigned int>(numPoints, 0U);
  unsigned int pc = 0;
  unsigned int pcClust = 0;
  if (verboseLevel > 1) {
    cout << "The used clusters are: " << largestClusterIds << endl;
  }
  for (unsigned int i = 0; i < largestClusterIds.size(); ++i) {
    // cluster large enough
    if ((i <= 0) || (clusters[largestClusterIds[i]].size() * cellVol >= volRedFactor * ligVol)) {
      for (unsigned int j = 0; j < clusters[largestClusterIds[i]].size(); ++j) {
	clusterNumbers[pc] = pcClust;
	result[pc++] = v[clusters[largestClusterIds[i]][j]];
      }
      ++pcClust;
    }
  }
  if (verboseLevel > 0) {
    cout << "Found docking point clusters: " << clusters.size() << " keeping only " << result.size() 
	 << " points." << endl;
  }
  return result;
}


// return only points, which have at least minCount neighbours closer or equal to radius
Vec<Vector3D>
eraseSparsePoints(const Vec<Vector3D>& points, double radius, unsigned int minCount)
{
  radius *= radius;
  Vec<unsigned int> goodPointFlags(points.size(), 0);
  unsigned int goodCount = 0;
  for (unsigned int i = 0; i < points.size(); ++i) {
    unsigned int neighCount = 0;
    for (unsigned int j = 0; j < points.size(); ++j) {
      if (i == j) {
	continue;
      }
      if (vecDistanceSquare(points[i], points[j]) <= radius) {
	++neighCount;
	if (neighCount >= minCount) {
	  goodPointFlags[i] = 1;
	  ++goodCount;
	  break;
	}
      }
    }
  }
  Vec<Vector3D> result(goodCount);
  unsigned int pc = 0;
  for (unsigned int i = 0; i < goodPointFlags.size(); ++i) {
    if (goodPointFlags[i]) {
      result[pc++] = points[i];
    }
  }
  return result;
}


Vec<Vector3D>
generateCoveringSpheres(double r, const Vec<Vector3D>& points)
{
  double r2 = r * r;
  unsigned int pSize = points.size();
  Vec<Vector3D> result;
  Vec<unsigned int> covered(pSize, 0U);
  unsigned int coveredCount = 0;
  while (coveredCount < points.size()) {
    // search lowest index uncovered point:
    unsigned int ucIdx = 0;
    for (unsigned int i = 0; i < pSize; ++i) {
      if (!covered[i]) {
	ucIdx = i;
	break;
      }
    }
    if (ucIdx >= pSize) {
      break; // done!
    }
    const Vector3D& pos = points[ucIdx];
    covered[ucIdx] = 1;
    ++coveredCount;
    result.push_back(pos);
    for (unsigned int i = 0; i < pSize; ++i) {
      if (!covered[i]) {
	if (vecDistanceSquare(pos, points[i]) <= r2) {
	  covered[i] = 1;
	  ++coveredCount;
	}
      }
    }
  }
  return result;
}

/* return vector which are x, y, z axis plus and minus, with length of r around origin */
Vec<Vector3D>
getCubicVectors(double r, const Vector3D& origin)
{
  Vec<Vector3D> result(6);
  result[0] = Vector3D(1.0, 0.0, 0.0);
  result[1] = Vector3D(0.0, 1.0, 0.0);
  result[2] = Vector3D(0.0, 0.0, 1.0);
  result[3] = Vector3D(-1.0, 0.0, 0.0);
  result[4] = Vector3D(0.0, -1.0, 0.0);
  result[5] = Vector3D(0.0, 0.0, -1.0);
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = (result[i] * r) + origin;
  }
  POSTCOND(result.size() == 6);
  return result;
}

double
drms(const Vec<Vector3D>& v1, const Vec<Vector3D>& v2)
{
  PRECOND(v1.size() == v2.size());
  unsigned int vSize = v1.size();
  if (vSize < 2) {
    return 0.0;
  }
  double sum = 0.0;
  for (unsigned int i = 1; i < vSize; ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      sum += fabs(vecDistanceSquare(v1[i], v1[j])
		  - vecDistanceSquare(v2[i], v2[j]));
    }
  }
  unsigned int num = vSize * (vSize - 1);
  return sqrt(sum / num);
}

void
stretchPoint(Vector3D& pos, const Vector3D& center, 
             double stretchX, double stretchY, double stretchZ)
{
  pos.x(((pos.x() - center.x()) * stretchX) + center.x());
  pos.y(((pos.y() - center.y()) * stretchY) + center.y());
  pos.z(((pos.z() - center.z()) * stretchZ) + center.z());
}

/**
 * stretch points with center point and 3 different stretch factors
 */
void
stretchPoints(Vec<Vector3D>& points,
              const Vector3D& center,
              double stretchX,
              double stretchY,
              double stretchZ)
{
  for (unsigned int i = 0; i < points.size(); ++i) {
    stretchPoint(points[i], center, stretchX, stretchY, stretchZ);
  }
}

void
rotateAroundZ(Vector3D& pos, const Vector3D& center, double w)
{
  pos = pos - center;
  double cosw = cos(w);
  double sinw = sin(w);
  pos.x(pos.x() * cosw - pos.y() * sinw);
  pos.y(pos.x() * sinw + pos.y() * cosw);
  pos = pos + center;
}

void
rotateAroundX(Vector3D& pos, const Vector3D& center, double w)
{
  pos = pos - center;
  double cosw = cos(w);
  double sinw = sin(w);
  pos.y(pos.y() * cosw - pos.z() * sinw);
  pos.z(pos.y() * sinw + pos.z() * cosw);
  pos = pos + center;
}

void
rotateAroundY(Vector3D& pos, const Vector3D& center, double w)
{
  pos = pos - center;
  double cosw = cos(w);
  double sinw = sin(w);
  pos.z(pos.z() * cosw - pos.x() * sinw);
  pos.x(pos.z() * sinw + pos.x() * cosw);
  pos = pos + center;
}

void
bendX(Vec<Vector3D>& pos, const Vector3D& center, double anglePerDist)
{
  if (anglePerDist == 0.0) {
    return; // do nothing
  }
  for (unsigned int i = 0; i < pos.size(); ++i) {
    double ang = anglePerDist * vecDistance(pos[i], center);
    rotateAroundX(pos[i], center, ang);
  }
}

void
bendY(Vec<Vector3D>& pos, const Vector3D& center, double anglePerDist)
{
  if (anglePerDist == 0.0) {
    return; // do nothing
  }
  for (unsigned int i = 0; i < pos.size(); ++i) {
    double ang = anglePerDist * vecDistance(pos[i], center);
    rotateAroundY(pos[i], center, ang);
  }
}

void
bendZ(Vec<Vector3D>& pos, const Vector3D& center, double anglePerDist)
{
  if (anglePerDist == 0.0) {
    return; // do nothing
  }
  for (unsigned int i = 0; i < pos.size(); ++i) {
    double ang = anglePerDist * vecDistance(pos[i], center);
    rotateAroundZ(pos[i], center, ang);
  }
}


 Vec<Vector3D>
 deleteTooClosePoints(const Vec<Vector3D>& dockPoints,
		      const Vec<Vector3D>& pCoord,
		      double minProteinCloseDist)
 {
   minProteinCloseDist *= minProteinCloseDist; // use square
   Vec<unsigned int> badFlags(dockPoints.size(), 0U);
   unsigned int badCount = 0;;
   for (unsigned int i = 0; i < dockPoints.size(); ++i) {
     for (unsigned int j = 0; j < pCoord.size(); ++j) {
       if (vecDistanceSquare(dockPoints[i], pCoord[j]) < minProteinCloseDist) {
	 badFlags[i] = 1;
	 ++badCount;
	 break;
       }
     }
   }
   unsigned int goodCount = dockPoints.size() - badCount;
   if (goodCount < 1) {
     return Vec<Vector3D>();
   }
   Vec<Vector3D> result(goodCount);
   unsigned int pc = 0;
   for (unsigned int i = 0; i < dockPoints.size(); ++i) {
     if (badFlags[i] == 0) {
       ASSERT(pc < result.size());
       result[pc++] = dockPoints[i];
     }
   }
   return result;
 }

 Vec<Vector3D>
 deleteTooFarPoints(const Vec<Vector3D>& dockPoints,
                    const Vec<Vector3D>& pCoord,
                    double farDist)
 {
   farDist *= farDist; // use square
   Vec<unsigned int> badFlags(dockPoints.size(), 1U);
   unsigned int badCount = badFlags.size();
   for (unsigned int i = 0; i < dockPoints.size(); ++i) {
     for (unsigned int j = 0; j < pCoord.size(); ++j) {
       if (vecDistanceSquare(dockPoints[i], pCoord[j]) < farDist) {
	 badFlags[i] = 0;
	 --badCount;
	 break;
       }
     }
   }
   ASSERT(dockPoints.size() >= badCount);
   if (badCount >= dockPoints.size()) {
     return Vec<Vector3D>();
   }
   unsigned int goodCount = dockPoints.size() - badCount;
   Vec<Vector3D> result(goodCount);
   unsigned int pc = 0;
   for (unsigned int i = 0; i < dockPoints.size(); ++i) {
     if (badFlags[i] == 0) {
       ASSERT(pc < result.size());
       result[pc++] = dockPoints[i];
     }
   }
   return result;
 }

double
combinedDistance(const Vector3D& pos,
                 const Vec<Vector3D>& v,
                 const Vec<unsigned int>& subs)
{
  double d = 0.0;
  for (unsigned int i = 0; i < subs.size(); ++i) {
    d += vecDistance(pos, v[subs[i]]);
  }
  return d;
}

unsigned int
findMaxDistanceIndex(const Vec<Vector3D>& v,
                     const Vec<unsigned int>& subs)
{
  PRECOND(subs.size() < v.size());
  PRECOND(!containsDuplicates(subs));
  unsigned int bestId = v.size();
//    if (v.size() < 2) {
//      return bestId;
//    }
  double dBest = 0.0;
  double d = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (findFirstIndex(subs, i) < subs.size()) {
      continue; // already part of refererence subset
    }
    d = combinedDistance(v[i], v, subs);
    if (d > dBest) {
      dBest = d;
      bestId = i;
    }
  }
  return bestId;
}

Vec<unsigned int>
furthestSpanningSubset(const Vec<Vector3D>& v,
                       unsigned int numPoints)
{
  PRECOND(numPoints <= v.size());
  unsigned int nSize = numPoints;
  if (nSize > v.size()) {
    nSize = v.size();
  }
  Vector3D center = computeCenterOfMass(v);
  unsigned int furthestId = findMaxDistanceIndex(center, v);
  Vec<unsigned int> sofarId(1, furthestId);
  if (numPoints < 2) {
    return sofarId;
  }
  unsigned int newId = 0;
  for (unsigned int i = 1; i < nSize; ++i) {
    newId = findMaxDistanceIndex(v, sofarId);
    if (newId < v.size()) {
      sofarId.push_back(newId);
    }
    else {
      break;
    }
  }
  // POSTCOND(sofarId.size() <= minimum(v.size(),numPoints));
  POSTCOND(!containsDuplicates(sofarId));
  return sofarId;
}




