// --*- C++ -*------x---------------------------------------------------------
#ifndef __MULTI_VAL_ARRAY__
#define __MULTI_VAL_ARRAY__


#include <Vec.h>

class MultiValArray {
  
 public:
  
  typedef double double_t;
  
  typedef Vec<double_t>::size_type size_type;
  
  typedef Vec<size_type> index_array;
  
 private:
  
  double_t * data;

  index_array dimensions;
  
  index_array multipliers;
  
  size_type sz;
  
 public:

  /** Initializes a "quader" with the side length given in _dimensions */
  MultiValArray(const index_array& _dimensions) : data(0) { init(_dimensions);  }

  /** Initializes a hyper-cube with nDim dimensions and side length sideLen */    
  MultiValArray(size_type nDim, size_type sideLen) : data(0) {
    PRECOND(nDim > 0);
    PRECOND(sideLen > 0);
    index_array _dimensions(nDim, sideLen);
    init(_dimensions);
    POSTCOND(getDimensionCount() == _dimensions.size());
  }

  virtual ~MultiValArray() { delete data; }

  /** Sets all array elements to zero. */  
  virtual void clear() { 
    for (size_type i = 0; i < sz; ++i) {
      data[i] = 0.0;
    }
  }

  /** Fills all array elements with particular value. */
  virtual void fill(double_t val) { 
    for (size_type i = 0; i < sz; ++i) {
      data[i] = val;
    }
  }

  /** Returns all side lengths */
  const index_array& getDimensions() const { return dimensions; }

  /** Returns size of a particular dimension */
  size_type getDimension(size_type dimId) { 
    ASSERT(dimId < getDimensionCount());
    return dimensions[dimId];
  }

  size_type getDimensionCount() const { return dimensions.size(); }

  double_t getRaw(size_type id) const {
    ASSERT(id < sz);
    return data[id];
  }
  
  size_type computeIndex(const index_array& indices) const {
    PRECOND(indices.size() == getDimensionCount());
    size_type result = 0;
    for (size_type i = 0; i < indices.size(); ++i) {
      ASSERT(indices[i] < dimensions[i]);
      result += multipliers[i] * indices[i];
    }
    POSTCOND(result < size());
    return result;
  }

  double_t get(const index_array& indices) const {
    PRECOND(indices.size() == getDimensionCount());
    size_type id = computeIndex(indices);
    ASSERT(id < size());
    return data[id];
  }

  const index_array& getMultipliers() const { return multipliers; }

  void set(const index_array& indices, double_t value) {
    PRECOND(indices.size() == getDimensionCount());
    size_type id = computeIndex(indices);
    ASSERT(id < size());
    data[id] = value;
  }

  size_type size() const { return sz; }

  size_type computeElementCount(const index_array& _dimensions) const {
    size_type result = 1;
    for (size_type i = 0; i < _dimensions.size(); ++i) {
      ASSERT(_dimensions[i] > 0);
      result *= _dimensions[i];
    }
    POSTCOND(result > 0);
    return result;
  }

 private:

  /** Copy constructor not implemented */
  MultiValArray(const MultiValArray& other) { 
    ASSERT(false); // not implemented
  }

  MultiValArray& operator = (const MultiValArray& other) {
    if (this != &other) {
      copy(other);
    }
    return *this;
  }

  void copy(const MultiValArray& other) {
    ERROR("Copying of MultiValArray not yet implemented!");
  }

  void init(const index_array& _dimensions) {
    PRECOND(_dimensions.size() > 0);
    PRECOND(data == 0);
    dimensions = _dimensions;
    sz = computeElementCount(dimensions);
    data = new double_t[sz]; // sorry, no delete here; init should only be called once
    clear();
    multipliers = dimensions;
    multipliers[multipliers.size()-1] = 1;
    for (size_type i = 1; i < multipliers.size(); ++i) {
      multipliers[multipliers.size() - i - 1] = multipliers[multipliers.size() - i] * dimensions[multipliers.size() - i];
      ASSERT(multipliers[multipliers.size() - i -1 ] > 0);
    }
    //     cout << "Finished init!" << endl;
    //     cout << "Dimensions: " << dimensions << endl;
    //     cout << "Multipliers: " << multipliers << endl;
  }

};

#endif
