library(bio3d)

# R script computes theoretical hydrodynamic radius for a PDB file of an RNA structure.
# see: http://en.wikipedia.org/wiki/Hydrodynamic_radius
# Usage: Rscript trhydro.R PDBFILE
# or R --vanilla --args PDBFILE < trhydro.R
# with PDBFILE being the filename of the PDB structure. The script has in this example to be in the 
# same directory as the PDB file.
# Eckart Bindewald, 2010

helpOutput <- function() {
  cat("Program computes the maximum distance between any two residues of an RNA structure. Each residue position is represented through its phosphate atom position. Input is a PDB file. This corresponds to the minimum radius of a sphere containing the structure.\n")
}

dist.euclid <- function(x1,y1,z1,x2,y2,z2) {
  sqrt((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2) )
}

dist.euclidSq <- function(x1,y1,z1,x2,y2,z2) {
  (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2)
}

dist.euclidSqD <- function(x1,x2,x3) {
 x1*x1 + x2*x2 + x3*x3
}

test.dist.euclid <- function() {
   print(dist.euclid(3,0,0,0,4,0))
}

radius.max <- function(pdb) {
  p.inds <- atom.select(pdb, "///////")
#  n <- length(p.inds)
  xyz <- pdb$xyz[ p.inds$xyz ]
  n3 <- length(xyz)
  xv <- xyz[seq(1,n3,3)]
  yv <- xyz[seq(2,n3,3)]
  zv <- xyz[seq(3,n3,3)]
  n <- length(xv)
  stopifnot((n * 3) == n3)
  rMax <- 0
  for (i in 1:n) {
    x <- xv[i]
    y <- yv[i]
    z <- zv[i]
    for (j in 1:n) {
        dx <- x - xv[j]
        dy <- y-yv[j]
        dz <- z-zv[j]
        r <- dx*dx + dy*dy + dz*dz
#        r <- dist.euclidSqD(x-xv[j],y-yv[j],z-zv[j])
        if (r > rMax) {
            rMax <- r
        } else if (r < (rMax/2)) {
          j <- j + 10
        }
    }
  }
  result <- sqrt(rMax) 
  result
}

## main program

args <- commandArgs(trailingOnly=TRUE)
if (length(args) < 1) {
	helpOutput()
} else {
 filename <- args[1]
 pdb <- read.pdb(filename)
 rMax <- radius.max(pdb)
 cat("Maximum residue distance:", rMax, "\n")
 cat("Maximum residue distance:", rMax/2, "\n")
}
