library(bio3d)

# R script computes theoretical hydrodynamic radius for a PDB file of an RNA structure.
# see: http://en.wikipedia.org/wiki/Hydrodynamic_radius
# Usage: Rscript trhydro.R PDBFILE
# or R --vanilla --args PDBFILE < trhydro.R
# with PDBFILE being the filename of the PDB structure. The script has in this example to be in the 
# same directory as the PDB file.
# Eckart Bindewald, 2010

helpOutput <- function() {
  cat("Program computes theoretical hydrodynamic radius (see Wikipedia definition) of an RNA structure. Each residue position is represented through its phosphate atom position. Input is a PDB file.\n")
}

dist.euclid <- function(x1,y1,z1,x2,y2,z2) {

  sqrt((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2) )

}

test.dist.euclid <- function() {
   print(dist.euclid(3,0,0,0,4,0))
}

radius.hydro <- function(pdb) {
  p.inds <- atom.select(pdb, "//////P/")
#  n <- length(p.inds)
  xyz <- pdb$xyz[ p.inds$xyz ]
  n3 <- length(xyz)
  xv <- xyz[seq(1,n3,3)]
  yv <- xyz[seq(2,n3,3)]
  zv <- xyz[seq(3,n3,3)]
  n <- length(xv)
  stopifnot((n * 3) == n3)
  sum <- 0
  for (i in 1:n) {
    for (j in 1:n) {
        if (i != j) {
           sum <- sum + (1/dist.euclid(xv[i],yv[i],zv[i], xv[j],yv[j],zv[j]))
        }
    }
  }
  sum <- sum * (1/(n*n))
  result <- 1/sum
  result
}

## main program

args <- commandArgs(trailingOnly=TRUE)
if (length(args) < 1) {
	helpOutput()
} else {
 filename <- args[1]
 pdb <- read.pdb(filename)
 cat("Theoretical hydrodynamic radius:", radius.hydro(pdb), "\n")
}
