/* mk_ref_nucs.c -
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "../src/rna.h"

BASEPAIR_TEMPLATE bp_template = { /* the AU,UA,CG and GC basepairs */
	{{6.837,  -5.134,  1.770},	/* C1' of 5' nuc */ /*bp_triad*/
	 {6.837,   5.134,  -1.770},	/* C1' of 3' nuc */
	 {5.837,0.0,0.0} /* midpoint of C1'-C1' line translate one unit
			    towards Z-axis (the helix axis) */
	},
	{
	{{6.837,  -5.134,   1.770},	/* C1' of 5' nuc */ /*nuc5_triad*/
	 {6.614,  -7.068,  3.100},	/* C4' of 5' nuc */
	 {5.671,  -4.305,   1.390}	/* N9(N1) of 5' nuc */
	},
	{{6.837,   5.134,  -1.770},	/* C1' of 3' nuc */ /*nuc3_triad*/
	 {6.614,   7.068,  -3.100},	/* C4' of 3' nuc */
	 {5.671,   4.305,  -1.390}	/* N1(N9) of 3' nuc */
	}
	}
	/* note that nuc_v[][][] are not initialized here */
};

BASE_TEMPLATE b_template = { 
	{{6.837,  -5.134,  1.770},	/* the b_triad[][] */
	 {6.837,   0.0,  0.0},
	 {5.837,0.0,0.0} 
	},
	{{6.837,  -5.134,   1.770},	/* the reference nuc triad */
	 {6.614,  -7.068,  3.100},
	 {5.671,  -4.305,   1.390}
	}

	/* note that nuc_v[][][] are not initialized here */
};

char *resgrp_name[] = { /* the residue groups */
	"A_nuc.pdb",
	"C_nuc.pdb",
	"G_nuc.pdb",
	"U_nuc.pdb"
};

char *duplex_name[] = { /* the duplexes */
	"AU_duplex.pdb",
	"CG_duplex.pdb",
	"GC_duplex.pdb",
	"UA_duplex.pdb"
};

NUCLEOTIDE nuc[12];

main()
{
	int i,j,k;
	FILE *fp, *fopen();
	char line[100],hdr[10],atom_name[10],base[10];
	int res_no,atom_no;
	float c[3];
	int nuc_cnt = 0, atom_cnt = 0;
	NUCLEOTIDE *pn;

    /* read in the pdb info of the four residue groups */

	for (k = 0; k < 4; k++) {	
	   fp = fopen(resgrp_name[k],"r");
    	   while (1) {
		fgets(line,100,fp);
		sscanf(line,"%s%d%s%s%d%f%f%f",
				hdr,&atom_no,atom_name,base,&res_no,&c[0],&c[1],&c[2]);
		if (!strcmp(hdr,"TER")) {
			nuc[nuc_cnt++].atom_count = atom_cnt;
			atom_cnt = 0;
			break;
		}

		else if (!strcmp(atom_name,"P") || atom_no == 1) { /* if start of next nuc */
		    	if (atom_no != 1) { /* if not first nuc in kth group */
				nuc[nuc_cnt++].atom_count = atom_cnt;
				atom_cnt = 0;
			}
			nuc[nuc_cnt].base = base[strlen(base)-1];
			switch(nuc[nuc_cnt].base) {
				case 'A':  strcpy(nuc[nuc_cnt].name,"adenine");break;
				case 'G':  strcpy(nuc[nuc_cnt].name,"guanine");break;
				case 'U':  strcpy(nuc[nuc_cnt].name,"uridine");break;
				case 'C':  strcpy(nuc[nuc_cnt].name,"cytosine");break;
			}
		   	if (res_no == 1)
				strcat(nuc[nuc_cnt].name,"5");
			else if  (res_no == 3)
				strcat(nuc[nuc_cnt].name,"3");
		}

		strcpy(nuc[nuc_cnt].atom[atom_cnt].name,atom_name);
		for (j = 0; j < 3; j++)
			nuc[nuc_cnt].atom[atom_cnt].pdb[j] = c[j];
		atom_cnt++;
	    }
	    fclose(fp);
    	}


    /* assign the atom_index[] array for the triangle vertices */

	for (k = 0; k < nuc_cnt; k++) {
	    pn = &nuc[k];
	    for (j = 0; j < pn->atom_count; j++) {
		if (!strcmp(pn->atom[j].name,"C1'"))
			pn->atom_index[0] = j;
		else if (!strcmp(pn->atom[j].name,"C4'"))
			pn->atom_index[1] = j; /* C4' */
		else if ((pn->base == 'C' || pn->base == 'U') && 
					!strcmp(pn->atom[j].name,"N1"))
			pn->atom_index[2] = j;
		else if ((pn->base == 'A' || pn->base == 'G') && 
					!strcmp(pn->atom[j].name,"N9"))
			pn->atom_index[2] = j;
	    }
	    /*
	    fprintf(stderr,"%s (%d): %d %d %d\n",pn->name,pn->atom_count,
		pn->atom_index[0],pn->atom_index[1],pn->atom_index[2]);
	    */
	}


    /* make the local (V-frame) coordinates */

	for (k = 0; k < nuc_cnt; k++)
		mk_local_coords(&nuc[k]);

    /* make the sugar index arrays (for display purposes) */

	for (k = 0; k < nuc_cnt; k++) {
		pn = &nuc[k];
		for (j = 0; j < pn->atom_count; j++) {
			if (!strcmp(pn->atom[j].name,"C1'"))
				pn->sugar[0] = j;
			else if (!strcmp(pn->atom[j].name,"C2'"))
				pn->sugar[1] = j;
			else if (!strcmp(pn->atom[j].name,"C3'"))
				pn->sugar[2] = j;
			else if (!strcmp(pn->atom[j].name,"C4'"))
				pn->sugar[3] = j;
			else if (!strcmp(pn->atom[j].name,"O4'"))
				pn->sugar[4] = j;
		}
	}

    /* make the base_ring index arrays (for display purposes) */

	for (k = 0; k < nuc_cnt; k++) { 
	    pn = &nuc[k];
	    for (j = 0; j < pn->atom_count; j++) {
		if (pn->base == 'U' || pn->base == 'C') {
			if (!strcmp(pn->atom[j].name,"C4"))
				pn->base_ring[0][0] = j;
			else if (!strcmp(pn->atom[j].name,"N3"))
				pn->base_ring[0][1] = j;
			else if (!strcmp(pn->atom[j].name,"C2"))
				pn->base_ring[0][2] = j;
			else if (!strcmp(pn->atom[j].name,"N1"))
				pn->base_ring[0][3] = j;
			else if (!strcmp(pn->atom[j].name,"C6"))
				pn->base_ring[0][4] = j;
			else if (!strcmp(pn->atom[j].name,"C5"))
				pn->base_ring[0][5] = j;
		}
		else if (pn->base == 'A' || pn->base == 'G') {
			if (!strcmp(pn->atom[j].name,"C6"))
				pn->base_ring[0][0] = j;
			else if (!strcmp(pn->atom[j].name,"N1"))
				pn->base_ring[0][1] = j;
			else if (!strcmp(pn->atom[j].name,"C2"))
				pn->base_ring[0][2] = j;
			else if (!strcmp(pn->atom[j].name,"N3"))
				pn->base_ring[0][3] = j;
			else if (!strcmp(pn->atom[j].name,"C4")) {
				pn->base_ring[0][4] = j;
				pn->base_ring[1][0] = j;
			}
			else if (!strcmp(pn->atom[j].name,"C5")) {
				pn->base_ring[0][5] = j;
				pn->base_ring[1][1] = j;
			}
			else if (!strcmp(pn->atom[j].name,"N7"))
				pn->base_ring[1][2] = j;
			else if (!strcmp(pn->atom[j].name,"C8"))
				pn->base_ring[1][3] = j;
			else if (!strcmp(pn->atom[j].name,"N9"))
				pn->base_ring[1][4] = j;

		}
	    }
	}

    /* make other atom indices */

	for (k = 0; k < nuc_cnt; k++) {
	    pn = &nuc[k];
	    for (j = 0; j < pn->atom_count; j++) {
		if (!strcmp(pn->atom[j].name,"P"))
			pn->P_index = j;
		else if (!strcmp(pn->atom[j].name,"O1P"))
			pn->O1P_index = j;
		else if (!strcmp(pn->atom[j].name,"O2P"))
			pn->O2P_index = j;
		else if (!strcmp(pn->atom[j].name,"O5'"))
			pn->O5p_index = j;
		else if (!strcmp(pn->atom[j].name,"O4'"))
			pn->O4p_index = j;
		else if (!strcmp(pn->atom[j].name,"O3'"))
			pn->O3p_index = j;
		else if (!strcmp(pn->atom[j].name,"O2'"))
			pn->O2p_index = j;
		else if (!strcmp(pn->atom[j].name,"C1'"))
			pn->C1p_index = j;
		else if (!strcmp(pn->atom[j].name,"C2'"))
			pn->C2p_index = j;
		else if (!strcmp(pn->atom[j].name,"C3'"))
			pn->C3p_index = j;
		else if (!strcmp(pn->atom[j].name,"C4'"))
			pn->C4p_index = j;
		else if (!strcmp(pn->atom[j].name,"C5'"))
			pn->C5p_index = j;
		else if (!strcmp(pn->atom[j].name,"N1"))
			pn->N1_index = j;
		else if (!strcmp(pn->atom[j].name,"N9"))
			pn->N9_index = j;
		else if (!strcmp(pn->atom[j].name,"N3"))
			pn->N3_index = j;

	    }

	    for (j = 0; j < pn->atom_count; j++) {

		if (!strcmp(pn->atom[j].name,"N2"))
			pn->N2_index = j;
		else if (!strcmp(pn->atom[j].name,"O2"))
			pn->O2_index = j;
		else if (!strcmp(pn->atom[j].name,"N4"))
			pn->N4_index = j;
		else if (!strcmp(pn->atom[j].name,"O4"))
			pn->O4_index = j;
		else if (!strcmp(pn->atom[j].name,"N6"))
			pn->N6_index = j;
		else if (!strcmp(pn->atom[j].name,"N7"))
			pn->N7_index = j;
		else if (!strcmp(pn->atom[j].name,"O6"))
			pn->O6_index = j;
		else if (!strcmp(pn->atom[j].name,"C2"))
			pn->C2_index = j;
		else if (!strcmp(pn->atom[j].name,"C4"))
			pn->C4_index = j;
		else if (!strcmp(pn->atom[j].name,"C5"))
			pn->C5_index = j;
		else if (!strcmp(pn->atom[j].name,"C6"))
			pn->C6_index = j;
		else if (!strcmp(pn->atom[j].name,"C8"))
			pn->C8_index = j;

		else if (!strcmp(pn->atom[j].name,"H5'")) /*adenine*/
			pn->H5p_index = j;
		else if (!strcmp(pn->atom[j].name,"HO3'"))
			pn->HO3p_index = j;
		else if (!strcmp(pn->atom[j].name,"HO5'"))
			pn->HO5p_index = j;
		else if (!strcmp(pn->atom[j].name,"H1'"))
			pn->H1p_index = j;
		else if (!strcmp(pn->atom[j].name,"H2'"))
			pn->H2p_index = j;
		else if (!strcmp(pn->atom[j].name,"H3'"))
			pn->H3p_index = j;
		else if (!strcmp(pn->atom[j].name,"H4'"))
			pn->H4p_index = j;
		else if (!strcmp(pn->atom[j].name,"1H5'"))
			pn->h1H5p_index = j;
		else if (!strcmp(pn->atom[j].name,"2H5'"))
			pn->h2H5p_index = j;
		else if (!strcmp(pn->atom[j].name,"HO2'"))
			pn->HO2p_index = j;
		else if (!strcmp(pn->atom[j].name,"1H6"))
			pn->h1H6_index = j;
		else if (!strcmp(pn->atom[j].name,"2H6"))
			pn->h2H6_index = j;
		else if (!strcmp(pn->atom[j].name,"H8"))
			pn->H8_index = j;
		else if (!strcmp(pn->atom[j].name,"H2"))
			pn->H2_index = j;

		else if (!strcmp(pn->atom[j].name,"1H4")) /*cytosine*/
			pn->h1H4_index = j;
		else if (!strcmp(pn->atom[j].name,"2H4"))
			pn->h2H4_index = j;
		else if (!strcmp(pn->atom[j].name,"H6"))
			pn->H6_index = j;
		else if (!strcmp(pn->atom[j].name,"H5"))
			pn->H5_index = j;

		else if (!strcmp(pn->atom[j].name,"1H2")) /*guanine*/
			pn->h1H2_index = j;
		else if (!strcmp(pn->atom[j].name,"2H2"))
			pn->h2H2_index = j;
		else if (!strcmp(pn->atom[j].name,"H1"))
			pn->H1_index = j;

		else if (!strcmp(pn->atom[j].name,"H3")) /*uridine*/
			pn->H3_index = j;
	    }

	}

    /* define the app_index[] (to be used in preRefine.c) */

	for (k = 0; k < nuc_cnt; k++) {
	    	pn = &nuc[k];
		pn->app_index[0] = pn->P_index;
		pn->app_index[1] = pn->O1P_index;
		pn->app_index[2] = pn->O2P_index;
		pn->app_index[3] = pn->O5p_index;
		pn->app_index[4] = pn->O3p_index;
		pn->app_index[5] = pn->C3p_index;
	}
							

    /* store the reference nucs */

	for (k = 0; k < nuc_cnt; k++)
		store_ref_nuc(&nuc[k]);


    /* check */
	
    fp = fopen("ref_nucs.ascii","w");
    for (k = 0; k < nuc_cnt; k++) {
	fprintf(fp,"\n%d: %s %d\n ",k,nuc[k].name,nuc[k].atom_count);
	for (j = 0; j < nuc[k].atom_count; j++)
		fprintf(fp,"\t%d: %s %8.3f %8.3f %8.3f\n",j,
			nuc[k].atom[j].name,nuc[k].atom[j].a[0],
			nuc[k].atom[j].a[1],nuc[k].atom[j].a[2]);
    }
    fclose(fp);

    mk_bp_template_coords();
    mk_b_template_coords();
}

mk_bp_template_coords() /* define the bp_template.nuc_v[][][] arrays */
{
	int i,j,k,n;
	float V[3][3],norm[3],a[3];
	float scalar_prod(),vecj;
	BASEPAIR_TEMPLATE *pbpt = &bp_template;
    

    /* determine the orthonormal basis vectors V[0],V[1] and V[2] */
	
	norm[0] = norm[1] = 0;
	for (j = 0; j < 3; j++) { /* note: atom_index[0] is the C3' atom */
	    V[0][j] = pbpt->bp_triad[1][j] - pbpt->bp_triad[0][j];
	    norm[0] += V[0][j]*V[0][j];
	    V[1][j] = pbpt->bp_triad[2][j] - pbpt->bp_triad[0][j];
	    norm[1] += V[1][j]*V[1][j];
	}
	norm[0] = fsqrt(norm[0]);
	norm[1] = fsqrt(norm[1]);
	for (k = 0; k < 2; k++)
		for (j = 0; j < 3; j++)
			V[k][j] /= norm[k];

	vector_prod(V[0],V[1],V[2]);
	
	norm[2] = 0;
	for (j = 0; j < 3; j++)
		norm[2] += V[2][j]*V[2][j];
	norm[2] = fsqrt(norm[2]);
	for (j = 0; j < 3; j++)
		V[2][j] /= norm[2];

	vector_prod(V[1],V[2],V[0]);

	norm[0] = 0;
	for (j = 0; j < 3; j++)
		norm[0] += V[0][j]*V[0][j];
	norm[0] = fsqrt(norm[0]);
	for (j = 0; j < 3; j++)
		V[0][j] /= norm[0];

    /* determine the nuc_v[][][] arrays */

	for (i = 0; i < 2; i++) {
       	    for (k = 0; k < 3; k++) {
	    	for (j = 0; j < 3; j++)
		    a[j] = pbpt->nuc_triad[i][k][j]-pbpt->bp_triad[0][j];
	    	for (j = 0; j < 3; j++) {
		    pbpt->nuc_v[i][k][j] = scalar_prod(a,V[j]);
	    	}
	    }
	}

	fprintf(stderr,"\nthe nuc_v[i][k][j] values\n");
	for (i = 0; i < 2; i++) {
		fprintf(stderr,"%d:\n",i);
		for (k = 0; k < 3; k++) {
			fprintf(stderr,"\t%d: ",k);
			for (j = 0; j < 3; j++)
				fprintf(stderr,"%8.3f",pbpt->nuc_v[i][k][j]);
			fprintf(stderr,"\n");
		}
		fprintf(stderr,"\n");
	}

	fprintf(stderr,"\nthe ref nuc_triad coordinates (recovered as a check)\n");
	for (i = 0; i < 2; i++) {
		fprintf(stderr,"%d:\n",i);
		for (k = 0; k < 3; k++) {
			fprintf(stderr,"\t%d: ",k);
			for (j = 0; j < 3; j++) {
				vecj = 0;
				for (n = 0; n < 3; n++)
					vecj += pbpt->nuc_v[i][k][n]*V[n][j];
				fprintf(stderr,"%8.3f",pbpt->bp_triad[0][j]+
							vecj);
			}
			fprintf(stderr,"\n");
		}
		fprintf(stderr,"\n");
	}
	


    /* now store the bp_template as "ref_bp_template" */

	store_bp_template();
}

mk_b_template_coords() /* define the b_template.nuc_v[][] array */
{
	int i,j,k,n;
	float V[3][3],norm[3],a[3];
	float scalar_prod(),vecj;
	BASE_TEMPLATE *pbt = &b_template;
    

    /* determine the orthonormal basis vectors V[0],V[1] and V[2] */
	
	norm[0] = norm[1] = 0;
	for (j = 0; j < 3; j++) { /* note: atom_index[0] is the C3' atom */
	    V[0][j] = pbt->b_triad[1][j] - pbt->b_triad[0][j];
	    norm[0] += V[0][j]*V[0][j];
	    V[1][j] = pbt->b_triad[2][j] - pbt->b_triad[0][j];
	    norm[1] += V[1][j]*V[1][j];
	}
	norm[0] = fsqrt(norm[0]);
	norm[1] = fsqrt(norm[1]);
	for (k = 0; k < 2; k++)
		for (j = 0; j < 3; j++)
			V[k][j] /= norm[k];

	vector_prod(V[0],V[1],V[2]);
	
	norm[2] = 0;
	for (j = 0; j < 3; j++)
		norm[2] += V[2][j]*V[2][j];
	norm[2] = fsqrt(norm[2]);
	for (j = 0; j < 3; j++)
		V[2][j] /= norm[2];

	vector_prod(V[1],V[2],V[0]);

	norm[0] = 0;
	for (j = 0; j < 3; j++)
		norm[0] += V[0][j]*V[0][j];
	norm[0] = fsqrt(norm[0]);
	for (j = 0; j < 3; j++)
		V[0][j] /= norm[0];

    /* determine the nuc_v[][] array */


       	    for (k = 0; k < 3; k++) {
	    	for (j = 0; j < 3; j++)
		    a[j] = pbt->nuc_triad[k][j]-pbt->b_triad[0][j];
	    	for (j = 0; j < 3; j++) {
		    pbt->nuc_v[k][j] = scalar_prod(a,V[j]);
	    	}
	    }

	fprintf(stderr,"\nthe nuc_v[k][j] values\n");

		for (k = 0; k < 3; k++) {
			fprintf(stderr,"\t%d: ",k);
			for (j = 0; j < 3; j++)
				fprintf(stderr,"%8.3f",pbt->nuc_v[k][j]);
			fprintf(stderr,"\n");
		}
		fprintf(stderr,"\n");

	fprintf(stderr,"\nthe ref nuc_triad coordinates (recovered as a check)\n");

		for (k = 0; k < 3; k++) {
			fprintf(stderr,"\t%d: ",k);
			for (j = 0; j < 3; j++) {
				vecj = 0;
				for (n = 0; n < 3; n++)
					vecj += pbt->nuc_v[k][n]*V[n][j];
				fprintf(stderr,"%8.3f",pbt->b_triad[0][j]+
							vecj);
			}
			fprintf(stderr,"\n");
		}
		fprintf(stderr,"\n");

	


    /* now store the b_template as "ref_b_template" */

	store_b_template();
}
mk_local_coords(pnuc)
	NUCLEOTIDE *pnuc;
{
	int i,j,k;
	float V[3][3],norm[3];
	float scalar_prod();
	float sum, diff;
    

    /* Define the triangle sides (ab_eqlen[]). */

	for (i = 0; i < 3; i++) {
		sum = 0;
		for (j = 0; j < 3; j++) {
			diff = pnuc->atom[pnuc->atom_index[(i+1)%3]].pdb[j]-
			       pnuc->atom[pnuc->atom_index[i]].pdb[j];
			sum += diff*diff;
		}
		pnuc->ab_eqlen[i] = fsqrt(sum);
	}


    /* determine the orthonormal basis vectors V[0],V[1] and V[2] */
	
	norm[0] = norm[1] = 0;
	for (j = 0; j < 3; j++) { /* note: atom_index[0] is the C3' atom */
	    V[0][j] = pnuc->atom[pnuc->atom_index[1]].pdb[j]-
			pnuc->atom[pnuc->atom_index[0]].pdb[j];
	    norm[0] += V[0][j]*V[0][j];
	    V[1][j] = pnuc->atom[pnuc->atom_index[2]].pdb[j]-
			pnuc->atom[pnuc->atom_index[0]].pdb[j];
	    norm[1] += V[1][j]*V[1][j];
	}
	norm[0] = fsqrt(norm[0]);
	norm[1] = fsqrt(norm[1]);
	for (k = 0; k < 2; k++)
		for (j = 0; j < 3; j++)
			V[k][j] /= norm[k];

	vector_prod(V[0],V[1],V[2]);
	
	norm[2] = 0;
	for (j = 0; j < 3; j++)
		norm[2] += V[2][j]*V[2][j];
	norm[2] = fsqrt(norm[2]);
	for (j = 0; j < 3; j++)
		V[2][j] /= norm[2];

	vector_prod(V[1],V[2],V[0]);

	norm[0] = 0;
	for (j = 0; j < 3; j++)
		norm[0] += V[0][j]*V[0][j];
	norm[0] = fsqrt(norm[0]);
	for (j = 0; j < 3; j++)
		V[0][j] /= norm[0];

    /* Determine the C1'-relative coordinates a[] for the atoms and the coordinates
	v[] relative to the fram V[] */ 

       for (k = 0; k < pnuc->atom_count; k++) {
	    for (j = 0; j < 3; j++)
		 pnuc->atom[k].a[j] = pnuc->atom[k].pdb[j]-
					pnuc->atom[pnuc->atom_index[0]].pdb[j];
	    for (j = 0; j < 3; j++) {
		 pnuc->atom[k].v[j] = scalar_prod(pnuc->atom[k].a,V[j]);
	    }
	}

}
store_ref_nuc(pnuc) /* Store the reference nucleotide *pnuc */
	NUCLEOTIDE *pnuc;
{
	char fname[40];
	FILE *fp;


	strcpy(fname,pnuc->name);
	strcat(fname,".refdata");
	fp = fopen(fname,"w");
	fwrite(pnuc,sizeof(NUCLEOTIDE),1,fp);
	fclose(fp);
}
store_bp_template() /* Store the reference bp_template */
{
	FILE *fp;

	fp = fopen("ref_bp_template","w");
	fwrite(&bp_template,sizeof(BASEPAIR_TEMPLATE),1,fp);
	fclose(fp);
}
store_b_template() /* Store the reference b_template */
{
	FILE *fp;

	fp = fopen("ref_b_template","w");
	fwrite(&b_template,sizeof(BASE_TEMPLATE),1,fp);
	fclose(fp);
}
vector_prod(u,v,n) /* n = u X v */
	float *u,*v,*n;
{
	n[0] = u[1]*v[2] - v[1]*u[2];
	n[1] = v[0]*u[2] - u[0]*v[2];
	n[2] = u[0]*v[1] - v[0]*u[1];
}
float scalar_prod(u,v)
	float *u,*v;
{
	int k;
	float sum = 0;

	for (k = 0; k < 3; k++)
		sum += u[k]*v[k];

	return(sum);
}
